/* --------------------------------------------------------------- 

trace -- returns the i-th trace of a integer polynomial

Call:    polylib::trace(p,i);

Parameter:  p -- a monic univariate polynomial (DOM_POLY),
                 if p is not monic, it will be divided by its
                 lcoeff, if that is not possible, an error is
                 returned.
                 
            i -- a natural number >= 1

Return value: an integer representing the i-th trace mods m.

Synopsis:     The i-th trace of a polynomial is the sum of
              the i-th powers of all its roots. 
                                          
Remark:       The trace is determined without calculating
              the roots. Instead, the Newton-Identities are
              used to calculate the trace efficiently.
              
              This function is mainly used in faclib::knapsack.
--------------------------------------------------------------- */

polylib::trace:=
proc(f : DOM_POLY,
     i : Type::PosInt)
local d, k, x, tmp, e, R, l, m, mmod;
option remember;
begin
  userinfo(1, i."th trace of",f,"requested.");

  // polynomial has to be univariate
  assert(nops(op(f, 2)) = 1);

  // Function e needed 'cause coeff will return error
  // if number is < 0.
  e := n -> _if((n < 0), 0, coeff(f,n));
  mmod := (a,b) -> _if((b=0), a, a mod b); 

  x := op(op(f,2),1);
  R := op(f,3);
	
  if (op(R,0) = IntMod) then
    // We got a polynomial in Z/mZ so we can calculate
    // efficiently, reducing our intermediate results
    // modulo m.
    m := op(R,1);
  else
    m := 0;
  end_if; // poly over IntMod(m)
	
  d:=degree(f,x);
  if (l := lcoeff(f)) <> 1 then
    if 0 <> traperror((f := mapcoeffs(f,_divide,l)))
        or (f = FAIL) then
      error ("Polynomial should be monic or divisible ".
	     "by lcoeff.");
    end_if;
  elif i=0 then
    return(d);
  end_if;
  // Newton identities
  tmp := - _plus(mmod(polylib::trace(f,k,m)* e(d-i+k), m)
		 $ k=1..i-1)-i*e(d-i);
  return (mmod(tmp,m));
end_proc: // polylib::trace

