// 

/*--
prog::genidentgenerator(prefix, postfix, startAt, nLength, exclude) 
returns a procedure that generates identifiers of the form <prefix><counter><postfix>, 
where <counter> starts at 'startAt'. If the number of digits of <counter> is less then
'nDigits' then it is filled up with leading "0"s up to 'nDigits' digits. If <exclude>
holds an expression then the generator function will not generate identifiers which
occur in this expression.

Let 'genVar' be the name of the returned procedure: 

genVar() generates a new identifier by incrementing <counter>. It repeats this step
until this identifier does not hold any value and does not occur in <exclude>.

genVar(o,...) generates a new identifier by incrementing <counter>. It repeats this 
step until this identifier does not hold any value, does not occur in <exclude> and
does not occur in the objects 'o,...'.

ATTENTION: 
The identifier that is generated may already exist, e.g. occur in any MuPAD object. 
As long as it does not hold a value, it is not exluded. This is different from using 
the function 'genident'!

Examples:

>> miGen:= prog::genidentgenerator("mi"):
>> miGen() $ i=1..12
   mi0, mi1, mi2, mi3, mi4, mi5, mi6, mi7, mi8, mi9, mi10, mi11

>> miGen:= prog::genidentgenerator("mi_", "mo", 1, 3):
>> miGen() $ i=1..5
   `mi_{001}mo`, `mi_{002}mo`, `mi_{003}mo`, `mi_{004}mo`, `mi_{005}mo`

>> miGen:= prog::genidentgenerator("t", "", 0, 0, [excludeMe, t1, t3]):
>> miGen() $ i=1..5
   t0, t2, t4, t5, t6

>> miGen:= prog::genidentgenerator("t"):
>> miGen(t0 = t1 + t2)
   t3

This function was introduced as an internal utility function
for use in generate::C and generate::fortran.
--*/

prog::genidentgenerator:= proc(prefix     : DOM_STRING,
                               postfix="" : DOM_STRING,
                               startAt=0  : DOM_INT,
                               nDigits=0  : DOM_INT, 
                               exclude={}) 
   local  counter; 
   option escape; 
begin
   if prefix = "" then
      error("first parameter must be a non-empty string");
   end_if;
   if startAt < 0 then
      error("third parameter must be a non-negative integer");
   end_if;
   if nDigits < 0 then
      error("fourth parameter must be a non-negative integer");
    end_if;
    
   counter:= startAt - 1;

   if prefix[-1] = "_" then
      prefix := prefix."{";
      postfix:= "}".postfix;
   end_if;
      
   if exclude <> {} then
      exclude:= map(map(indets([exclude], All), expr2text), stringlib::subs, "`" = "");
   end_if;
    
   proc()
      local identifiers, newName;
    begin 
      if args(0) = 0 then
         identifiers:= {};
      else
         // make available all names of variables in 
         identifiers:= map(map(indets([args()], All), expr2text), stringlib::subs, "`" = "");
      end_if;
      repeat
         counter:= counter + 1;
         if nDigits < 2 then    // no leading 0s needed
            newName:= prefix.counter.postfix;
         elif counter = 0 then  // special case to avoid singularity [log] in next branch
            newName:= prefix._concat("0"$nDigits).postfix;
         else                   // add leading 0s to counter to get at least nDigits digits
            newName:= prefix._concat("0"$(nDigits-(trunc(log(10,float(counter)))+1))).counter.postfix;
          end_if;
          // make sure the new name is not in the given expressions and does not hold any value
      until not(newName in identifiers) and not(newName in exclude) and hold(``).newName = eval(hold(``).newName) end_repeat;
      hold(``).newName; 
   end;
end:

