/*
after prog::trace(f) where f is a procedure or a function environment, 
all the calls of f will print their input and output.

Examples:

>> prog::trace(has): has(x,y);
enter 'has'                            with args   : x, y
leave 'has'                            with result : FALSE

                                   FALSE

>> prog::untrace(has): has(x,y);

                                   FALSE

>> fib := proc(n) begin if n<2 then n else fib(n-1)+fib(n-2) end_if end_proc:
>> prog::trace(fib): fib(2);
enter 'fib'                            with args   : 2
enter 'fib'                            with args   : 1
leave 'fib'                            with result : 1
enter 'fib'                            with args   : 0
leave 'fib'                            with result : 0
leave 'fib'                            with result : 1

                                     1

>> g:= x->x + 1: prog::trace(g): g(3);
enter 'g'                              with args   : 3
leave 'g'                              with result : 4

                                     4

>> prog::trace(domtype(infinity)):
>> infinity+3;
enter 'stdlib::Infinity::_plus'        with args   : infinity, 3
enter 'stdlib::Infinity::thisDomain'   with args   : infinity
leave 'stdlib::Infinity::thisDomain'   with result : TRUE
enter 'stdlib::Infinity::thisDomain'   with args   : 3
leave 'stdlib::Infinity::thisDomain'   with result : FALSE
enter 'stdlib::Infinity::thisDomain'   with args   : infinity
leave 'stdlib::Infinity::thisDomain'   with result : TRUE
enter 'stdlib::Infinity::thisDomain'   with args   : 3
leave 'stdlib::Infinity::thisDomain'   with result : FALSE
enter 'stdlib::Infinity::removeRe'     with args   : 3
leave 'stdlib::Infinity::removeRe'     with result : 0
leave 'stdlib::Infinity::_plus'        with result : infinity

                                 infinity
*/

proc() // for static vars
  option escape, noDebug;
  local tracedFn, doTrace, doUnTrace;
begin
  // collect traced functions, for prog::untrace()
  tracedFn := table();

  // function setting the actual trace information
  // input is stored in a one-element list, to avoid implicit procname
  doTrace := proc()
    local fn, opts;
  begin
    fn := [args()];
    case domtype(fn[1])
    of DOM_EXEC do
      _pref(hold(TraceExecs)=_pref(hold(TraceExecs)) union {fn[1]});
      fn[1];
      break;
    of DOM_PROC do
      opts := op(fn[1], 3);
      if opts = NIL then opts := null(); end_if;
      opts := opts, hold(traced);
      subsop(fn[1], 3=opts);
      break;
    of DOM_FUNC_ENV do
      subsop(fn[1], 1=doTrace(op(fn[1], 1)), 3=map(op(fn[1], 3), doTrace));
      break;
    of DOM_DOMAIN do
      // Avoid tracing A::foo = id etc.
      map([op(fn[1])], 
        eq -> if domtype(op(eq, 2)) <> DOM_EXEC and
                 not (domtype(op(eq, 2)) = DOM_FUNC_ENV and domtype(op(eq, [2, 1])) = DOM_EXEC)
              then slot(fn[1], op(eq, 1), doTrace(op(eq, 2))) end_if);
      fn[1];
      break;
    otherwise
      fn[1]
    end_case;
  end:
  
  // function removing the actual trace information
  // input is a one-element list, to avoid implicit procname
  doUnTrace := proc()
    local fn, opts;
  begin
    fn := [args()];
    case domtype(fn[1])
    of DOM_EXEC do
      _pref(hold(TraceExecs)=_pref(hold(TraceExecs)) minus {fn[1]});
      fn[1];
      break;
    of DOM_PROC do
      opts := op(fn[1], 3);
      opts := op({opts} minus {hold(traced)});
      if opts = null() then opts := NIL; end_if;
      subsop(fn[1], 3=opts);
      break;
    of DOM_FUNC_ENV do
      subsop(fn[1], 1=doUnTrace(op(fn[1], 1)), 3=map(op(fn[1], 3), doUnTrace));
      break;
    of DOM_DOMAIN do
      map([op(fn[1])], 
        eq -> if domtype(op(eq, 2)) <> DOM_EXEC and
                 not (domtype(op(eq, 2)) = DOM_FUNC_ENV and domtype(op(eq, [2, 1])) = DOM_EXEC)
              then slot(fn[1], op(eq, 1), doUnTrace(op(eq, 2))) end_if);
      fn[1];
      break;
    otherwise
      fn[1]
    end_case;
  end:
  

  prog::trace := proc(fn)
    option hold;
    local fnval, opts, old_opts, opt, Args, recursive;
  begin
    if iszero(args(0)) then
      old_opts := _pref(hold(TraceOptions));
      return(table(
          Depth  = old_opts[1],
          Mem    = old_opts[2],
//          hold(Time)   = old_opts[3], // Undocumented
          NoArgs = old_opts[4],
          Parent = old_opts[5]));
    end_if;
    if not type(fn) in {DOM_IDENT, "slot", "_equal", DOM_TABLE} and
      not testtype(fn, Type::SetOf(Type::Union(DOM_IDENT, "slot"))) then
      error("first argument must be identifier or slot expression or a set of those")
    end_if;
    fnval := context([fn]); // list to avoid implicit procname
    case type(fnval[1])
    of DOM_SET do
      context(hold(map)(hold(hold)(fn), prog::trace, args(2..args(0))));
      return(null());
    of DOM_IDENT do
    of "_equal" do
      old_opts := _pref(hold(TraceOptions));
      opts := prog::getOptions(1, [args()],
        table(
          Depth  = old_opts[1],
          Mem    = old_opts[2],
          hold(Time) = old_opts[3],
          NoArgs = old_opts[4],
          Parent = old_opts[5],
          hold(Backup) = FAIL,
          Name   = FAIL,
          hold(Proc) = FAIL,
          Plain  = FAIL,
          Width  = FAIL,
          All    = FAIL),
        TRUE, table(
          Depth  = DOM_INT,
          Mem    = DOM_BOOL,
          hold(Time) = DOM_BOOL,
          NoArgs = DOM_BOOL,
          Parent = DOM_BOOL))[1];
      for opt in [hold(Backup), Name, hold(Proc), Plain, Width, All] do
        if opts[opt] <> FAIL then
          error("Option ".opt." is no longer supported. See ?prog::trace for details.");
        end_if;
      end_for;
      _pref(hold(TraceOptions) = [opts[Depth], opts[Mem], opts[hold(Time)], opts[NoArgs], opts[Parent]]);
      return(null());
    end_case;
    context(hold(prog::init)(fn));
    fn := context(hold(hold)(fn));
    [recursive, Args] := prog::getOptions(2, [args()],
      table(Recursive = TRUE,
        hold(Backup) = FAIL,
        Name   = FAIL,
        hold(Proc) = FAIL,
        Plain  = FAIL,
        Width  = FAIL,
        All    = FAIL),
        FALSE, table(Recursive = DOM_BOOL));
    for opt in [hold(Backup), Name, hold(Proc), Plain, Width, All] do
      if recursive[opt] <> FAIL then
        error("Option ".opt." is no longer supported. See ?prog::trace for details.");
      end_if;
    end_for;
    if nops(Args) > 0 then
      error("Per-procedure options are no longer supported. See ?prog::trace for details.");
    end_if;
    tracedFn[fn] := TRUE;
    if recursive[Recursive] = FALSE then
      if domtype(fnval[1]) = DOM_FUNC_ENV then
        context(hold(sysassign)(fn, subsop(fnval[1], 1=doTrace(op(fnval[1], 1)))));
        return();
      end_if;
    end_if;
    context(hold(sysassign)(fn, doTrace(fnval[1])));
    null();
  end:
  
  prog::untrace := proc(fn)
    option hold;
    local fnval;
  begin
    if args(0) = 0 then
      map([op(tracedFn)], f -> eval(hold(prog::untrace)(op(f, 1))));
      return(null());
    end_if;
    fnval := context([fn]); // list to avoid implicit procname
    if domtype(fnval[1]) = DOM_SET then
      context(hold(map)(hold(hold)(fn), prog::untrace, args(2..args(0))));
      return(null());
    end_if;
    fn := context(hold(hold)(fn));
    delete tracedFn[fn];
    context(hold(sysassign)(fn, doUnTrace(fnval[1])));
    null();
  end:
  
  prog::traced := proc(fn)
    option hold;
  begin
    if args(0) = 0 then
      return(map([op(tracedFn)], op, 1));
    end_if;
    fn := context(hold(hold)(fn));
    contains(tracedFn, fn);
  end_proc:

  // will learn about options, too
  prog::fullTrace := stdlib::trace:
end_proc():

null():

