//

/* property::hasroot
 *
 * Tests if there are roots of f inside the interval ]l, r[.
 * Note: The caller should make sure that there are no roots
 *       on the bounds (i.e. f(l)<>0 and f(r)<>0). If there
 *       are roots, this function will return UNKNOWN intentionally.
 */
property::hasroot := proc(f, ind, l, r)
  local i, S, eps;
begin
  // polylib::realroots only handles coefficient that can be rationalized
  if map({coeff(f)}, type@float)<>{DOM_FLOAT} then
    return(UNKNOWN);
  end_if;
  /* a real polynomial of odd degree always has a real root */
  if l=-infinity and r=infinity and degree(f) mod 2 = 1 then 
    return(TRUE)
  end_if;  
  eps := 1;
  if traperror(([l, r] := float([l, r])))<>0 or {type(l), type(r)}<>{DOM_FLOAT} then return(UNKNOWN); end_if;
  if r-l<eps then eps := r-l end_if;
  for i from 1 to 5 do
    S := polylib::realroots(f, eps);
    if S=[] then return(FALSE); end_if;
    // test if all roots are outside the interval
    if {op(map(S, X->bool(op(X,1)>r or op(X,2)<l)))}={TRUE} then
      return(FALSE);
    end_if;
    // test if one root is inside the interval
    if contains(map(S, X->bool(op(X,2)<r and op(X,1)>l)), TRUE)>0 then
      return(TRUE);
    end_if;
    eps := eps/2;
  end_for;
  return(UNKNOWN);
end_proc:

/* property::_decide
  * tries to make some fast decisions whether cond is TRUE or FALSE
  * property::_decide can be seen as the small brother of is()
  */
property::_decide := proc( cond, options=table() )
  local inds, sidea, sideb, v1, v2, R, lhs, rhs, i, p, v, j, base, ex, props,
    zeroFactored, sidear, sidebr, 
    check: DOM_PROC,
    removeRoots;
begin
  /* removeRoots (local property::_decide)
   * Input:
   *   xpr: Any mathematical expression
   * Output:
   *   An mathematical expression g with sign(xpr)=sign(g).
   *   If possible roots are removed.
   * Example:
   *   removeRoots(x-sqrt(x)) assuming x>0 => x^2-x
   */
  removeRoots := proc(xpr)
    local X, Y, fact, xp, res;
  begin
    res := property::splitRoots(xpr);
    if res=FAIL then return(xpr); end_if;
    [fact, Y, xp, X] := res;
    // xpr = fact*Y + X
    /* We want to decide X+Y<0 */
    if xp mod 2=0 then
      // X+Y^(a/b) < 0 => X < -Y^(a/b) => X^b < -Y^a => X^b + Y^a < 0
      if property::_decide(X<=0, options)=TRUE and property::_decide(fact*Y>=0, options)=TRUE then
        /* X+Y^(a/b) < 0 && X<=0<=Y
        * <=> X < -Y^(a/b)
        * <=> X^b > Y^a
        * <=> - X^b + Y^a < 0
        */
        xpr := -X^xp + fact^xp*Y^xp;
      elif property::_decide(X>=0, options)=TRUE and property::_decide(fact*Y<=0, options)=TRUE then
        /* X+Y^(a/b) < 0 && Y<=0<=X
        * <=> X < -Y^(a/b)
        * <=> X^b < Y^a
        * <=> X^b - Y^a < 0
        */
        xpr := X^xp - fact^xp*Y^xp;
      end_if;
    else
      // X+Y^(a/b) < 0 => X < -Y^(a/b) => X^b < -Y^a => X^b + Y^a < 0
      xpr := X^xp + fact^xp*Y^xp;
    end_if;
    xpr;
  end_proc:

  inds := freeIndets( cond );
  case type( cond )
    of "_not" do
      return( not property::_decide( op(cond), options ) );
    of "_leequal" do
    of "_less" do
      sidear := null();
      sidebr := null();

      /* schneller Test auf konstante Relationen */
      if inds={} then
        /* this is necessary to have true for infinity = infinity */
        if op(cond,1)=op(cond,2) then
          case type(cond)
            of "_leequal" do
              if op(cond,1)=infinity or (sidebr := sidear := property::_typereal(op(cond,1), options))=TRUE then
                return(TRUE);
              end_if;
              break;
            of "_unequal" do
            of "_less" do
              return(FALSE);
          end_case;
        end_if;
        v := property::constRel( cond );
        return( v );
      end_if;
      if options[ "Constant" ]=TRUE then return( UNKNOWN ); end_if;
      if sidear=null() then sidear := property::_typereal( op(cond,1), options ); end_if;
      if sidebr=null() then sidebr := property::_typereal( op(cond,2), options ); end_if;

      if sidear<>TRUE or sidebr<>TRUE then
        if sidear<>TRUE then
          if ( lhs := property::_rectform(op(cond,1), options) )<>FAIL then
            if property::_decide( lhs[2]=0 )=FALSE then return(FALSE); end_if;
          end_if;
        else
          lhs := [ op(cond,1), 0 ];
        end_if;
        if sidebr<>TRUE then
          if ( rhs := property::_rectform(op(cond,2), options) )=FAIL then return(UNKNOWN); end_if;
          if property::_decide( rhs[2]=0 )=FALSE then return(FALSE); end_if;
        else
          rhs := [ op(cond,2), 0 ];
        end_if;
        if lhs=FAIL then return(UNKNOWN); end_if;
        // X<Y => Re(X)<Re(Y). If not Re(X)<Re(Y) then X<Y can't hold.
        if (op(cond,1)<>lhs[1] or op(cond,2)<>rhs[1]) and property::_decide(subsop(cond,1=lhs[1], 2=rhs[1]), options)=FALSE then return(FALSE); end_if;
      else
        [ lhs, rhs ] := [op(cond)];
        if nops(inds)=1 and ( p:=Type::Linear(lhs-rhs, [op(inds)]) )<>FAIL and p<>FALSE then
          if p[1]=0 then
            lhs := 0;
            rhs := -p[2];
          else
            lhs := op(inds);
            rhs := -p[2]/p[1];
          end_if;
          case property::_decide(p[1]>=0)
            of FALSE do:
              [lhs,rhs] := [rhs,lhs];
            of TRUE do
              cond := subsop( cond, 1=lhs, 2=rhs );
          end_case;
        end_if;
      end_if;

        j := _if( indets( op(cond,1) ) minus Type::ConstantIdents<>{}, 1, 0 ) + _if( indets( op(cond,2) ) minus Type::ConstantIdents<>{}, 2, 0 );
        if j=0 then
          return( property::constRel(cond) );
        end_if;
        if j=1 or j=2 then
          if j=2 then [ sidear, sidebr ] := [ sidebr, sidear ]; end_if;
          sidea := op( cond, j );
          sideb := op( cond, 3-j );
          if type(sidea)="_mult" then
            v := split(sidea, X->contains({DOM_INT, DOM_RAT, DOM_FLOAT}, type(X)));
            if v[1]<>1 and v[1]>0 then
              sidea := sidea/v[1];
              sideb := sideb/v[1];
            elif v[1]<0 then
              j := 3-j;
              sidea := sidea/v[1];
              sideb := sideb/v[1];
            end_if;
          end_if;
          case type(sidea)
            of "_power" do
              if sideb=0 then
                [ base, ex ] := [ op(sidea) ];
                /* a>0   and b in R_ => a^b > 0   => NOT a^b <= 0
                  * a=>0 and b in R_ => a^b >= 0  => NOT a^b < 0
                  */
                if property::_typereal(base, table(options, "seek"=TRUE))=TRUE and property::_typereal(ex, table(options, "seek"=TRUE))=TRUE then
                  if ( j=1 and type(cond)="_leequal" ) or ( j=2 and type(cond)="_less" ) then
                    /* case 1: sidea <= 0 or 0 < sidea */
                    v := property::_decide( 0 < base );
                  else
                    /* case 2: sidea < 0 or 0 <= sidea */
                    v := property::_decide( 0<= base );
                  end_if;
                  if v=TRUE then return( bool(j=2) ); end_if;

                  /* Check if we have a odd exponent */
                  if v=FALSE and property::_typeodd( ex, options )=TRUE then
                    return( bool(j=1) );
                  end_if;

                  /* Base can not be decided to be positive or negative, check if exponent is even */
                  v := property::_typeeven( ex, options );
                  if v=TRUE then
                    if j=2 and type(cond)="_leequal" or j=1 and type(cond)="_less" then return( bool(j=2) ); end_if;
                    v := property::_decide( base<>0, options );
                    if v=TRUE then
                      if j=2 and type(cond)="_less" or j=1 and type(cond)="_leequal" then return( bool(j=2) ); end_if;
                    end_if;
                  end_if;
                end_if;
              end_if;
              break;
            of "fact" do
              if j=1 then
                /* case: foo! </<= bar */
                /* this is FALSE if bar is less then 1*/
                if type(cond)="_leequal" and property::_decide(op(sidea)<>1)<>TRUE then
                  v := property::_decide(sideb<1);
                else
                  v := property::_decide(sideb<=1);
                end_if;
                if v=TRUE then return(FALSE); end_if;
              else
                /* case: foo </<= bar! */
                /* this is TRUE if foo is less then 1*/
                if type(cond)="_leequal" or property::_decide(op(sidea)<>1)=TRUE then
                  v := property::_decide(sideb<=1);
                else
                  v := property::_decide(sideb<1);
                end_if;
                if v=TRUE then return(TRUE); end_if;
              end_if;
              break;
            of "abs" do
              if j=1 then
                /* case: abs(foo) </<= bar */
                /* this is FALSE if bar is negative */
                if type(cond)="_leequal" and property::_decide(op(sidea)<>0)<>TRUE then
                  v := property::_decide(sideb<0);
                else
                  v := property::_decide(sideb<=0);
                end_if;
                if v=TRUE then return(FALSE); end_if;
              else
                /* case: foo </<= abs(bar) */
                /* this is TRUE if foo is negative */
                if type(cond)="_leequal" or property::_decide(op(sidea)<>0)=TRUE then
                  v := property::_decide(sideb<=0);
                else
                  v := property::_decide(sideb<0);
                end_if;
                if v=TRUE then return(TRUE); end_if;
              end_if;
              /* If x is real then:
               * abs(x)<Y <=> x in ]-Y, Y[
               * abs(x)<=Y <=> x in [-Y, Y]
               * abs(x)>Y <=> not x in [-Y, Y]
               * abs(x)>=Y <=> not x in ]-Y, Y[
               */
              if property::_typereal(op(sidea),table(options, "seek"=TRUE))=TRUE then
                if (type(cond)="_leequal") <=> (j=1) then
                  v := [-sideb], [sideb];
                else
                  v := -sideb, sideb;
                end_if;
                return(property::_decide(op(sidea) in Dom::Interval(v)) <=> bool(j=1));
              end_if;
              break;
            of "ln" do /* ln(x)<0 <= x<1, ln(x)>0 <= x>1 */
              if sidebr=TRUE then
                v := property::_decide(subsop(cond, j=op(sidea), 3-j=exp(sideb)));
                if v=TRUE then return(TRUE); end_if;
                if v=FALSE and sidear=TRUE then return(FALSE); end_if;
              end_if;
              break;
            of "exp" do
              if sidear<>TRUE then break; end_if;
              if property::_decide(sideb<=0)=TRUE then
                if j=2 then return(TRUE); end_if; /* foo < exp(bar) is TRUE if foo <= 0 */
                if j=1 then return(FALSE); end_if; /* exp(bar) <  foo is FALSE if foo <= 0 */
              end_if:
              if sidebr=TRUE then
                v := property::_decide(subsop(cond, j=op(sidea), 3-j=ln(sideb)));
                if v<>UNKNOWN then return(v); end_if;
              end_if;
              break;
            of "sinh" do
              if property::_typereal(op(sidea), options)<>TRUE then return(UNKNOWN); end_if;
              /* sinh(X) < C  <=>  X < arcsinh(C)*/
              if j=1 then
                return(property::_decide(op(cond,0)(op(sidea), arcsinh(sideb)), options));
              else
                return(property::_decide(op(cond,0)(arcsinh(sideb), op(sidea)), options));
              end_if;
            of "cosh" do
              if property::_typereal(op(sidea), options)<>TRUE then return(UNKNOWN); end_if;
              if j=1 then
                /* Case cosh(x) <|<= C <=> x in ]-arccosh(C), arccosh(C)[ */
                case property::constRel(sideb<1)
                  of TRUE do
                    /* cosh(x) <|<= C < 1 is FALSE */
                    return(FALSE);
                  of FALSE do
                    if type(cond)="_leequal" then
                      /* cosh(x) <= C  <=>  x in [-arccosh(C), arccosh(C)] */
                      return(property::_decide(op(sidea) in Dom::Interval([-arccosh(sideb)], [arccosh(sideb)]), options));
                    else
                      /* cosh(x) < C  <=>  x in ]-arccosh(C), arccosh(C)[ */
                      return(property::_decide(op(sidea) in Dom::Interval(-arccosh(sideb), arccosh(sideb)), options));
                    end_if;
                end_case;
              elif j=2 then
                /* Case cosh(x) >|>= C <=> x<-arccosh(C) or x>=arccosh(C)*/
                if type(cond)="_leequal" then
                  v := property::constRel(sideb<=1);
                else
                  v := property::constRel(sideb<1);
                end_if;
                case v
                  of TRUE do
                    /* cosh(x) > C < 1 is TRUE
                     * cosh(x) >= C <= 1 is TRUE */
                    return(TRUE);
                  of FALSE do
                    if type(cond)="_leequal" then
                      /* Case cosh(x) >= C <=> x<=-arccosh(C) or x>=arccosh(C)*/
                      return(property::_decide(op(sidea)<=-arccosh(sideb), options) or property::_decide(op(sidea)>=arccosh(sideb), options));
                    else
                      /* Case cosh(x) > C <=> x<-arccosh(C) or x>arccosh(C)*/
                      return(property::_decide(op(sidea)<-arccosh(sideb), options) or property::_decide(op(sidea)>arccosh(sideb), options));
                    end_if;
                end_case;
              end_if;
              break;            
            of "sin" do
            of "cos" do
              if property::_typereal(op(sidea), options)<>TRUE then return( UNKNOWN ); end_if;
              if j=1 then
                if (property::constRel( op(cond,0)(1,sideb)) )=TRUE then return( TRUE ); end_if;
                if (property::constRel( op(cond,0)(sideb,-1)) )=TRUE then return( FALSE ); end_if;
              else
                if (property::constRel( op(cond,0)(1,sideb)) )=TRUE then return( FALSE ); end_if;
                if (property::constRel( op(cond,0)(sideb,-1) ))=TRUE then return( TRUE ); end_if;
              end_if;
              break;

            /* fast decision of X <= constant depending on properties */
            of DOM_IDENT do
              if ( v1:=property::_getprop( sidea, "Constant"=TRUE,
                                          "Targets"={Dom::Interval},
                                          "StrictTargets"=TRUE ) )<>C_ then
                case type(v1)
                  of "_intersect" do
                    if ( i := contains( map( ( v2 := [op(v1)] ), type ),
                                       Dom::Interval) )=0 then
                      break;
                    end_if;
                    v1 := v1[i];
                    // fall through !!!
                  of solvelib::BasicSet do
                    if v1 = R_ then
                      if j=1 then /* X </<= constant. */
                        v2 := [ infinity, -infinity ];
                      else /* const </<= X */
                        v2 := [ -infinity, infinity ];
                      end_if;
                    else
                      assert(type(v1) = Dom::Interval)
                    end_if;
                    // fall through
                  of Dom::Interval do

                    if v1 <> R_ then
                      if j=1 then /* X </<= constant. */
                        v2 := [ Dom::Interval::rightB( v1 ), Dom::Interval::leftB( v1 ) ];
                      else /* const </<= X */
                        v2 := [ Dom::Interval::leftB( v1 ), Dom::Interval::rightB( v1 ) ];
                      end_if;
                    end_if;

                    /* 1. check if the interval X is in the interval spanned by cons.
                      * we have sidea <= sideb  =>  [a,b] <= const. if b <= const this is true
                      */
                    if type(v2[1])=DOM_LIST then
                      R := subsop( cond, j=op(v2,[1,1]) )
                    else
                      /* if the right side is open, it can be less or equal const. */
                      R := _leequal( _if(j=1, (op(v2,1),sideb), (sideb,op(v2,1)) ) );
                    end_if;
                    if indets( R ) minus Type::ConstantIdents={} then
                      if property::constRel(R)=TRUE then return(TRUE); end_if;
                    end_if;
                    /* 2. check if the interval X is outside the interval spanned by cons
                      * we have sidea <= sideb => [a,b] <= const. if a>const this is false
                      */
                    if type(v2[2])=DOM_LIST then
                      if type(cond)="_less" then
                        R := _leequal(_if(j=2, (op(v2,[2,1]),sideb), (sideb,op(v2,[2,1])) ) );
                      else
                        R := _less(_if(j=2, (op(v2,[2,1]),sideb), (sideb,op(v2,[2,1])) ) );
                      end_if;
                    else
                      /* if the left side is open, it can be greater or equal const. */
                      R := _leequal(_if(j=2, (op(v2,2),sideb), (sideb,op(v2,2)) ) );
                    end_if;
                    if indets( R ) minus Type::ConstantIdents={} then
                      if property::constRel(R)=TRUE then return(FALSE); end_if;
                    end_if;
                    break;
                  of piecewise do
                    break;
                  otherwise
                    if not testtype( v1, Type::Constant ) then break; end_if;
                    if type(cond)="_leequal" then v2 := [sideb]; else v2 := sideb; end_if;
                    if j=1 then v2 := Dom::Interval( -infinity, v2 ); else v2 := Dom::Interval( v2, infinity ); end_if;
                    if _subset( v1, v2 )=TRUE then return( TRUE ); end_if;
                    v2 := v2 intersect v1;
                    if solvelib::isEmpty( v2 )=TRUE then return( FALSE ); end_if;
                    break;
                end_case;
              end_if;
              /* Wenn das hier nicht entschieden werden konnte, werden wir unten keine weitere Rechenzeit
               * darein investieren.
               */
              return( UNKNOWN );
            otherwise
          end_case;
        end_if;
        /* check if sidea-sideb is a univariant polynom */
        if nops(inds)<>1 then break; end_if;
        inds := op(inds);
        lhs := op(cond,1)-op(cond,2);;

        if type(lhs)="_plus" and hastype(lhs, "_power") then
          lhs := removeRoots(lhs);
          if indets(lhs) minus Type::ConstantIdents={} and sidear=TRUE and sidebr=TRUE then
            return(property::constRel(subsop(cond,1=lhs,2=0)));
          end_if;
        end_if;
        rhs := lhs;
        if ( indets( lhs, PolyExpr )<> {inds} ) then break; end_if:

        props := property::_getprop( inds, "Constant"=TRUE, "Targets"={Dom::Interval}, "StrictTargets"=TRUE );
        if type(props)="_intersect" then
          if (i := contains( map( ( props := [op(props)] ), type ),
                            Dom::Interval ) )<>0 then
            props := props[i]
          elif (i:= contains(props, R_)) > 0 then
            props := props[i]
          else
            props := FAIL;
          end_if;
        elif type(props)<>Dom::Interval then
          if property::_typereal(inds, table(options, "seek"=TRUE))=TRUE then
            props := R_
          else
            props := FAIL;
          end_if;
        end_if;
        if props=FAIL then break end_if;
        if props = R_ then
          v:= -infinity .. infinity
        else
          v := Dom::Interval::left( props )..Dom::Interval::right( props )
        end_if;

        if freeIndets(v)<>{} then break; end_if;
        if sidear<>TRUE or sidebr<>TRUE then break; end_if; /* everything real? */

        
        if ( p := poly( lhs, [inds] ) )=FAIL then break; end_if; /* is this a polynom? */
        
        
        if nterms(p) = 2 and not iszero(coeff(p, 0)) then
          // we want to avoid factoring x^1000 - 1 and similar polynomials
          // a*x^n + b <= 0 for all x iff this holds at all extremal points 
          
          // local method check(X, isopen)
          // returns:
          //         TRUE if p(X) < 0, or if p(X) = 0 at a value X that can be attained, if this does not contradict the properties of inds
          //         FALSE if p(X) > 0, or p(X) = 0 at a value that can be attained by inds (closed border), if this contradicts the property of inds (namely, p(inds) < 0)
          //         null()  if p(X) = 0 at a value X that cannot be attained by inds
          check:=
          proc(X, isopen)
          begin
            if X = infinity then 
              property::_decide(lcoeff(p) < 0)
            elif X = -infinity then
              if degree(p) mod 2 = 0 then 
                property::_decide(lcoeff(p) < 0)
              else
                property::_decide(lcoeff(p) > 0)
              end_if  
            elif property::_decide(p(X) = 0) = TRUE then
              if not isopen then 
                return(bool(type(cond) = "_leequal"))
              else 
                null()
              end_if   
            else 
              property::_decide(p(X) < 0)
            end_if
          end_proc;  
          
          
          // extremal points may be: x=0, x=op(v, 1), x = op(v, 2)
          if property::_decide(op(v, 1) < 0 ) = TRUE and property::_decide(op(v, 2) > 0) = TRUE then 
            ex:= {check(0, FALSE), check(op(v, 1), Dom::Interval::isleftopen(props)), check(op(v, 2), Dom::Interval::isrightopen(props))}
          else
            ex:= {check(op(v, 1), Dom::Interval::isleftopen(props)), check(op(v, 2), Dom::Interval::isrightopen(props))}
          end_if;  
          case ex 
            of {TRUE} do  
            of {} do   
              return(TRUE)
            of {FALSE} do 
              return(FALSE)
            of {FALSE, TRUE} do 
              return(UNKNOWN)
          end_case;
          // NOT REACHED
          break;
        end_if;  
              

        /* Nullstellen am Rand prfen */
        zeroFactored := FALSE;
        if op(v,1)<>-infinity then
          while not iszero(p) and iszero(p(op(v,1))) do
            R := [ divide( p, poly(inds-op(v,1), [inds]) )];
            if R = [FAIL] then break; end_if;
            if iszero(R[2]) then p := R[1]; else break; end_if;
            zeroFactored := not Dom::Interval::isleftopen(props);
          end_while;
        end_if;
        if op(v,2)<>infinity then
          while not iszero(p) and iszero(p(op(v,2))) do
            R := [ divide( p, poly(inds-op(v,2), [inds]))];
            if R = [FAIL] then break; end_if;
            if iszero(R[2]) then p := R[1]; else break; end_if;
            zeroFactored := zeroFactored or not Dom::Interval::isrightopen(props);
          end_while;
        end_if;
        
     
        
        if freeIndets( coeff(p,0) )<>{} then break; end_if; /* we have to check the constant coefficient to be real constant */
        if degree( p )<1 then
          // Note: We use coeff(p, 0) since lhs could syntactically have
          // an identifier (like "(a+1)^2 - a^2 - 2a") which could lead
          // to an endless recursion..
          if property::_decide(coeff(p, 0)=0, options )<>FALSE then
            break;
          end_if;
        else
          if property::hasroot( expr(p), inds, op(v)) <> FALSE then break; end_if; /* do we have roots? */
        end_if;

        if op(v,1)=-infinity then
          if op(v,2)=infinity then
            v := 0;
          else
            v := op(v,2)-1;
          end_if;
        else
          if op(v,2)=infinity then
            v := op(v,1)+1;
          else
            v := ( op(v,1)+op(v,2) )/2;
          end_if;
        end_if;

        if zeroFactored then
          /* wenn eine Null am Rand rausfaktorisiert wurde und der Bedingungstyp <= ist, kann die Bedingung nicht mit
           * FALSE beantwortet werden (da am Rand eine Nullstelle ist)
           * Wenn der Bedingungstyp < ist, kann die Bedingung nicht mit TRUE beantwortet werden, da am Rand eine Nullstelle
           * ist.
           */
          if type(cond)="_leequal" then
            return( UNKNOWN or property::constRel( subs(rhs, inds=v)<0 ) ); /* if we have no roots, then check the sign */
          else
            return( UNKNOWN and property::constRel( subs(rhs, inds=v)<0 ) ); /* if we have no roots, then check the sign */
          end_if;
        else
          return( property::constRel( subs(rhs, inds=v)<0 ) ); /* if we have no roots, then check the sign */
        end_if;
        // NOT REACHED
        assert(FALSE);
        break;
    of "_equal" do
      /* check constant equation */
      if inds={} then return( property::constRel(cond) ); end_if;
      [lhs, rhs] := [op(cond)];
      if bool(lhs=rhs)=TRUE then return(TRUE); end_if;
      if nops(inds)=1 and ( p:=Type::Linear(lhs-rhs, [op(inds)]) )<>FAIL and p<>FALSE and p[1]<>0 then
        lhs := op(inds);
        rhs := -p[2]/p[1];
      end_if;
      /* if one side is real but not the other than this is FALSE */
      if ( property::_typereal(lhs, options) <=> property::_typereal(rhs, options) )=FALSE then return(FALSE); end_if;
      /* check the properties */
      if indets(lhs) minus Type::ConstantIdents<>{} and indets(rhs) intersect indets(lhs) minus Type::ConstantIdents={} then
        v:=property::_getprop( lhs, "Constant"=TRUE );
        if v <>C_ then
          /* if v is a set and its one item equals rhs the equation is true */
          if {rhs}=v then return(TRUE); end_if;
          if property::_decide( rhs in v )=FALSE then return(FALSE); end_if;
        end_if;
      end_if;
      /* check some special cases for X = 0 */
      if lhs=0 then [ rhs, lhs ] := [ lhs, rhs ]; end_if;
      if rhs=0 then
        case type(lhs)
          of "exp" do return(FALSE);
          of "fact" do return(FALSE);
          of "_power" do if type(op(lhs,2))=DOM_INT and op(lhs,2)<0 then return(FALSE); end_if;
            break;
        end_case;
      end_if;

      /* check if sidea-sideb is a univariant polynom */
      if nops(inds)<>1 then break; end_if;
      inds := op(inds);
      lhs := op(cond,1); rhs := op(cond,2);
      if property::_typereal(lhs, options)<>TRUE or property::_typereal( rhs, options )<>TRUE then break; end_if; /* everything real? */
      lhs := lhs-rhs;
      if type(lhs)="_plus" and hastype(lhs, "_power") then
        lhs := removeRoots(lhs);
        if indets(lhs) minus Type::ConstantIdents={} then
          return(property::constRel(lhs=0));
        end_if;
      end_if;

      if ( indets( lhs, PolyExpr )<> {inds} ) then break; end_if:

      /* check if we have a domain */
      props := property::_getprop( inds, "Constant"=TRUE );
      if type(props)="_intersect" then
        if (i := contains( map( ( props := [op(props)] ), type ), Dom::Interval ) )<>0 then
          props := props[i];
        else
          props := FAIL;
        end_if;
      elif type(props)<>Dom::Interval then
        if property::_typereal(inds, table(options, "seek"=TRUE))=TRUE then
          props := R_
        else
          props := FAIL
        end_if;
      end_if;
      if props=FAIL then break; end_if;
      if props = R_ then
        v:= -infinity .. infinity
      else
        v := Dom::Interval::left( props )..Dom::Interval::right( props )
      end_if;
      if freeIndets(v)<>{} then break; end_if;

      // remove roots at the end of the interval
      if op(v,1)<>-infinity then
        while lhs <> 0 and iszero(evalAt(lhs, inds = op(v,1))) do
          R := [ divide( lhs, inds-op(v,1) )];
          if R = [FAIL] then return(UNKNOWN); end_if;
          if R[2]=0 then lhs := R[1]; else return(UNKNOWN); end_if;
          if not Dom::Interval::isleftopen(props) then return(UNKNOWN); end_if;
        end_while;
      end_if;
      if op(v,2)<>infinity then
        while lhs <> 0 and iszero(evalAt( lhs, inds = op(v,2) )) do
          R := [ divide( lhs, inds-op(v,2) )];
          if R = [FAIL] then return(UNKNOWN); end_if;
          if R[2]=0 then lhs := R[1]; else return(UNKNOWN); end_if;
          if not Dom::Interval::isrightopen(props) then return(UNKNOWN); end_if;
        end_while;
      end_if;

      if ( p := poly( lhs, [inds] ) )=FAIL then break; end_if; /* is this a polynom? */
      if freeIndets( coeff(p,0) )<>{} then break; end_if; /* we have to check the constant coefficient to be real constant */
      if degree( p )<1 then
        // Note: We use coeff(p, 0) since lhs could syntactically have
        // an identifier (like "(a+1)^2 - a^2 - 2a") which could lead
        // to an endless recursion.
        return(property::_decide(coeff(p, 0)=0, options));
      end_if;
      if property::hasroot( expand(lhs), inds, op(v)) <> FALSE then break; end_if; /* do we have roots? */
      return( FALSE ); /* if we have no roots, then check the sign */
      break;
    of "_unequal" do
      return( not property::_decide( op(cond,1)=op(cond,2) ) );
    of "_in" do
      sidea := op(cond,1);
      sideb := op(cond,2);
      if type(sideb) = "solve" then
        v := expand(cond);
        if type(v)=DOM_BOOL then return(v); end_if;
        if type(v)="_equal" then
          if map( [op(v)], testtype, Type::Arithmetical)=[TRUE,TRUE] then return( property::_decide(v) ); end_if;
        end_if;
      end_if;
      if sideb=C_ then return(TRUE); end_if;
      if sideb=R_ then return( property::_typereal(sidea, options) ); end_if;
      if sideb=Z_ then return( property::_typeinteger(sidea,options) ); end_if;
      if sideb=Q_ then return( property::_typerational(sidea,options) ); end_if;
      if not hastype(sideb, {"solve"} union rationalize::hardToEval) and (R := hull( sideb )) <> FAIL then
        if traperror( (v:=float(sidea)) ) <> 0 then return(FALSE) end_if;
        if type(v)=DOM_FLOAT then
          if bool(v in R)=FALSE then return( FALSE ); end_if;
        end_if;
      end_if;
      case type(sideb)
          of DOM_SET do
            return( _or( op(map( (sideb), X->property::_decide( X=sidea ) )) ) );
          of "_union" do
            return(_lazy_or(op(map([op(sideb)], X->property::_decide(sidea in X)))));
          of "_intersect" do
            return(_lazy_and(op(map([op(sideb)], X->property::_decide(sidea in X)))));
          of "_minus" do
            return( property::_decide( sidea in op(sideb,1) ) and not property::_decide( sidea in op(sideb,2) ) );
          of piecewise do
            break;
          of Dom::Interval do
            R := property::_typereal(sidea, options);
            if R=FALSE then return( FALSE ); end_if;
            if R=UNKNOWN then break; end_if;
            if Dom::Interval::left( sideb )=-infinity then
              v := TRUE;
            elif Dom::Interval::isleftopen(sideb) then
              v := property::_decide( Dom::Interval::left( sideb ) < sidea, options );
            else
              v := property::_decide( Dom::Interval::left( sideb ) <= sidea, options );
            end_if;
            if v = FALSE then return( FALSE ); end_if;
            if Dom::Interval::right( sideb )=infinity then
              //
            elif Dom::Interval::isrightopen(sideb) then
              v := v and property::_decide( sidea < Dom::Interval::right( sideb ), options );
            else
              v := v and property::_decide( sidea <= Dom::Interval::right( sideb ), options );
            end_if;
            return( v );
          of Dom::ImageSet do
            cond := Dom::ImageSet::simpexIn(op(cond), op(cond, 2));
            if cond=TRUE or cond=FALSE then return(cond); end_if;
            break;
      end_case;
      if freeIndets(sidea)<>{} then
        case type(sidea)
          of DOM_IDENT do
            v := property::_getprop( sidea, "Constant"=TRUE );
            if v<>C_ then
              if _subset( v, sideb )=TRUE then return(TRUE); end_if;
              if testtype( v, Type::Constant ) then
                v := _intersect( v, sideb);
                if v={} then return(FALSE); end_if;
              end_if;
            end_if;
            if _subset( {sidea}, sideb )=TRUE then return(TRUE); end_if;
            case type(sideb)
              of "_minus" do
                v := property::_decide( sidea in op(sideb,1) );
                if v=FALSE then return(FALSE); end_if;
                v := property::_decide( sidea in op(sideb,2) );
                if v=TRUE then return(FALSE); end_if;
                break;
              of Dom::Interval do
                if sidea=Dom::Interval::left(sideb) then return( _lazy_and( not Dom::Interval::isleftopen(sideb), property::_decide( sidea<sideb )  ) ); end_if;
                if sidea=Dom::Interval::right(sideb) then return( _lazy_and( not Dom::Interval::isrightopen(sideb), property::_decide( sidea<sideb) ) ); end_if;
                break;
            end_case;
            break;
          of "sin" do
          of "cos" do
            if property::_typereal(op(sidea),table(options, "seek"=TRUE))=TRUE then
              v := _subset( Dom::Interval( [-1], [1] ), sideb );
              if type(v)=DOM_BOOL then return(v); end_if;
            end_if;
        end_case;
      end_if;
  end_case;
  UNKNOWN;
end_proc: /* property::_decide */
