/* property::checkPolyInteger
 *
 * Input:
 *  xpr: A polynomial in Q_[x1...xn]
 *  doSimplify: If TRUE an more simple equivalent expression is returned instead of UNKNOWN
 * Output:
 *  TRUE or FALSE if xpr in Z_ can be decided
 *  UNKNOWN or an equivalent expression according to the input doSimplify
 */
property::checkPolyInteger := proc(xpr, doSimplify=FALSE)
  local inds, p, fact, modulo, facts, factors, exponents, p2, insertMod, conds, i, j, res,
    systems, Mi, g, s, t;
begin
  if not doSimplify then
    xpr := property::simplifyZPolyResidueProp(xpr);
  end_if;

  /* insertMod
   * Inserts values and calculates a result set modulo the value "modulo"
   * Input:
   *  xpr: The mathematical expression, containing indets
   *  inds: The indets that are to be replaced
   *  modulo: The modulo to be calculated
   * Output:
   *  FAIL if the calculation doesn't work
   *  In any other case a set of integers
   */
  insertMod := proc(xpr, inds, modulo)
    local ind, i, subxpr, res, tmp;
  begin
    res := {};
    ind := op(inds,1);
    inds := [op(inds,2..nops(inds))];
    for i from 0 to modulo-1 do
      subxpr := subs(xpr, ind=i);
      if type(subxpr)=DOM_INT then
        res := res union {subxpr mod modulo};
        if nops(res)=modulo then return(res); end_if;
        next;
      end_if;
      if nops(inds)>0 then
        tmp := insertMod(subxpr, inds, modulo);
        if tmp=FAIL then return(FAIL); end_if;
        res := res union tmp;
        if nops(res)=modulo then return(res); end_if;
        next;
      end_if;
      return(FAIL);
    end_for;
    return(res);
  end_proc:

  inds := indets( xpr ) minus Type::ConstantIdents;
  /* berprfen, dass xpr ein Polynom in Q_[x1...xn] */
  p := poly(xpr, [op(inds)], Dom::Rational);
  if p=FAIL then return(UNKNOWN); end_if;

  /* Hauptnenner berechnen */
  fact := ilcm(op(map([coeff(p)], denom)));

  /* Hauptnenner = 1 => xpr ist Polynom in Z_[x1..xn] */
  if fact=1 then
    return(TRUE);
  end_if;

  /* Polynom um Hauptnenner erweitern */
  p := mapcoeffs(p, _mult, fact);
  modulo := fact;

  /* Das Polynom ist immer ganzzahlig, wenn
   * xpr*Hauptnenner = 0  mod Hauptnenner
   * gilt
   */
  facts := ifactor(modulo);
  if type(facts)<>Factored then return(UNKNOWN); end_if;
  factors := Factored::factors(facts);
  exponents := Factored::exponents(facts);
  assert(nops(factors)=nops(exponents));
  conds := TRUE;

  if doSimplify then
    systems := [];
  else
    systems := FAIL;
  end_if;

  /* Wir gehen jeden Faktor durch */
  for i from 1 to nops(factors) do
    if nops(inds)=1 then
      p2 := property::singleRootsZPoly(expr(p), factors[i], op(inds));
    else
      p2 := property::reduceZPolyModPrime(expr(p), factors[i], inds);
      p2 := poly(p2, [op(inds)], IntMod(factors[i]));
    end_if;
    if systems<>FAIL then
      if exponents[i]=1 then
        systems := systems.[[p2,factors[i]]];
      else
        systems := FAIL;
      end_if;
    end_if;
    if exponents[i]=1 then
      /* Exponent = 1 */
      if degree(p2)=0 then
        conds := conds and property::_decide(lcoeff(p2)=0);
        if conds=FALSE then return(FALSE); end_if;
      else
        conds := UNKNOWN;
      end_if;
      next;
    end_if;

    /* Wenn der Exponent nicht =next 1 ist, ist dieser Test nur eine notwendige Bedingung */
    if degree(p2)=0 and property::_decide(lcoeff(p2)=0)=FALSE then return(FALSE); end_if;

    p2 := poly(p, IntMod(factors[i]^exponents[i]));
    res := insertMod(expr(p), [op(inds)], factors[i]^exponents[i]);
    if res=FAIL then
      conds := UNKNOWN;
      next;
    end_if;
    if not contains(res, 0) then
      return(FALSE);
    end_if;
    if res<>{0} then
      conds:=UNKNOWN;
    end_if;
  end_for;

  if systems<>FAIL and conds=UNKNOWN then
    if nops(systems)=1 then
      return(expr(op(systems, [1,1]))/op(systems, [1,2]));
    end_if;
    /* Chinesischer Restsatz: */
    res := 0;
    for i in systems do
      for j in monomials(op(i, 1)) do
        Mi := modulo/op(i, 2);
        assert(type(Mi)=DOM_INT);
        [g, s, t] := [igcdex(op(i,2), Mi)];
        assert(g=1);
        res := res + expr(j)*t*Mi;
      end_for;
    end_for;
    p := poly(res/modulo, Dom::Rational);
    i := igcd(op(map([coeff(p)], numer)));
    p := mapcoeffs(p, _divide, i);
    res := expr(p);
    return(res);
  end_if;

  return(conds);
end_proc:

/* property::singleRootsZPoly
 * Input:
 *  f: A polynomial in one variable
 * Output:
 *  DOM_POLY that is equivalent to f in the meaning that it has the same roots
 */
property::singleRootsZPoly := proc(f, modulo, ind:DOM_IDENT)
  local p, annihil;
begin
  assert(isprime(modulo));
  p := property::reduceZPolyModPrime(f, modulo, {ind});
  p := poly(p, [op(ind)], IntMod(modulo));
  if degree(p)<2 then return(p); end_if;
  annihil := poly(ind^modulo-ind, [op(ind)], IntMod(modulo));
  gcd(p, annihil);
end_proc:

