/* property::normalGroebner()
 */

property::normalGroebner :=
proc(xpr, polys = [], maxVariables = 5)
  local basis, inds, poly_inds, i, result, s, err,
    orig_inp, orig_inds,
    change, protectedvars, subs_ident, subs_I,
    substitutions;
  save MAXEFFORT;

begin
  // make normalGroebner overloadable
  if xpr::dom::normalGroebner <> FAIL then
    return(xpr::dom::normalGroebner(args()))
  end_if;

  /* Skip the call for low MAXEFFORT */
  if MAXEFFORT<10*length(xpr) then
    return( xpr )
  end_if;

 

  /* Substitute protected identifiers in the polynomials */
  protectedvars:= select(freeIndets(polys), X -> protected(X) <> None);
  inds:= [genident() $i=1..nops(protectedvars)];
  if nops(inds) > 0 then
    xpr:= subs(xpr, op(protectedvars, i) = inds[i] $i=1..nops(inds));
    polys:= subs(polys, op(protectedvars, i) = inds[i] $i=1..nops(inds));
    return(subs(property::normalGroebner(xpr, polys, maxVariables),
           inds[i] = op(protectedvars, i) $i=1..nops(inds)))
  end_if;

  /* We allow a single equation (e.g. a=-1) as parameter */
  if type(polys)="_equal" then polys := [op(polys,1)-op(polys,2)]; end_if;

  /* Do some mappings that are not handled by overloading.
   * These are special cases like bool conditions, sets, etc... */
  case expr2text(type(xpr))
    /* We need to map through bool expression, since we have the
     * convention, that illegal substitutions in conditions yield
     * the result FALSE. */
    of "\"_not\"" do
      case traperror( ( s := property::normalGroebner( op(xpr), polys, maxVariables ) ) )
        of 0 do
          return(not s);
        of 1025 do /* Division by zero. Remove this element */
          return(TRUE);
        otherwise
          lasterror();
        end_case;
    of "\"_and\"" do
    of "\"_union\"" do
    of "\"_minus\"" do
    of "\"_intersect\"" do
      MAXEFFORT:= MAXEFFORT/nops(xpr);
      s:= [op(xpr)];
      i:= map(s, property::normalGroebner, polys, maxVariables);
      if i = s then
        return(xpr)
      else
        return(eval(op(xpr, 0))(op(i)))
      end_if
    of "\"solve\"" do
      // simplify the equation, inequality, or system to solve
      i:= op(xpr, 1);
      /* TODO: tcov=0, write tests? */
      if type(i) = DOM_LIST then
        if nops(i)>1 then
          MAXEFFORT:= MAXEFFORT/nops(i)
        end_if;
        err:= traperror((s:= map(i, property::normalGroebner, polys, maxVariables)))
      else
        err:= traperror((s:= property::normalGroebner(i, polys, maxVariables)))
      end_if;

      case err
        of 0 do
          if length(s) < length(i) then
            // a simplification has occurred: call solve again
            return(eval(subsop(xpr, 1=s)))
          else
            // no simplification: do not evaluate again
            return(xpr)
          end_if
        of 1025 do
          // equation is undefined under the given constraints: no solution
          return({})
        otherwise
          lasterror()
      end_case;
    of "DOM_SET" do
    of "DOM_LIST" do
    of "\"_or\"" do
      change := FALSE;
      result := [];
      if nops(xpr) > 1 then
        MAXEFFORT:= MAXEFFORT/nops(xpr)
      end_if;
      for i in [op(xpr)] do
        case traperror( ( s := property::normalGroebner( i, polys, maxVariables ) ) )
          of 0 do
            if s<>i then change:=TRUE; end_if;
            result := result.[ s ];
            break;
          of 1025 do /* Division by zero. Remove this element */
                     /* In lists, replace it by undefined */
            change := TRUE;
            if type(xpr) = DOM_LIST then
              result:= result.[undefined]
            end_if;
            break;
          otherwise
            lasterror();
          end_case;
      end_for;
      /* Skip reevaluation if nothing changed */
      if not change then return( xpr ); end_if;
      if type(xpr)=DOM_SET then
        result := {op(result)};
      elif type(xpr)=DOM_LIST then
        result := [op(result)];
      else // type = "_or"
        result := _or( op(result) );
      end_if;
      return( result );
  end_case;

 /* Skip the call if infinities occur in an arithmetical expression */
  if testtype(xpr, Type::Arithmetical) and has(xpr, infinity) then
    return(xpr)
  end_if;


  subs_ident := [];
  orig_inp := xpr;

  /* get the basis for all idents in the input */
  orig_inds := inds := property::freeIndets(xpr);
  if inds={} then return(xpr); end_if;
  [basis, substitutions] := property::gbasis(inds);
  /* if the result contains substitutions then substitute the input parameters as well */
  if substitutions<>[] then
    polys := subs(polys, substitutions);
    xpr := subs(xpr, substitutions);
    subs_ident := subs_ident.map(substitutions, equ -> (op(equ, 2) = op(equ, 1)));
  end_if;
  polys := map(polys, X-> numer(X));

  /* if the input has the imaginary I then replace I by an identifier and
   * add the polyonom "I^2+1" to the basis.
   * Note: Only do check this, if we have other basis elements, so
   * we don't use "I^2+1" as the only one basis. */
  if (has(xpr, I) or has(polys, I)) and nops(basis.polys)>0 then
    if has(xpr, #I) or has(polys, #I) then
      i := genident();
    else
      i := #I;
    end_if;
    xpr := subs(xpr, I=i, Unsimplified);
    polys := subs(polys, I=i, Unsimplified).[i^2+1];
    subs_I := [i=I]
  else
    subs_I := []
  end_if;
  
  /* Rationalize the equations and add minimal polynomials to it */
  polys := map(polys, proc(X)
    local v;
  begin
    v := [rationalize(X, FindRelations=["_power"], MinimalPolynomials, Prefix=" normalGroebner")];
    subs_I := subs([op(v[2])], subs_I).subs_I;
    v[1], op(v[3]);
  end_proc);
  
  /* Get the indets. If there were no assumptions (i.e. basis is empty) take the indets
    * from xpr. */
  if nops(basis)>0 then
    inds := {op(op(basis,[1,2]))};
  else
    inds := indets(xpr);
  end_if;

  /* get additional basis from second parameter */
  poly_inds := select(indets(polys, PolyExpr), X->type(X)=DOM_IDENT or not testtype(X, Type::Constant));
  if poly_inds<>{} then
    polys := select(polys, testtype, Type::PolyExpr([op(poly_inds)], Type::Rational));
  else
    polys := [];
  end_if;

  /* we only need to do something, if we have a basis for the groebner-algorithm */
  if nops(basis.polys)>0 then
    /* If we have additional polynomials, add these to the basis and calculate the
     * new groebner basis. */
    if nops(polys)>0 then
      inds := [op(inds union poly_inds)];
      inds := sort(inds);
      /* check if further substitutions are necessary */
      substitutions:= map(select(inds, not testtype, DOM_IDENT), 
                          X -> (X = genident("normalGroebner")));
      subs_ident := subs_ident.map(substitutions, equ -> (op(equ, 2) = op(equ, 1)));
      inds := subs(inds, substitutions);
      if (nops(inds)<=maxVariables or maxVariables=0) then
        inds := [op({op(inds)} union orig_inds)];
        inds := sort(inds);
        basis := groebner::gbasis(map(subs(basis.polys, substitutions), poly, inds), DegInvLexOrder);
      else
        inds := [op({op(inds)} union orig_inds)];
        inds := sort(inds);
        basis := map(subs(basis.polys, substitutions), poly, inds);
      end_if;
    else
      inds := [op({op(inds)} union orig_inds)];
      inds := sort(inds);
      basis := map(subs(basis, substitutions), poly, inds);
      if nops(basis)=0 then
        return(orig_inp);
      end_if;
    end_if;

    // normalf can only handle polynomials with rational coefficients;
    // map to all such subexpressions
    result := misc::maprec(subs(xpr, substitutions),
                        (proc(X)
                          local f;
                        begin
                        f:= eval(op(X, 0));
                        _lazy_or(f = evalAt, _lazy_and(domtype(f) = DOM_FUNC_ENV, f::evalAt <> FAIL),
                          _lazy_and(domtype(f) = DOM_DOMAIN, f::evalAt <> FAIL))
                        end_proc) = id,
                        {Type::PolyExpr(inds, Type::Rational)} =
                        proc(t)
                          local p, g;
                          begin
                            p := poly(t, inds);
                            g := groebner::normalf(p,
                                                   basis, DegInvLexOrder);
                            /* If nothing has changed, return the input value.
                             * This avoids stupid expanding of terms
                             * like (x+1)^2. */
                            if p<>g then
                              expr(g);
                            else
                              t;
                            end_if;
                        end_proc
                        ,
                        NoOperators,
                        PostMap,
                        Unsimplified
                        );

      /* If we have substitutions revert them. */
      if subs_ident<>[] then
        result := subs(result, subs_ident, Unsimplified);
      end_if;
      if subs_I <> [] then
        // we have to re-substitute #I by I
        result:= subs(result, subs_I);
      end_if;
      if length(orig_inp)<length(result)/2-5 then
        return(orig_inp);
      end_if;
      return(eval(result));
  end_if;
  orig_inp;
end_proc:
