
/*
    rec::evaluateAt(r, a)

    with r = rec(f(u(n+k), ..., u(n)), u(n), initial_conditions),
    return u(a) for the unique solution u of r

    This may fail if the recurrence is not linear, or if a is not an integer

*/

rec::evaluateAt:=
proc(r: rec, a)
  local u, uu, unext, n, rr, inits, initpoints, ord, cmax, cmin, dn, p, i;
begin
  if domtype(a) <> DOM_INT then
    return(FAIL)
  end_if;
  
// Write the recurrence  f(u(n+n0), u(n+n0+1), ..., u(n+n1)) = 0
// as u(n+n1) = F(u(n+n0), ... , u(n+n1-1) and do it step by step 
// to obtain the value u(a) from the initial values. 

  u:= op(r, [2, 0]);
  n:= op(r, [2, 1]);

  //------------------------------------------------------------
  // inits = [u(n0) = a0, u(n0 + 1) = a1, ..., u(n0 + c) = a.c]: 
  //------------------------------------------------------------
  inits := sort([op(op(r, 3))], ((eq1, eq2) -> op(eq1, [1,1]) < op(eq2, [1,1])));
  initpoints:= map(inits, op, [1, 1]);
  if initpoints = [] then
     initpoints:= [0];
  end_if:

  cmax:= max(op(initpoints)): // max index of initial values
  cmin:= min(op(initpoints)): // min index of initial values

  //------------------------------------------------------------
  // Analyze the recurrence equation op(r, 1) = 0 and find
  // all entries of the form u(n + k) in the expression.
  // Write the arguments n + k into the set ord:
  //------------------------------------------------------------
  ord:= {}:
  misc::maprec(op(r, 1), {"function"} = proc(f) begin
                  if op(f, 0) = u and has([op(f)], n) then
                     ord:= ord union {op(f)};
                  end_if;
                end_proc);
  ord:= map(ord, c -> c-n):
  // Now, ord = [n1, n2, .., n.d] with n1 < n2 < ...,  where 
  // the initial conditions u(n1) = u1, u(n2) = u2, ... are
  // provided by the user.
  if ord = {} then
     error("cannot recognize a recursion");
  end_if:
  dn:= max(ord) - min(ord);   // the order of the recurrence

  //----------------------------------------------------
  // The 'forward' iteration towards higher values of n:
  // If the target index 'a' is smaller than cmax, we
  // still need to do something: if there are gaps
  // in the list of initial conditions, we fill them
  // by using the recursion.
  //----------------------------------------------------
  if a >= cmin then 
     if a < cmin + dn then // we cannot use the recursion
                           // to derive u(a)
        i:= contains(map(inits, op, [1, 1]), a);
        if i>0 then
           return(op(inits[i], 2));
        else
           return(u(a))
        end_if:
    end_if:
    //---------------------------------------------------------------
    // Shift n so that the recurrence relates u(n),u(n+1),..,u(n+dn)
    //---------------------------------------------------------------
    rr:= subs(op(r, 1), n = n - min(ord));
    //---------------------------------------------------------------
    // Compute the 'forward map' uu=u(n+dn)=F(u(n),u(n+1),..,u(n+dn-1))
    // assuming that the recurrence is linear in u(n+dn)
    //---------------------------------------------------------------
    p:= poly(subs(rr, u(n + dn) = `#u`), [`#u`]);
    if p = FAIL or degree(p) <> 1 then 
       return(FAIL);
    end_if:
    uu:= -coeff(p, 0)/coeff(p, 1):
    if has(uu, `#u`) then
       return(FAIL);
    end_if;
    // -----------------
    // Do the iteration:
    // -----------------
    for i from cmin+dn to a do
        unext:= normal(subs(uu, n = i-dn, inits, EvalChanges));
        if i = a then 
           return(unext);
        else
           if i > cmax then
             inits:= inits.[u(i) = unext];
           else
             inits:= select(inits, eq -> op(eq, [1,1]) < i).
                     [u(i) = unext].
                     select(inits, eq -> op(eq, [1,1]) > i);
           end_if:
        end_if;
        if i >= op(inits[1], [1, 1]) + dn then
           // inits[1] does not affect u(i+1) 
           // and is not needed anymore.
           delete inits[1]:
        end_if:
    end_for;
  end_if;

  //-----------------------------------------------------
  // The 'backwards' iteration towards lower values of n:
  //-----------------------------------------------------
  if a < cmin then 
    //-----------------------------------------------------------------
    // Shift n so that the recurrence relates u(n-dn),u(n-dn+1),..,u(n)
    //-----------------------------------------------------------------
    rr:= subs(op(r, 1), n = n - max(ord));
    //---------------------------------------------------------------
    // Compute the 'backward map' uu=u(n-dn)=F(u(n-dn+1),...,u(n-1),u(n))
    // assuming that the recurrence is linear in u(n-dn)
    //---------------------------------------------------------------
    p:= poly(subs(rr, u(n - dn) = `#u`), [`#u`]);
    if p = FAIL or degree(p) <> 1 then 
       return(FAIL);
    end_if:
    uu:= -coeff(p, 0)/coeff(p, 1):
    if has(uu, `#u`) then
       return(FAIL);
    end_if;
    //------------------
    // Do the iteration:
    //------------------
    for i from cmin-1 downto a do 
        unext:= normal(subs(uu, n = i+dn, inits, EvalChanges)):
        if i = a then 
           return(unext);
        end_if;
        inits:= [u(i) = unext].inits;
        if i <= op(inits[-1], [1, 1]) - dn then
           // inits[-1] does not affect u(i-1)
           // and is not needed anymore.
           delete inits[-1]:
        end_if:
    end_for;
  end_if;

  FAIL
end_proc:
