/* solves the recurrence u(n+1)=r*u(n) where r is rational in n
   returns a set with the solution or piecewise
   returns {} if no solution is found */
rec::hypergeom :=
proc(r,u,n)
  local p,q,d,C,rat;
begin
  C := genident("C");
  r := normal(r);

  if not has(r, n) then
    d := denom(r);
    if testtype(d, Type::Constant) then
      return({C * r^n});
    else
      return(piecewise([d <> 0, {C * r^n}]))
    end_if;
  elif testtype(r,Type::RatExpr(n)) then
    userinfo(2,"hypergeometric recurrence with ratio",r);

    // compute largest possible rational part
    [p, q, rat] := rec::rational_part(numer(r), denom(r), n);
    d := lcoeff(p, [n]) * lcoeff(q, [n]);

    if has(p, n) then
      if type(p) = "_mult" then
        p := map(p, rec::unratio, n)
      else
        p := rec::unratio(p, n)
      end_if;
    else
      p := p^n
    end_if;
    if has(q, n) then
      if type(q) = "_mult" then
        q := map(q, rec::unratio, n)
      else
        q := rec::unratio(q, n)
      end_if;
    else
      q := q^n
    end_if;
    if p <> FAIL and q <> FAIL then
      if testtype(d, Type::Constant) then
        return({C * rat * p / q})
      else
        return(piecewise([d <> 0, {C * rat * p / q}]))
      end_if
    end_if
  end_if;
  {}
end_proc:


/* Extract factors g(n) from numer for which there is an integer-shifted factor
   g(n-i) in denom and remove them from numer and denom.
   numer, denom are coprime polynomials in the variable n
   returns a list [rnumer, rdenom, p],
   where rnumer, rdenom are numer, denom with the found factors removed
   and in factored form,
   and p is a rational function such that rec::hypergeom(numer/denom) =
   p * rec::hypergeom(rnumer/rdenom) and the second factor contains only
   gamma functions, no rational expressions in n
*/

rec::rational_part := proc(numer, denom, n)
  local i,j,p,num,den,g,found,shift,s,tmp;
begin
  if not (has(numer, n) and has(denom, n)) then
    return([numer, denom, 1]);
  end_if;

  /* g extracts the degree and the ratio of second highest coeff by degree
     times leading coeff of f, the resulting list with three elements is
     repeated m times if the degree of f w.r.t. n is nonzero */
  g := proc(f, m) local k; begin    
    k := degree(f, n);
    if k = 0 then
      [f^m, 0, 0]
    else
      [f, k, normal(coeff(f, n, k-1) / k / coeff(f, n, k))] $ m
    end_if
  end_proc;

  /* num := factors of numer, den := factors of denom */
  tmp := factor(numer);
  // Anpassung an Aenderung von Factored::_index (Walter, 29.12.04)
  tmp:= coerce(tmp, DOM_LIST):
  num := [[tmp[1], 0, 0], g(tmp[2*i], tmp[2*i+1]) $ i = 1..(nops(tmp)-1)/2];
  tmp := factor(denom);
  // Anpassung an Aenderung von Factored::_index (Walter, 29.12.04)
  tmp:= coerce(tmp, DOM_LIST):
  den := [[tmp[1], 0, 0], g(tmp[2*i], tmp[2*i+1]) $ i = 1..(nops(tmp)-1)/2];

  p := 1;
  i := 1;
  while i <= nops(num) do // process all factors of numer
    if num[i][2] > 0 then // degree > 0

      // compare num[i] to all elements of den
      found := 0;
      for j from 1 to nops(den) do
        if num[i][2] = den[j][2] and
           domtype((shift := normal(den[j][3] - num[i][3]))) = DOM_INT then

          // den[j] is an integral shift of num[i] if the degree is 1,
          // otherwise check
          if (num[i][2] > 1) and not
             iszero(expand(subs(num[i][1], n = n + shift) - den[j][1])) then
            // try next factor
            next
          end_if;

          found := num[i][1]; // the factor from the numerator
          break;
        end_if
      end_for;

      if found <> 0 then
	// remove the factors found from the lists
        delete num[i], den[j];

        // we know that shift<>0 since numer and denom are coprime
        if shift > 0 then
          // p := p * 1/found(n)*found(n+1)* ... * found(n+shift-1)
          p := p / _mult(expand(subs(found, n = n + s)) $ s = 0..shift-1);
        else // shift < 0
          // p := p * found(n-1)*found(n-2)* ... * found(n+shift)
          p := _mult(p, expand(subs(found, n = n - s)) $ s = 1..-shift);
        end_if
      else
        i := i + 1
      end_if
    else
      i := i + 1
    end_if
  end_while;

  [_mult(op(map(num, op, 1))), _mult(op(map(den, op, 1))), p]
end_proc:


/* e is a polynomial in n
  output f such that f(n)=C*e(n-1)*e(n-2)*...*e(n-D) for some constants
  C,D
  answer may be wrong when e contains parameters */
rec::unratio :=
proc(e,n)
  local a,b,c,d;
begin
  if testtype(e,Type::PolyExpr(n)) then
    case (d:=degree(e,n))
      of 0 do
        return(e^n)
      of 1 do
        a:=coeff(e,n,1); b:=coeff(e,n,0);
        b:=b/a; a:=a^n;
        return(a*gamma(n+b))
      of 2 do
        /* factor e into linear factors with algebraic coeffs */
        a := coeff(e, n, 2);
        b := coeff(e, n, 1);
        c := coeff(e, n, 0);
        return(a^n * rec::unratio(n + (b + sqrt(b^2-4*a*c))/2*a, n)
               * rec::unratio(n + (b - sqrt(b^2-4*a*c))/2*a, n));
      otherwise /* try to recognize a power */
        if type(e) = "_power" then
          return(rec::unratio(e[1], n)^e[2]);
        end_if;
    end_case;
  end_if;
  FAIL
end_proc:

// end of file

