/* This is the main file for solving recurrences.
  rec::solve(r,u(n),inits) solves the recurrence r (or r=0) for u(n)
  with initial conditions inits.
  Output: a set of solutions or solve(rec(...)) if no solution is found

Examples: 
>> solve(rec(r(n+2)-2*r(n+1)+r(n)=2,r(n),{r(0)=-1,r(1)=m}));

                                      2     
                              {m n + n  - 1}

>> solve(rec(r(n+2)-3*r(n+1)+r(n)=2,r(n),{r(0)=-1,r(1)=m}));

 solve(rec(r(n) - 3 r(n + 1) + r(n + 2) = 2, r(n), {r(0) = -1, r(1) = m}))

>> solve(rec(n*u(n+2)-5*u(n+1)-(n+1)*u(n),u(n)));

                         {    /        2      \ }
                         {    | n   3 n     3 | }
                         { a3 | - - ---- + n  | }
                         {    \ 2    2        / }

>> solve(rec(u(n+1)=2*u(n)*(n+1)/n,u(n)));

                                       n - 1
                              {n u(1) 2     }

>> solve(rec(2*u(n)+2*(n+1)*u(n+1)+(n+1)*(n+2)*u(n+2),u(n),{u(0)=0,u(1)=1}));
                {                    n                  n }
                { (- 1/2 I) (- 1 + I)  + 1/2 I (- 1 - I)  }
                { --------------------------------------- }
                {                 fact(n)                 }
*/

/* this function does no argument checking
   expects that u(..) occurs in r and that all terms u(f(n))
   are of the form u(n+i) for integral i
   puts the recurrence in normal form u(n+ord) ... u(n),
   and calls the main solving function rec::Solve */

rec::solve :=
proc(R=FAIL,UOFN=FAIL,INITS={})
  local r,u,n,sol,ord,m,uofn,inits;
begin
  r:= R;
  uofn:= UOFN;
  inits:= INITS;

  case domtype(r)
    of rec do /* interactive call, argument checking done in rec::new */
      uofn:=extop(r,2);
      inits:=(if extnops(r)>=3 then extop(r,3) else {} end_if);
      r:=extop(r,1);

      // check whether second argument to solve, if present, is identical
      // with uofn
      if args(0) > 1 and UOFN <> uofn then
        warning("additional arguments to solve are ignored");
      end_if;

      /* break through */
    of DOM_EXPR do /* library call, no argument checking */
      u:=op(uofn,0); n:=op(uofn,1);

      /* find order of the equation */
      ord := select(indets(r, RatExpr), proc(f) begin
          op(f, 0) = u and has([op(f)], n)
        end_proc);
      ord := map(ord, proc(f) begin op(f) - n end_proc);

      /* now put the recurrence in normal form, i.e. wrt u(n+ord) ... u(n) */
      m := min(op(ord));
      if m = 0 then
        ord := max(op(ord))
      else
        r := subs(r, n = n-m);
        userinfo(2, "Replacing by recurrence ".expr2text(r));
        ord := max(op(ord)) - m
      end_if;
      userinfo(5, "Order is ".expr2text(ord));

      sol := rec::Solve(r, u, n, ord, inits);

      if sol = FAIL then
        // no solution found, then return symbolic solve call
        if domtype((r := R)) <> rec then
          r := rec(args())
        end_if;
        hold(solve)(r)
      else
        sol
      end_if;
      break
    otherwise
      error("invalid first argument")
  end_case
end_proc:


/* library call: r must be an expression
                 involving u(n), u(n+1), ..., u(n+ord)
                 and we want so solve r=0 for u
   inits is a set of initial conditions of the form u(foo)=bar
   Output: a set of solutions or FAIL */
rec::Solve :=
proc(r,u,n,ord,inits)
  local sol,i,l,s,newr,news;
begin
  // rewrite factorials as gammas
  r:= expand(rewrite(r, gamma));  
  r := expand(numer(r));
  /* this is not quite correct, what about the roots of denom(r)? */
  sol:={};
  userinfo(2,"recurrence is ".expr2text(r));
  // test whether l is linear in u(n) ... u(n+ord)
  if (l:=Type::Linear(r,[u(n+i)$i=0..ord]))<>FALSE then
    userinfo(1,"recurrence is linear");
    if map({op(l)},testtype,Type::PolyExpr(n))={TRUE} then
      userinfo(1,"   with polynomial coefficients");
      userinfo(5, "Coefficient list is ".expr2text(l));
      if ord=1 and l[3]=0 then
        /* homogeneous of order 1 l[2]*u(n+1)+l[1]*u(n)=0 */
        userinfo(1, "Calling hypergeometric solver");
        sol := rec::hypergeom(-l[1]/l[2],u,n);
      else
        sol:={};
        if not has(l,n) then
          sol := rec::constant(l,n,ord)
        end_if;
        if sol={} then
          sol:=rec::polysol(r,u,n,ord);
          userinfo(10, "Calling the poly solver gives ".expr2text(sol));
        end_if;
        if (sol={} or sol={0}) and l[ord+2]=0 then /* tries the recurrence for u(n)/n! */
          // to do: do this also for piecewise
          s:=1;
          for i from 1 to ord do
            s:=s*(n+i);
            l[i+1]:=normal(l[i+1]/s);
          end_for;
          if not has(l,n) then
            userinfo(10, "Calling constant solver for ".expr2text(l));
            sol := rec::constant(l,n,ord);
            if sol<>{} then
              sol:=map(sol,_mult,1/fact(n))
            end_if
          end_if
        elif not has(sol,n) then
          /* if we have found a solution of the inhomogeneous
             equation that is independent of n and has no free
             parameters (so it is one particular solution of the
             inhomogeneous equation),
             then retry homogeneous equation */
          if type(sol) = DOM_SET and nops(sol) = 1 and op(sol) <> 0
            // and sol has no free parameters
            and nops((indets(sol) minus Type::ConstantIdents)
                     minus indets(r)) = 0 then
            // retry homogeneous equation
            newr := subs(r, u(n+i) = u(n+i) + op(sol) $ i = 0..ord, EvalChanges):
            news := rec::Solve(newr, u, n, ord, {}):
            sol := sol + news:
// The following probably does not work any more, but it should not
// happen anyway, since none of the calls up to here produce piecewise
//          elif type(sol)=piecewise then
//            sol:= piecewise::map
//            (sol,
//             proc(t)
//               local result;
//             begin
//               userinfo(10, "Handling set ".expr2text(t));
//               if type(t) = DOM_SET and nops(t) = 1 and op(t) <> 0 then
//                 result:=rec::Solve(eval(subs(r,u=u+op(t,1))),u,n,ord,
//                                       {});
//                 userinfo(10, "Solving ".expr2text(eval(subs(r,u=u+op(t,1)))).
//                             " gives ".expr2text(result));
//                 eval(subs(result,u=u-op(t,1)));
//               map(%,_plus, op(t,1))
//               elif t=C_ then
//                 t
//               else
//                 {}
//               end_if
//             end_proc);
          end_if
        end_if
      end_if
    end_if;
  end_if;
  if sol={} or sol=FAIL then
    return(FAIL)
  end_if;
  if args(0) >= 5 and nops(inits) > 0 then
    userinfo(1,"trying to fulfill initial conditions");
    case domtype(sol)
    of DOM_SET do
      sol := rec::Solve_inits(sol,r,u,n,inits);
      break;
    of piecewise do
      sol := piecewise::extmap(sol, rec::Solve_inits, r, u, n, inits);
      break;
    otherwise
      return(FAIL);
    end_case;
  end_if;
  if sol={} then
    return(FAIL)
  end_if;
  sol
end_proc:


/* try to fulfill initial conditions
   sol is the set with the general solution of the recurrence r
   in u(n)
   inits is a set of initial conditions of the form u(foo)=bar
   returns a set with the solution */
rec::Solve_inits :=
proc(sol,r,u,n,inits)
  local sys,eq,unk,s;
begin
  sol := op(sol);
  if nops(sol) = 0 then
    return({})
  end_if; 

  /* substitute the solution into the initial conditions */
  sys := {};
  for eq in inits do
    /* The traperror is somewhat crude, and the result may be wrong in
       the parametric case: e.g., if sol = gamma(n+a)/gamma(a), then
       for negative integer values of the parameter a we should not
       substitute a nonnegative integer <= -a for n.
       Alternatives: discont: output is too horrible
                     limit: is too slow
    */
    if traperror((s := subs(sol, n = op(eq, [1,1]), EvalChanges))) <> 0 then
//      if contains({infinity, -infinity, undefined, FAIL},
//                  (s := limit(sol, n = op(eq, [1,1])))) then
      /* could not evaluate solution at one of the initial conditions,
         due to a singularity */
      userinfo(2, "singularity encountered");
      return({});
    end_if;
    sys := sys union {normal(s) - op(eq, 2)}
  end_for;

  /* recover set of free parameters */
  unk := ((indets(sys) minus Type::ConstantIdents)
          minus indets(r)) minus indets(inits);

  if unk = {} then
    if sys = {0} or sys = {} then
      {sol}
    else // unsatisfiable
      {}
    end_if
  else
    /* solve for the free parameters */
    sys := linsolve(sys, unk);
    if sys = FAIL then // unsatisfiable
      {}
    /* linsolve makes implicit assumptions about pivots being nonzero,
       so the following would be
       a better answer for parametric problems. However, the conditions
       may then include something like gamma(a)<>0, which can currently
       not be simplified, and therefore the following is commented out. 
       Beware: linsolve with option ShowAssumptions does something
       different than without this option! */
//    elif nops(sys[2].sys[3]) > 0 then
//      // linsolve has made implicit assumptions
//      piecewise([_and(op(sys[2]), op(sys[3])), eval(subs(sol, sys[1]))])
    else
      // no assumptions were made by linsolve
      {subs(sol, sys, EvalChanges)}
    end_if
  end_if
end_proc:

