/*--
        Chi::series -- the function attribut "series" for Chi
        
        series expansion of Chi is based on the series expansion of Ci:

        Chi(z) = Ci(I*z) + ln(z) - ln(I*z): 
--*/

// Known problems:   series(Chi(1/x),x, Left) fails

Chi::series:= proc(e, x, n, dir, opt) 
  local s, l;
begin 
  // We use Chi(e) = Ci(e*I) + ln(e) - ln(I*e).

  // Take care of infinite expansion points.
  // Here the expansion of Ci(e*I) is a gseries, which
  // cannot be married properly with the Puiseux series
  // ln(e) - ln(I*e). So simplify 
  // ln(e) - ln(I*e) = some multiple of I*PI/2 explicitly:
  if dir <> Undirected then // directed expansion
     l:=limit(e,x,dir);
     if l = infinity then 
        s := Series::series(Ci(e*I) -I*PI/2, x, n, dir, opt);
     elif l = I*infinity then 
        s := Series::series(Ci(e*I) -I*PI/2, x, n, dir, opt);
     elif l = -infinity then 
        s := Series::series(Ci(e*I) + 3*I*PI/2, x, n, dir, opt);
     elif l = -I*infinity then 
        s := Series::series(Ci(e*I) - I*PI/2, x, n, dir, opt);
     else
        s:= FAIL   
     end_if:
     if s <> FAIL then 
       return(s)
     end_if;  
  end_if;

  // For finite expansion points Ci(e*I) yields a Puiseux series
  // that can be combined with ln(e) - ln(I*e):
  s := Series::series(Ci(e*I) + ln(e) - ln(I*e), x, n, dir, opt);
  if type(s) = DOM_FAIL then
     return(FAIL)
  else
     // Taylor expansions around x = x0 should look like
     // Chi(x0) + O(x - x0). At this stage, we have
     // (Ci(I*x0) + ln(x0) - ln(I*x0)) + O(x - x0).
     // Rewrite such expressions to the target Chi(x0).
     // Further, we want to rewrite
     //     cos(I*x0) = cosh(x0),
     //     sin(I*x0) = I*sinh(x0).
     if has(s, Ci) then
        return(subs(s, [hold(Ci) = (z -> Chi(z/I) - ln(z/I) + ln(z)),
                             hold(cos) = proc(z) begin
                                          if type(z) = "_mult" and
                                             has([op(z)], I) then
                                                cosh(z/I)
                                          else  cos(z)
                                          end_if;
                                         end_proc,
                             hold(sin) = proc(z) begin
                                          if type(z) = "_mult" and
                                             has([op(z)], I) then
                                                I*sinh(z/I)
                                          else  sin(z)
                                          end_if;
                                         end_proc
                            ],EvalChanges));
     else
        return(s);
     end_if;
  end_if;
end_proc:
// end of file 
