/* this file contains several functions to deal with formal power series,
  differential equations and recurrences:

  Series::expr2coeff(f,x) - tries to find a closed-form formula for the n-th
	coefficient in the Taylor expansion of f in x=0

  Series::expr2diffeq(f,x,y) - tries to find a linear differential equation 
	satisfied by the expression f in x; y is the name of the unknown 
	function in the ODE.

  Series::diffeq2coeff(eq,yofx,n,f) - finds the coefficient of x^n in the 
	Taylor expansion at the origin of y(x) solution of eq=0

  Series::diffeq2rec(s,y(x),u,n) - transforms the ODE s=0 for y(x) into a
	recurrence for the Taylor coefficients u(n)

  Example:

>> Series::expr2coeff(exp(-x)*sin(x),x,n);

                             n 1/2    / 3 n PI \
                           (2 )    sin| ------ |
                                      \   4    /
                           ---------------------
                                  fact(n)

*/

Series::maxord:=2:

Series::expr2coeff :=
proc(f,x,n)
  local eq,y;
begin
  y := genident("y");
  eq:= Series::expr2diffeq(f,x,y);
  if eq=FAIL then
    eq
  else
    Series::diffeq2coeff(eq,y(x),n,f)
  end_if
end_proc:

Series::expr2diffeq :=
proc(f,x,y)
  local i,T,s,j,A,n,ind,k;
begin
   /* tries to find a linear relation with polynomial coefficients between
       f, f', f'', ... f^{maxord} */
   T[0]:=Series::split(f,x);
   s:={T[0][j][2]$j=1..nops(T[0])};
   for i from 1 to Series::maxord do
      // looks for an equation of order i 
      f:=diff(f,x);
      T[i]:=Series::split(f,x);
      s:=s union {T[i][j][2]$j=1..nops(T[i])};
      if nops(s)<=i then break end_if
   end_for;
   if nops(s)>i or i>Series::maxord then return(FAIL) end_if;
   // now constructs the matrix 
   n:=i; for j from 1 to nops(s) do ind[op(s,j)]:=j end_for;
   A:=Dom::/*Dense*/Matrix()(n+1,n+1);
   for i from 0 to n do
      for j from 1 to nops(T[i]) do
         k:=ind[T[i][j][2]];
         A[k,i+1]:=A[k,i+1]+T[i][j][1];
      end_for;
      A[n+1,i+1]:=y.i
   end_for;
   A:=op(A::dom::gaussElim(A, DiagonalStrategy), 1);
   // now looks for the equation in the matrix 
   s:={y.i $ i=0..n};
   for i from 1 to n+1 do
      if has(A[i,i],s) then break end_if
   end_for;
   subs(A[i,i],y.i=diff(y(x),x$i)$ i=0..n);
end_proc:

Series::diffeq2coeff :=
proc(eq, yofx, n: DOM_IDENT, f)
  local r,u,ord,s,i;
begin
  u := genident("u");
  r:=Series::diffeq2rec(expand(eq),yofx,u,n);
  ord:=select(indets(r,PolyExpr),proc()
                                 begin
                                   type(args(1))="function"
                                 end_proc );
  ord:=select(ord,proc() begin op(args(1),0)=u end_proc );
  ord:=select(map(ord, X -> op(X, 1)- n),testtype,DOM_INT);
  if nops(ord) = 0 then
    return(FAIL)
  end_if;
  ord:=max(op(ord));
  s:=series(f,op(yofx),ord);
  s:=solve(rec(r,u(n),{u(i)=coeff(s,i)$i=0..ord-1}));
  if type(s)<>DOM_SET or nops(s)<>1 then return(FAIL) end_if;
  s:=op(s,1);
  save n;
  assume(n in Z_);
  if has(s,I) then
    // s:=expr(rectform(s,{n}))  -> rectform now uses assume
    s:=expr(rectform( s ))
  end_if;
  expand(s)
end_proc:

Series::diffeq2rec:=
proc(s,yofx,u,n)
begin
  if type(s)="_plus" then
    map(s,Series::diffeq2rec2,yofx,u,n)
  else
    Series::diffeq2rec2(s,yofx,u,n)
  end_if
end_proc:

Series::diffeq2rec2:=
proc(f,yofx,u,n)
  local p,c,i,j,y,x;
begin
  y:=op(yofx,0); x:=op(yofx,1);
  if type(f)="_mult" then
    p:=select(f,not has,y); f:=f/p; i:=degree(p,[x]); c:=p/x^i
  elif has(f,y) then
    c:=1; i:=0
  else
    return(0) // polynomial in x
  end_if;
  if f=yofx then
    j:=0
  elif type(f)="diff" then
    j:=nops(f)-1
  elif type(f)="function" and op(f)=x then
    j:=Series::numberOfD(op(f,0),y)
  else
    Series::error("not expected:".expr2text(f))
  end_if;
  // c*x^i*diff(y(x),x$j)
  _mult(c,n-i+p$ p=1..j,u(n+j-i))
end_proc:

Series::numberOfD := 
proc(a)
begin
  if type(a)="D" then
    1+Series::numberOfD(op(a))
  else
    0
  end_if
end_proc :

/* returns a list l such that f=l[1][1]*l[1][2]+...+l[n][1]*l[n][2]
   and l[i][1] is a polynomial in x */
Series::split :=
proc(f,x)
  local i;
begin
  if type(f)="_plus" then
    [Series::split2(op(f,i),x)$i=1..nops(f)]
  else
    [Series::split2(f,x)]
  end_if
end_proc:

Series::split2 :=
proc(f,x)
  local g;
begin
  if type(f)="_mult" then
    g:=select(f,testtype,Type::PolyExpr(x)); [g,f/g]
  elif testtype(f,Type::PolyExpr(x)) then
    [f,1]
  else
    [1,f]
  end_if
end_proc:
