/*
 * series slot of abs - computes series(abs(f), x, n, dir)
 *
 * If dir <> Left, Right, Real ==> simply returns sign(1/f)*series(f).
 * Otherwise expands f, writes it as f0 * x^d * (1 + f1), simplifies
 * sign(1/f0) * sign(x)^(-d) by taking into account the direction,
 * and returns sign(1/f0) * sign(x)^(-d) * f0 * x^d * series(abs(1 + f1)).
 *
 * The code is completely analogous to sign::series.
 */

abs::series := proc(f, x, n, dir)
  local t, d, f0, f1, s, tr, ti, prop;
begin 

  // handle special case abs(x) separately for efficiency reasons
  if f = x then
    case dir
    of Right do // x + O(x^(n+1))
      return(Series::Puiseux::new(x, x, n, dir));
    of Left do // -x + O(x^(n+1))
      return(Series::Puiseux::new(-x, x, n, dir));
    otherwise // x/sign(x) + O(x^(n+1))
      return(Series::Puiseux::scalmult(Series::Puiseux::new(x, x, n, dir),
                                       sign(1/x), 0));
    end_case
  end_if;

  if dir <> Undirected then // directional expansion

    // assign the properties implied by the direction to the series variable
    prop := getprop(x);
    case dir
    of Left do
      assume(x < 0);
      break
    of Right do
      assume(x > 0);
      break
    of Real do
      assume(x, Type::Real);
      break
    end_case;
    if prop <> x then
      assume(x, prop, _and);
    end_if;

    // recursively expand the argument
    t := Series::series(f, x, n, dir);

    if domtype(t) = Series::Puiseux and (d := ldegree(t)) <> FAIL then

      // write t as f0 * x^d * (1 + f1), then abs(f) =
      // sign(1/f0) * sign(x)^(-d) * f0 * x^d * abs(1 + f1)
      f0 := lcoeff(t);
      f1 := Series::Puiseux::scalmult(lmonomial(t, Rem)[2], 1/f0, -d);

      s := f0 * sign(1/f0) * sign(x)^(-d);

      // return s * x^d * sqrt((1 + Re(f1)^2) + Im(f1)^2)

      // some shortcuts
      if normal(f - f0*x^d) = 0 then // f = f0*x^d
        // attach previous property to series variable
        if prop = x then
          delete x
        else
          assume(x, prop)
        end_if;
        return(Series::Puiseux::scalmult(Series::Puiseux::one(x, n, dir), s, d));
      elif ldegree(f1) = FAIL then // f = f0 * x^d * (1 + O(..))
        // attach previous property to series variable
        if prop = x then
          delete x
        else
          assume(x, prop)
        end_if;
        return(Series::Puiseux::scalmult(Series::Puiseux::one(x, n, dir) + f1, s, d));
      end_if;
    
      // split f1 into real and imaginary part tr + ti
      tr := map(f1, Re);
      ti := f1 - tr;

      // attach previous property to series variable
      if prop = x then
        delete x
      else
        assume(x, prop)
      end_if;

      if ldegree(ti) = FAIL then // ti = O(..)
        return(Series::Puiseux::scalmult(Series::Puiseux::one(x, n, dir) + f1 + ti, s, d))
      else
        // uses Puiseux::_power
        return(Series::Puiseux::scalmult(((Series::Puiseux::one(x, n, dir) + tr)^2 - ti^2)^(1/2), s, d))
      end_if

    end_if;

    // attach previous property to series variable
    if prop = x then
      delete x
    else
      assume(x, prop)
    end_if;

  end_if;

  // default: move sign(1/f) into the 0th coefficient
  return( Series::Puiseux::scalmult(Series::series(f, x, n, dir),
                                    sign(1/f), 0) );

end_proc:
