/*
 * series slot of dirac - computes series(dirac(f, n), x, ord, dir)
 *
 * Computes l := limit(f, x = 0, dir), or l := f(0) if dir = Undirected
 * returns symbolic "series" call if l cannot be determined
 *
 * If l is not real (so dirac(l,n) = undefined) ==> returns error
 *
 * otherwise, if l is real:
 * If l = 0 (expansion around 0) ==> dirac goes into the constant coefficient
 * If l <> 0 is real (so dirac(l,n) = 0) and dir <> Undirected ==> returns 0
 * Otherwise, if l <> 0 is real and dir = Undirected ==> symbolic "series" call
 */


dirac::series := proc(f, n, x, ord, dir, opt)
  local l, d;
begin

  if dir = Undirected then
    // undirected expansion, compute f(0)
    if traperror((l := subs(f, x = 0, EvalChanges))) <> 0 then
      userinfo(1, "could not compute limit of first argument of dirac");
      return(FAIL);
    end_if
  else
    // directional expansion, compute limit(f, x = 0)
    l := limit(f, x = 0, dir);
    if type(l) = "limit" or l = FAIL then
      userinfo(1, "could not compute limit of first argument of dirac");
      return(FAIL);
    end_if;
  end_if;
 
  d := dirac(l, n);
  if l <> 0 and d = undefined then
    Series::error("dirac is not defined for non-real expansion points");
  elif l = 0 then // expansion around zero
    // dirac(..) -> constant coefficient
    return(Series::Puiseux::const(dirac(f, n), x, ord, dir));
  elif d = 0 and dir <> Undirected then
    // directional expansion around real l <> 0
    return(Series::Puiseux::zero(x, ord, dir)) // O(x^ord)
  else
    return(FAIL)
  end_if

end_proc:
