/*++
        gseries  --  a domain for generalized series expansions 

        Call:
        gseries(ex, x [=a] [,ord] [,direction])

        Parameters:
        ex       : expression in x
        x        : identifier
        a        : (optional) real arithmetical expression (default: 0)
        ord      : (optional) positive integer
        direction: (optional) option 'Left', or 'Right' (default: Right)

        Synopsis:
        Compute the asymptotic series of ex of order ord with respect to x 
        at the point a. If a is missing then x is assumed to go to 0+.  
        If ord is missing then the value of ORDER is used (6 by default).
        If an expansion could not be computed, then FAIL is returned.

        A generalized series expansion is of the domain 'Series::gseries', 
        and has 4 operands:

        1. a list containing lists of the factors for each product in ex
        2. the bigO term (possibly 0, if the expansion is exact)
        3. the series variable x
        4. the series point a

        Example: 3*exp(1/x) + y*ln(x) + O(x^4) in x = infinity

                -> [[3,exp(1/x)],[y,ln(x)]], x^4, x, infinity


        The algorithm based on the dissertation of Dominik Gruntz:
        "On Computing Limits in a Symbolic Manipulation System", 
        submitted to the SWISS FEDERAL INSTITUTE OF TECHNOLOGY ZUERICH
        in 1995.

        This domain is used by 'series' and 'asympt'.
++*/

Series::gseries:= newDomain("Series::gseries"):
Series::gseries::create_dom:=hold(Series::gseries):
Series::gseries::info:="domain for generalized series expansions":
Series::gseries::interface:= {}:

alias( gseries=Series::gseries ):
alias( Puiseux=Series::Puiseux ):

/*--
    new  --  the main procedure for computing the expansion
--*/
gseries::new :=
proc(ex, lx, ord=ORDER)
  local a, asy, XX, X, x, direction, nex;
begin
  x:= genident("X");
  assume( x > 1 );

  if args(0) < 2 or args(0) > 4 then
    error("wrong no of args")
  elif type(lx) = "_equal" then
    a:= op(lx,2);
    lx := op(lx, 1);
    if domtype(lx) <> DOM_IDENT then
      error("invalid series variable")
    elif not testtype(a,Type::Constant) and
      not testtype(a,Type::Arithmetical) then
      error("expansion point: not an arithmetical expression")
    elif not testtype(a,Type::Numeric) and a <> infinity and a <> -infinity then
      case is(a,Type::Real)
        of TRUE    do
          break
        of FALSE   do
          error("expansion point must be a real arithmetical expression")
        of UNKNOWN do
          error("expansion point cannot be checked on being real")
      end_case
    end_if
  elif domtype(lx) <> DOM_IDENT then
    error("invalid series variable")
  else
    a:= 0 // lx -> +0
  end_if;

  case args(0)
    of 2 do
      direction:= hold(Right);
      break
    of 3 do
      if not testtype(ord,Type::PosInt) then
        direction:= ord;
        ord:= ORDER;
        if not contains({hold(Right),hold(Left)},direction) then
          error("unknown or invalid direction")
        elif a = infinity and direction = hold(Right) or
          a = -infinity and direction = hold(Left)
          then
          error("inconsistent direction")
        end_if
      else
        direction:= hold(Right);
      end_if;
      break
    of 4 do
      if not testtype(ord,Type::PosInt) then
        error("3rd argument, the order of the series expansion, must be a positive integer")
      else
        direction:= args(4);
        if not contains({hold(Right),hold(Left)},direction) then
          error("unknown or invalid direction")
        elif a = infinity and direction = hold(Right) or
          a = -infinity and direction = hold(Left)
          then
          error("inconsistent direction")
        end_if
      end_if
  end_case;

  // variable transformation: Find X(x) such that X(x) -> infinity for x -> a+
  if a = infinity then
    XX:= x;
    X:= lx
  elif a = -infinity then
    XX:= -x;
    X:= -lx
  elif direction = hold(Left) then
    XX:= a-1/x;
    X:= -1/(lx-a)
  else
    XX:= a+1/x;
    X:= 1/(lx-a)
  end_if;

  if not has(ex, lx) then
    // return the exact expansion ex:
    return(new(gseries, [[ex, 1]], 0, lx, a))
  end_if;

  // x -> infinity:
  nex:= subs(ex, lx=XX, EvalChanges);
  nex:= limit::rewriteExpLog(nex, x);

  
  asy:= FAIL;
  if traperror((asy:= gseries::asympt(nex, x, ord ))) <> 0 
     or 
     asy = FAIL then
    return(FAIL)
  elif nops(asy[1]) = 1 and asy[2] = 0 then
    // exact series representation with one term: return ex itself
    // and not any equivalent expression
    return(new(gseries, [[1, ex]], 0, lx, a))
  else
    asy := map(asy, eval@subs, x=X);
    return(new(gseries, op(asy,1), op(asy,2), lx, a))
  end_if
end_proc:


/*-----------------------------------------------------------------*/
/*----------------------- INTERNAL METHODS ------------------------*/
/*-----------------------------------------------------------------*/

/*------------------------------------------------------------------------
    gseries::asympt(ex, x, n, <"extraorder" = extraorder>)

    ex : expression in x
    x  : identifier
    n  : order of series computation
    extraorder: increase the order internally in recursive 
                calls of gseries::asympt, if an internal
                Puiseux expansion with order n does not have
                enough terms

    Compute the asymptotic series of ex, assuming that x goes to
    infinity. This function returns FAIL if an expansion can not
    be computed, and the list [aex,ord] otherwise where aex is
    a list of lists [coefficient, function of x] representing
    an expansion of ex in x and ord the corresponding big-O term.
------------------------------------------------------------------------*/
gseries::asympt :=
proc(ex, x, ord, extraorder = 0: DOM_INT)
  local nex, S, SS, s, i, t, lc, c, w, W;
begin
  W:= genident("W");
  assume(W > 0):
  
  userinfo(1,"(1)  Find mrv set of ",ex);
  
  S := limit::getMRV( ex,x );
  if S = FAIL then
    userinfo(1,"** getMRV fails");
    return( FAIL )
  elif nops(S) = 0 then
    return( [gseries::mkgseries(ex,x),x^ord] )
  end_if;
 

  if contains(S, x) then // Upward Movement
    t:= gseries::asympt(
                        subs(ex, [ln(x) = x, x = exp(x)],EvalChanges),
                        x, ord
                        );
    if t = FAIL then
      return(FAIL)
    else
      // move down
      return(subs(t,[exp(x)=x, x=ln(x)], EvalChanges))
    end_if
  end_if;

  userinfo(1,"(2)  Rewrite the MRV set");

  s := limit::substitutions( S,x,W );
  if s = FAIL then
    return( FAIL )
  end_if;

  w:= subs(W, map(s, x -> op(x,2) = op(x,1)),EvalChanges);
  t:= op(w);
  c:= limit::asymptoticSign(t, x);
  if c = FAIL then
    return(FAIL)
  elif c = 1 then
    s:= subs(s, W = 1/W, EvalChanges);
    w:= 1/w;
    t:= -t;
  end;
    
  nex:= subs(ex, s, EvalChanges);

  if not has(nex, W) then
    // series approximation is an exact one:
    return( [gseries::mkgseries(ex,x),0] )
  end_if;

  userinfo(1,"(3)  Compute the generalized power series in W=0+ of ".
           expr2text(nex));
 

  // Walter 12.3.98: Boost the order if requested, i.e., 
  // call series(nex, W, order + extraorder)

  s:= limit::powerSeries(nex, W, ord + extraorder, t);


  if s = FAIL then
    userinfo(1,"** series computation fails");
    return(FAIL)    
  elif iszero(normal(expr(s) - nex)) then
    // the series approximation is an exact one:
    // (note: if nex depends on x, then they must be expanded
    // recursively (see below)).

    if not has(nex,x) then
      return([[[op(c, 1), w^op(c, 2)] $c in extop(s, 1)], 0])
    end_if:
  end_if;

  userinfo(2,"Result is ",expr(s));


  
  lc:= [];
  S:= extop(s, 1);

  for i from 1 to nops(S) do
    t:= S[i];
    if nops(lc) >= ord then
      // do not have to add more coefficients,
      // just use the exponent as error term
      return([lc, w^t[2]])
    end_if;
    if not has(t[1], x) then
      lc:= lc.[[limit::rewriteBack(t[1]), w^t[2]]]
    else
      if t[1] = ex then
        return(FAIL)
      end_if;
      // try to get all terms we need from this t
      SS:= gseries::asympt(t[1], x, ord-nops(lc));
      if SS = FAIL then
        return( FAIL )
      end_if;
      lc:= lc.gseries::insFactor(SS[1],w^t[2]);
      if extop(SS, 2) <> 0 then
        // not a constant expansion. We have all terms we need
        return([lc, SS[2]*w^t[2]])
      end_if;
    end_if
  end_for;
  
  
   //-----------------------------------------------------------
  // Walter 12.3.08: added the following precision management: 
  // the Puiseux expansion of ex, now encoded in lc, has a 
  // certain order. The order concept for gseries, however, 
  // is different ("count the number of terms)". If there are
  // not enough terms in the Puiseux series, then recompute the
  // series with a sufficiently high extraorder (passed as the
  // 4th argument to gseries::asympt).
  //-----------------------------------------------------------


   if nops(lc) < ord and
     limit::powerSeries::errorterm(s) <> infinity then
     extraorder:= ord + extraorder - nops(lc);
     return(gseries::asympt(ex, x, ord, extraorder))
   end_if:
  //-----------------------------------------------------------
  // end of precision management
  //-----------------------------------------------------------


  
  
  // all terms have been used, the error term of the gseries stems
  // from the error term of the generalized power series
  if (c:= limit::powerSeries::errorterm(s)) = infinity then
    return([limit::rewriteBack(lc), 0])
  else
    return([limit::rewriteBack(lc), w^c])
  end_if;
end_proc:

/*------------------------------------------------------------------------
    gseries::insFactor(L,f)
------------------------------------------------------------------------*/
gseries::insFactor:=
proc(L: DOM_LIST, f) 
  local j;
begin 
  return( subsop(L,j=[L[j][1],L[j][2]*f] $ j=1..nops(L)) )
end_proc:

/*------------------------------------------------------------------------
    gseries::mkgseries(ex,x) 

    Returns a list of pairs [[a1, b1], [a2, b2], ...]
    such that ex = a1*b1 + a2*b2 + ...,
    a1, a2, ... do not contain x,
    and for each bi <> 1, every factor of bi contains x
------------------------------------------------------------------------*/
gseries::mkgseries :=
proc(ex, x)
begin
  ex := expand(ex);
  case type(ex)
    of "_plus" do
      ex := [op(ex)];
      break;
    of "_mult" do
      ex := [ex];
      break;
    otherwise
      if has(ex, x) then
        return([[1, ex]])
      else
        return([[ex, 1]])
      end_if
  end_case;

  map(ex, proc(z)
          begin
            if type(z) = "_mult" then
              z := split(z, has, x);
              [z[2], z[1]]
            elif has(z, x) then
              [1, z]
            else
              [z, 1]
            end_if
          end_proc)
end_proc:

/*------------------------------------------------------------------------
    scalmult(a,c,f) -- multiply coefficients by c and terms by f 
------------------------------------------------------------------------*/
gseries::scalmult := proc(a,c,f) 
    local l,i;
begin
    if extop(a,1) = [] then
      if args(0) < 3 then
         return(a)
      else
         return(extsubsop(a, 2 = extop(a, 2) * f))
      end_if
    end_if;

    if args(0)<3 then f:= 1 end_if;
    l:= extop(a,1);
    return(extsubsop(a, 
      1 = [[l[i][1]*c, normal(l[i][2]*f)] $ i = 1..nops(l)],
      2 = normal(extop(a, 2)*f)
    ))
end_proc:

/*------------------------------------------------------------------------
    dominates( a,b,x )  --  determine the dominant expression

    a, b : arithmetical expressions
    x    : DOM_IDENT

    This method returns 1, if a >> b, 0 if a and b are equivalent,
    and -1 if a << b for x -> 0.

    a >> b means that for x -> 0+, a grows infinity to faster than b or b tends to zero
    faster than a.

    Examples: x >> x^2, exp(x) >> x, a >> 0 for any a <> 0, exp(1/x) >> 1/x
------------------------------------------------------------------------*/
gseries::dominates :=
proc(a: Type::Arithmetical, b: Type::Arithmetical, x: DOM_IDENT):
  Type::Union(DOM_INT, DOM_FAIL)
  local c, d;
begin
  if iszero(a-b) then
    return( 0 )
  elif iszero(b) = TRUE then
    return( 1 )
  elif iszero(a) = TRUE then
    return( -1 )
  end_if;
  
  
  c:= normal(a/b);
  if c = 1 then
    return( 0 )
  elif c = x then
    // shortcut
    return( -1 )
  elif type(c) = "_power" and op(c, 1) = x
    and testtype(op(c, 2), Type::Rational) then
    // shortcut
    return(-sign(op(c, 2)))
  else
    // assume(x in Dom::Interval(0, 10^(-8)));
    d := limit::main(c, x, 0, hold(Right),
                     limit::defaultOptions);
    if d <> FAIL and type(d) <> "limit" then
      if iszero(d) then
        return( -1 )
      elif has(d, infinity) then
        return( 1 )
      else
        return( 0 )
      end_if
    end_if;
    // limit(c) could not be computed, try limit(1/c)
    d := limit::main(1/c, x, 0, hold(Right), limit::defaultOptions);
    if d <> FAIL and type(d) <> "limit" then
      if iszero(d) then
        return( 1 )
      elif domtype(d) = stdlib::Infinity then
        return( -1 )
      else
        return( 0 )
      end_if
    end_if;
    // limit(1/c) didn't work either
    return(FAIL)
  end_if;
  assert(FALSE);
end_proc:

/*------------------------------------------------------------------------
   gseries::series: convert a Series::gseries into a series, maybe with 
   respect to a different variable (different expansion point makes no 
   sense). Just converts the argument into an expression via 
   gseries::expr + O-term and applies "series" to the result. 
   Analogous to Puiseux::series.
------------------------------------------------------------------------*/
gseries::series :=
proc(s)
begin
  if has(extop(s,4), infinity) then
    return(FAIL)
  end_if;

  series(gseries::expr(s) + O(extop(s, 2), extop(s,3) = extop(s,4)),
         args(2..args(0)))
end_proc:

/*------------------------------------------------------------------------
    set_var( s,x,y,a )  --  substitute the variable x by y, where y -> +a.
------------------------------------------------------------------------*/
gseries::set_var :=
proc(s,x,y,a) 
  local l;
begin
  l:= subs(
           [extop(s,1), extop(s,2)],
           x = if domtype(a)=stdlib::Infinity then
                 sign(a)/y
               else
                 y-a
               end
           );
  return(new(gseries, op(l), y, a ))
end_proc:


/*-----------------------------------------------------------------*/
/*------------------------ PUBLIC METHODS -------------------------*/
/*-----------------------------------------------------------------*/

/*------------------------------------------------------------------------
    gseries::create(l,bigO,x=a)

    l    - list containing lists of the factors for each product in ex.
    bigO - the bigO term, possibly zero.
    x    - identifier
    a    - arithmetical expression, not depending on x (default: 0).

    Returns an object of the type 'Series::gseries'. This slot
    should be used with caution, because no argument checking is
    performed. It should only be used to create a gseries without
    calling the algorithm implemented by the domain 'gseries'.
------------------------------------------------------------------------*/
gseries::create:= proc(l,bigO,dirx)
begin
    if l = [[]] then
        // to be sure that O(...) has the empty list [] as its first operand:
        return( FAIL )
    end_if;

    if domtype(dirx) <> DOM_IDENT then
        if type(dirx) <> "_equal" or domtype(op(dirx,1)) <> DOM_IDENT then
            error("3rd argument: expecting an identifier" )
        end_if;
        return( new(gseries,l,bigO,op(dirx)) )
    else
        return( new(gseries,l,bigO,dirx,0) )
    end_if
end_proc:

/*------------------------------------------------------------------------
    map  --  maps onto the coefficients
------------------------------------------------------------------------*/
gseries::map := proc(a, f)
    local l, j, param;
begin
    param:= args(3..args(0));

    l := extop(a,1);
    if l = [] then return(a) end_if;

    ( l[j][1] := f(l[j][1],param) ) $ j=1..nops(l);
    return( extsubsop(a,1 = l) )
end_proc: 

/*------------------------------------------------------------------------
    combine  --  combine monomials and error term; overloads combine
------------------------------------------------------------------------*/
gseries::combine := proc(a)
    local l, j, param, prop, err, x, x0;
begin
    param := args(2..args(0));

    [l, err, x, x0] := [extop(a)];

    // assign appropriate property to x
    prop := getprop(x);
    if x0 = infinity then
      assume(x > 0, _and)
    elif x0 = -infinity then
      assume(x < 0, _and)
    else
      assume(x, Type::Real, _and)
    end_if;

    (l[j][2] := combine(l[j][2], param)) $ j = 1..nops(l);
    err := combine(err, param);

    // restore previous property of x
    if prop <> x then
      assume(x, prop)
    else
      unassume(x)
    end_if;

    return(extsubsop(a, 1 = l, 2 = err))
end_proc: 

/*------------------------------------------------------------------------
    print  --  print an asymptotic series
------------------------------------------------------------------------*/
gseries::print :=
proc(s) 
  local e, l;
begin 
  e:= extop(s,1);
  l:= select(e, x -> (_not@iszero)(x[1]));
  if nops(l) = 0 then
    if Series::printO() then
      return(hold(O)(extop(s,2)))
    else
      return(0);
    end_if;
  else
    l := map(l, generate::sortSums);
    l:= map(l, e -> 
	    if e[1] = 1 then e[2] 
	    elif e[2] = 1 then e[1]
	    else _mult(e[1],e[2]) 
	    end
	   );
    if extop(s,2) = 0 or
       not Series::printO() then
      if nops(l) > 1 then
	return(hold(_plus)(op(l)))
      else
	return(op(l))
      end_if
    else
      return(hold(_plus)(op(l), hold(O)(extop(s,2))))
    end_if
  end_if
end_proc:

gseries::expr2text :=
x-> expr2text(dom)."::create(".expr2text(extop(x, 1..2)).", ".
                             expr2text(extop(x, 3) = extop(x, 4)).")":


/*------------------------------------------------------------------------
    convert  --  convert a Puiseux series to an asymptotic series.
------------------------------------------------------------------------*/
gseries::convert := proc(a)
    local x, p, sgn, t, bigO, bo, v, l, i, ll, myl;
begin
    case domtype(a)
    of gseries do
        return( a )
    of Puiseux do
        x:= Puiseux::indet(a);
        p:= Puiseux::point(a);
        if p = complexInfinity then
            p := infinity;
            if Puiseux::direction(a) = Right then
              p := -infinity;
            end_if;
            sgn := -1;
            t := x;
        else
            sgn := 1;
            t := x - p;
        end_if;

        bo := extop(a, 2); // branch order
        v := extop(a, 3); // valuation
        l := extop(a, 5); // coefficient list

        // error term
        bigO:= t^(sgn*extop(a, 4)/bo);

        //------------------------------------
        // Walter (3.8.05): dirty hack
        // Puiseux series may contain sin, cos, ln, frac term in
        // their *coefficient* list l.
        // Ignore such coefficients (replacing them by 1):
        myl:= subs(l, [hold(sin) = 1, 
                            hold(cos) = 1, 
                            hold(signIm) = 1,
                            hold(frac) = 1,
                            hold(ln) = 1,
                            hold(log) = 1], EvalChanges);
        //------------------------------------
        if not has(myl, x) then
          // The input is a Puiseux series
          ll := [];
          for i from 1 to nops(l) do
            if iszero(l[i]) = FALSE then
               ll := append(ll, [l[i], t^(sgn*(v + i - 1)/bo)]) 
            end_if
          end_for;
          return(gseries::create(ll, bigO, x = p))
        end_if;

        // The input is not a valid Puiseux series
        // in a mathematical sense. Thus we convert the
        // input into an ordinary expression and apply 'asympt'
        a := gseries::new(Puiseux::expr(a), x = p);
        bigO := gseries::create([], bigO, x = p);
        return(a + bigO)
    otherwise
        return( FAIL )
    end_case
end_proc:

/*------------------------------------------------------------------------
    convert_to  --  convert a generalized series to another domain.
------------------------------------------------------------------------*/
gseries::convert_to := proc(s,T)
    local e;
begin
    if domtype(T) <> DOM_DOMAIN then T:= domtype(T) end_if;

    case T
    of gseries do
        return( s )
    of Puiseux do
        return(Puiseux::convert(s))
    of DOM_POLY do
        e:= gseries::expr( s );
        return( poly(e,[extop(s,3)]) )
    of DOM_EXPR do
        return( gseries::expr(s) )
    otherwise
        return( FAIL )
    end_case
end_proc:

/*------------------------------------------------------------------------
    Overload lmonomial (leading monomial) for gseries.
------------------------------------------------------------------------*/
gseries::lmonomial := proc(a)
    local l;
begin
    if args(0) <> 1 then
        error("expecting one argument")
    end_if;

    l:=extop(a,1);
    
    while nops(l) > 0 and gseries::zerotest(l[1][1]) do
       delete l[1]
    end_while;
    

    if nops(l)=0 then 
        return( FAIL )
    else 
        return( l[1][1]*l[1][2] )
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload lterm (leading term) for gseries.

    Example: lterm(3*exp(1/x)+y*ln(x)+O(x^4)) -> exp(1/x) 
------------------------------------------------------------------------*/
gseries::lterm := proc(a)
    local l;
begin
    if args(0) <> 1 then
        error("expecting one argument")
    end_if;

    l:=extop(a,1);

    while nops(l) > 0 and gseries::zerotest(l[1][1]) do
       delete l[1]
    end_while;    

    if nops(l) = 0 then 
        return( FAIL )
    else 
        return( l[1][2] )
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload ldegree (leading degree) for gseries.

    May return FAIL!
------------------------------------------------------------------------*/
gseries::ldegree := proc(a)
    local l,e,x,p,t,dir;
begin
    if args(0) <> 1 then
        error("expecting one argument")
    end_if;

    x:= extop(a,3); p:= extop(a,4);
    if domtype(p) = stdlib::Infinity then 
        t:= 1/x; dir:= null()
    else
        t:= x - p; dir:= hold(Right)
    end_if;
        
    l:= extop(a,1);
    while l <> [] do
        e:= l[1];
        if gseries::zerotest(e[1]) then
           delete l[1];
        elif e[2] = t then
            return( 1 )
        elif type(e[2]) = "_power" and op(e[2],1) = t and not has(op(e[2],2), x)
        then
            return( op(e[2],2) )
        else
            e:= series( e[1]*e[2], x=p, dir ); 
            case domtype(e)
            of gseries do 
                return( 0 )
            of Puiseux do
                return( Puiseux::ldegree( e ) )
            otherwise
                return( FAIL )
            end_case
        end_if
    end_while;
    return( FAIL )
end_proc:

/*------------------------------------------------------------------------
    Overload lcoeff (leading coeffient) for gseries.

    Example: lcoeff(3*exp(1/x)+y*ln(x)+O(x^4)) -> 3 
------------------------------------------------------------------------*/
gseries::lcoeff := proc(a)
    local l;
begin
    if args(0) <> 1 then
        error("expecting one argument")
    end_if;

    l:=extop(a,1);
    
    while nops(l) > 0 and gseries::zerotest(l[1][1]) do
       delete l[1]
    end_while;
        
    if nops(l)=0 then 
        return( FAIL )
    else 
        return( l[1][1] )
    end_if
end_proc:

gseries::zerotest:=
prog::remember(
proc(a)
begin
  Puiseux::iszero(a)
end_proc,
property::depends
):


/*------------------------------------------------------------------------
    Overload coeff for gseries.
------------------------------------------------------------------------*/
gseries::coeff := proc(a,n)
    local l;
begin
    case args(0)
    of 1 do
        return( op(extop(a,1)) )
    of 2 do
        if n = All then
          return( op(extop(a,1)) )
        end_if;
        if not testtype(n,Type::PosInt) then 
           error("2nd argument must be a positive integer") 
        else
            l:=extop(a,1);
            if n > nops(l) then 
                return( FAIL )
            else 
                return( l[n][1] )
            end_if
        end_if
    otherwise
        error("expecting one or two argument(s)")
    end_case
end_proc:

/*------------------------------------------------------------------------
    Overload nthterm for gseries.
------------------------------------------------------------------------*/
gseries::nthterm := proc(a,n)
    local l;
begin
    if args(0) <> 2 then
        error("expecting two arguments")
    elif not testtype(n,Type::PosInt) then 
        error("2nd argument must be a positive integer") 
    else
        l:= extop(a,1);
        if n > nops(l) then 
            return( FAIL )
        else 
            return( l[n][2] )
        end_if
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload nthmonomial for gseries.
------------------------------------------------------------------------*/
gseries::nthmonomial := proc(a,n)
    local l;
begin
    if args(0) <> 2 then
        error("expecting two arguments")
    elif not testtype(n,Type::PosInt) then 
        error("2nd argument must be a positive integer") 
    else
        l:= extop(a,1);
        if n > nops(l) then 
            return( FAIL )
        else 
            return( l[n][1]*l[n][2] )
        end_if
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload nthcoeff for gseries.
------------------------------------------------------------------------*/
gseries::nthcoeff := proc(a,n)
    local l;
begin
    if args(0) <> 2 then
        error("expecting two arguments")
    elif not testtype(n,Type::PosInt) then 
        error("2nd argument must be a positive integer") 
    else
        l:= extop(a,1);
        if n > nops(l) then 
            return( FAIL )
        else 
            return( l[n][1] )
        end_if
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload nterms
-------------------------------------------------------------------------*/

gseries::nterms:=
proc(S: Series::gseries): DOM_INT
begin
  nops(extop(S, 1))
end_proc:


/*------------------------------------------------------------------------
    Overload expr for gseries.
------------------------------------------------------------------------*/
gseries::expr := proc(a)
    local l,i;
begin
    l:= extop(a,1);
    if l = [] then 
        return(0)
    else
        return( _plus(_mult(op(l[i])) $ i=1..nops(l)) )
    end_if
end_proc:

/*------------------------------------------------------------------------
    Overload _plus for gseries.
------------------------------------------------------------------------*/
gseries::_plus :=
proc(a, b)
  local c, x, X, i, j, ca, cb, bigO, n, finish;
begin
  case args(0)
    of 0 do
      return( 0 )
    of 1 do
      return( a )
    of 2 do
      if domtype(a) <> gseries then
        // domtype(b) = gseries
        return(gseries::_plus(b, a));
      elif domtype(b) = Puiseux then
        b:= gseries::convert(b);
        if b = FAIL then
          return(FAIL)
        end_if;
      elif domtype(b) <> gseries then
        if iszero(b) then
          return(a);
        elif not has(b, extop(a, 3)) then
          // b is constant
          b:= extsubsop(a, 1 = [[b, 1]], 2 = 0);
        else // b is not constant
          // try to convert via gseries::new
          b:= gseries::new(b, extop(a, 3) = extop(a, 4));
          if domtype(b) <> gseries then
            return(FAIL);
          end_if
        end_if;
      end_if;
      assert(domtype(a) = gseries);
      assert(domtype(b) = gseries);

      if extop(a,3..4) <> extop(b,3..4) then
            // variables or expansion points don't match
        return(FAIL)
      end_if;

      // for gseries::dominates I need to perform a variable substitution: 
      // replace x by X s.t. X -> 0+ if x -> c = extop(a, 4):
      x := extop(a, 3);
      case extop(a, 4)
        of infinity do
          X := 1/x; break
        of -infinity do
          X := -1/x; break
        otherwise
          X := x + extop(a, 4);
      end_case;

        // handle the bigO-term as an ordinary term, but keep the information (here: NIL)
        // which term the bigO-term was:
      ca:= extop(a,1) . [[NIL, extop(a,2)]];
      cb:= extop(b,1) . [[NIL, extop(b,2)]];
      
      // insert the entries of the ordered list
      // cb = [cb[1] >> cb[2] >> ...] into the ordered list
      // ca = [ca[1] >> ca[2] >> ...] w.r.t. the asymptotical scale:
      bigO:= extop(a,2);
      finish:= FALSE;
      i:= 0;
      for j from 1 to nops(cb) do
        i:= i + 1;
        n:= nops(ca);
        while i <= n do
          c:= gseries::dominates(subs(ca[i][2], x=X), subs(cb[j][2], x=X), x);
          if c =FAIL then
            return(FAIL)
          elif c = 1 then
            // cb[j] << ca[i]
            i:= i + 1;
          elif c = -1 then
            // cb[j] >> ca[i] for the first value of i
            if cb[j][1] = NIL and cb[j][2] <> 0 then
              // cb[j] is the O-term of b
              bigO:= cb[j][2];
              ca:= [op(ca, 1..i-1)];
              finish:= TRUE;
            else
              ca:= listlib::insertAt(ca, [cb[j]], i);
            end_if;
            break // exit while-loop
          else
                    // c = 0:
                    // both are of the same asymptotical scale, so just add the two
                    // factors of corresponding terms and take the term of a as a
                    // representative:
            if ca[i][1] = NIL then
              // ca[i] is the O-term of a
              if ca[i][2] = 0 then
                // a is exact (O-term = 0)
                if cb[j][1] = NIL then
                  // cb[j] is the O-term of b
                  bigO:= cb[j][2];
                  delete ca[i];
                  finish:= TRUE;
                else
                  ca[i][1]:= cb[j][1];
                end_if
              else // a is not exact
                bigO:= ca[i][2];
                delete ca[i];
                finish:= TRUE;
              end_if
            elif cb[j][1] = NIL then
              // ca[i] is not the O-term of a and
              // cb[j] is the O-term of b
              if cb[j][2] <> 0 then
                // b is not exact (O-term <> 0)
                bigO:= cb[j][2];
                ca:= [op(ca, 1..i-1)];
                finish:= TRUE;
              end_if
            else
              // ca[i] is not the O-term of a and
              // cb[j] is not the O-term of b
              if ca[i][2] = cb[j][2] then
                ca[i][1]:= ca[i][1] + cb[j][1]
              else
                // ca[i][2] and cb[j][2] are different
                // (but their quotient remains bounded)
                // I see no other chance but putting their sum into
                // the asymptotic scale
                // Walter 29.4.08: falls ca[i][2] = 0, so wird
                // dieser Term später entfernt.
                ca[i][2]:= ca[i][1]*ca[i][2] + cb[j][1]*cb[j][2];
                ca[i][1]:= 1;
              end_if;
            end_if;
            break // exit while-loop
          end_if
        end_while;
        if finish or i > n then
          break // exit for-loop
        end_if;
      end_for;
      if not finish then
        delete ca[nops(ca)];
      end_if;

      // Walter 29.4.08: falls irgendwo ca[i][1] = 0 oder
      // ca[i][2] = 0 erzeugt wurde, so wird dieser Term 
      // nun entfernt.
      ca:= select(ca, x -> (_not@iszero)(x[1]) 
                       and (_not@iszero)(x[2]));
      
      return(extsubsop(a, 1 = ca, 2 = bigO))
    otherwise // args(0) >= 3; proceed left-associatively
      c:= _plus( a,b );
      if c = FAIL then
        return( FAIL )
      else
        return(_plus(c, args(3..args(0))))
      end_if
  end_case
end_proc:

/*------------------------------------------------------------------------
    Overload _subtract for gseries.
------------------------------------------------------------------------*/
gseries::_subtract:= (x,y) -> x + _negate(y):

/*------------------------------------------------------------------------
    Overload _mult for gseries.
------------------------------------------------------------------------*/
gseries::_mult:= proc(a,b)
    local n, i, j, k, e, c, x, X, la, lb, lm, nlb, bigO, bigOinX;
begin
    case args(0)
    of 0 do
        return( 1 )
    of 1 do
        return( a )
    of 2 do
        if domtype(a) <> gseries then
            // domtype(b) = gseries
            return(gseries::_mult(b, a))
        elif domtype(b) = Puiseux then
            b:= gseries::convert(b);
            if b = FAIL then
               return(FAIL)
            end_if;
        elif domtype(b) <> gseries then
            if iszero(b) then
                return(extsubsop(a, 1 = [], 2 = 0))
            elif b = 1 then
                return(a);
            elif not has(b, extop(a, 3)) then
                // scalar multiplication by a constant
                return(extsubsop(a, 1 = map(extop(a, 1), t -> [t[1]*b, t[2]])))
            else
                // try to convert via gseries::new
                b := gseries::new(b, extop(a, 3) = extop(a, 4));
                if domtype(b) <> gseries then
                   return(FAIL)
                end_if
            end_if;
        end_if;
        assert(domtype(a) = gseries);
        assert(domtype(b) = gseries);

        if extop(a,3..4) <> extop(b,3..4) then
            // variables or expansion points don't match
            return(FAIL)
        end_if;

        // shortcut if a or b has the form O(..)
        if extop(a, 1) = [] then
          if extop(b, 1) = [] then
            return(gseries::scalmult(a, 1, extop(b, 2)))
          else
            return(gseries::scalmult(a, 1, op(extop(b, 1), [1, 2])))
          end_if
        elif extop(b, 1) = [] then
          return(gseries::scalmult(b, 1, op(extop(a, 1), [1, 2])))
        end_if;

        // shortcut: use scalmult if a or b has only one operand
        if extop(a, 2) = 0 and nops(extop(a, 1)) = 1 then
          return(gseries::scalmult(b, op(extop(a, 1), [1, 1]),
                                      op(extop(a, 1), [1, 2])))
        elif extop(b, 2) = 0 and nops(extop(b, 1)) = 1 then
          return(gseries::scalmult(a, op(extop(b, 1), [1, 1]),
                                      op(extop(b, 1), [1, 2])))
        end_if;

        // for gseries::dominates I need to perform a variable substitution: 
        // replace x by X s.t. X -> 0+ if x -> c = extop(a, 4):
        x := extop(a, 3);
        case extop(a, 4)
        of infinity do
            X := 1/x; break
        of -infinity do
            X := -1/x; break
        otherwise
            X := x + extop(a, 4);
        end_case;

        la:= extop(a,1);
        lb:= extop(b,1);
        lm:= [];
        // the trivial cases were already handled above and
        // neither a nor b consist only of an O-term
        assert(la <> []);
        assert(lb <> []);

        // determine O-term of the product
        bigO := normal(la[1][2]*extop(b,2));
        e := normal(lb[1][2]*extop(a,2));
        if gseries::dominates(subs(bigO, x = X), subs(e, x = X), x) = -1 then
            bigO := e
        end_if;
        bigOinX := subs(bigO, x = X);
	
        lm:= [];
        nlb:= nops(lb);
        // sort the list of products of terms w.r.t. to the asymptotical scale
        for i from 1 to nops(la) do
            for j from 1 to nlb do
                e:= [la[i][1]*lb[j][1], normal(la[i][2]*lb[j][2])];
                c:= normal(e[2]);
                if gseries::dominates(subs(c,x=X), bigOinX, x) <> 1 then
                    // product of these terms already have higher order than bigO,
                    // so exit the for-loop:
                    break // j-loop!
                end_if;
	
                // insert e into the (sorted) list lm; compare e with
                // each element in lm:
                n:= nops(lm);
                for k from 1 to n do
                    c:= gseries::dominates(subs(lm[k][2],x=X), subs(e[2],x=X), x);
                    if c = -1 then
                        // e >> lm[k] (for x->0, hence lm[k] >> e for x -> oo)
                        lm:= listlib::insertAt(lm, [e], k);
                        break
                    elif c = 0 then
                        // ??? is this correct ???
                        lm[k][1]:= lm[k][1] + e[1];
                        break
                    end_if;
                end_for;
                if k > n then
                    lm:= append(lm, e);
                end_if;
            end_for;
        end_for;
	
        // Remove zero monomials:
        lm:= select(lm, x -> (_not@iszero)(x[1]));


        return(extsubsop(a, 1 = lm, 2 = bigO))
    otherwise // args(0) >= 3; multiply left-associatively
        a:= _mult( a,b );
        if a = FAIL then
            return( FAIL )
        else
            return(_mult( a,args(3..args(0)) ))
        end_if
    end_case
end_proc:

/*------------------------------------------------------------------------
    Overload _power for gseries. The exponent must not
    involve the series variable.

Bugfix: Walter, 13.9.05: no information on the requested
  order was passed to _power when called in Series::series.
  Example: series(sech(x), x = infinity, 19) produced only 
           only 6 terms instead of the requested 19.

  I added the additional optional argument ord. To make the 
  calling syntax consistent with Puiseux::_power, I also had 
  to introduce the dummy argument 'var' with some ridiculous 
  default value.
------------------------------------------------------------------------*/
gseries::_power:= proc(s, k, var = `#dummy`, ord = ORDER)
    local l, bigO, x, X, bigOinX, m, r, a, i, lr, p;
begin
    if has(k, extop(s, 3)) then
        error("exponent must not contain the series variable");
    end_if;

    case k
    of  1 do 
        return(s)
    of  0 do
        return(extsubsop(s, 1 = [[1, 1]], 2 = 0))
    end_case;

    [l, bigO, x] := [extop(s, 1..3)];

    // shortcuts
    if nops(l) = 0 then // s = O(..)
        if testtype(k, Type::Constant) and is(k >= 0) = TRUE then
          return(extsubsop(s, 2 = bigO^k))
        else
          error("cannot compute power")
        end_if
    elif bigO = 0 and nops(l) = 1 then
        // s is exact and has one term only
        return(extsubsop(s, 1 = [map(op(l), _power, k)]))
    end_if;
    // now s has at least two terms (including the O-term)

    if k = DOM_INT and k > 1 then
        // use repeated squaring
        a := 1;
        repeat
          // invariant: result = s^k * a
          if k mod 2 = 1 then
            a := gseries::_mult(s, a);
          end_if;
          if k > 1 then
            s := gseries::_mult(s, s);
          end_if;
          k := k div 2;
        until k = 0 end_repeat;
        return(a);
    end_if;

    // write s = m*(1 + r)
    m := l[1];
    delete l[1];
    l := map(l, z -> [z[1]/m[1], normal(z[2]/m[2])]);
    bigO := normal(bigO/m[2]);
    r := extsubsop(s, 1 = l, 2 = bigO);

    // for gseries::dominates I need to perform a variable substitution: 
    // replace x by X s.t. X -> 0+ if x -> c = extop(r, 4):
    case extop(r, 4)
    of infinity do
        X := 1/x; break
    of -infinity do
        X := -1/x; break
    otherwise
        X := x + extop(r, 4);
    end_case;

    // a := 1 + (O-term of r)
    a := extsubsop(r, 1 = [[1,1]]);
    if l <> [] then // r <> O(..)
        lr := subs(l[1][2], x = X); // leading term of r
        bigOinX := subs(extop(r, 2), x = X); // error term of r and a

        // compute (1 + r)^k = 1 + k*r + k*(k-1)/2 * r^2 + ... (binomial thm)
        // stop if the leading term of r^i is dominated by the O-term of
        // what has been computed so far, or if i = ORDER at last
//------------------------------------------------------
// Bugfix: Walter, 13.9.05, introduced optional argument
// ord and replaced 'for i from 1 to ORDER do' by
//------------------------------------------------------
        for i from 1 to ord do
            if gseries::dominates(lr^i, bigOinX, x) <> 1 then
                // lterm(r^i) is dominated by the O-term of a --> "early abort"
                break
            end_if;
            if i = 1 then
                p := k*r
            else
                // p := (k - i + 1)/i * r * p
                p := gseries::_mult(p, r);
                p := gseries::scalmult(p, (k - i + 1)/i);
            end_if;
            a := gseries::_plus(a, p);
            // invariants:
            //   p = binomial(k, i)*r^i
            //   a = 1 + k*r + ... + binomial(k, i)*r^i
        end_for;

        // "regular abort"
//------------------------------------------------------
// Bugfix: Walter, 13.9.05, introduced optional argument 
// ord that replaces ORDER
//------------------------------------------------------
        if i > ord then
            // add error term O(lterm(r)^ord)
            a := gseries::_plus(a, extsubsop(a, 1 = [], 2 = l[1][2]^ord))
        end_if;
    end_if;

    // return m^k*a = m^k*(1 + r)^k
    return(extsubsop(a, 
        1 = map(extop(a, 1), z -> [z[1]*m[1]^k, normal(z[2]*m[2]^k)]),
        2 = normal(extop(a, 2)*m[2]^k)
    ))
    
end_proc:


/*------------------------------------------------------------------------
    Overload _invert for gseries.
------------------------------------------------------------------------*/
gseries::_invert := a -> _power(a, -1):

/*------------------------------------------------------------------------
    Overload _divide for gseries.
------------------------------------------------------------------------*/
gseries::_divide := (a, b) -> a * _invert(b):

/*------------------------------------------------------------------------
    Overload _negate for gseries.
------------------------------------------------------------------------*/
gseries::_negate:= s -> extsubsop(s, 1 = map(extop(s,1), x -> [-x[1],x[2]])):

/*------------------------------------------------------------------------
    Overload simplify/Simplify for gseries:
------------------------------------------------------------------------*/
gseries::simplify :=
proc(S: Series::gseries)
begin
  extsubsop(S, 1=simplify(extop(S, 1)), 2=simplify(extop(S, 2)))
end_proc:

/*------------------------------------------------------------------------
    Overload testtype for gseries:
------------------------------------------------------------------------*/
gseries::testtype:=
proc(x, T)
begin
if T = Type::Arithmetical then 
  TRUE
else
  FAIL
end_if
end_proc:


/*------------------------------------------------------------------------
    Overload subs for gseries. This method applies the function 'subs' 
    to the first two operands of s
------------------------------------------------------------------------*/
gseries::subs := proc(s) 
begin 
    new(gseries,map( extop(s, 1..2),subs,args(2..args(0)) ), extop(s, 3..4))
end_proc:

//
// subsex ist gar nicht ueberladbar ...
//
///*------------------------------------------------------------------------
//    Overload subsex for gseries:
//------------------------------------------------------------------------*/
//gseries::subsex := proc(s) 
//begin 
//    new(gseries,map( extop(s, 1..2),subsex,args(2..args(0)) ), extop(s, 3..4))
//end_proc:

/*------------------------------------------------------------------------
    Overload iszero for gseries:
------------------------------------------------------------------------*/
gseries::iszero:= proc(s)
begin
    if extop(s,2) <> 0 then return( FALSE ) end_if;
    return(iszero( expand(_plus( op(map( extop(s,1),x -> _mult(x[1],x[2]) )) )) ))
end_proc:

/*------------------------------------------------------------------------
    Overload generate::TeX for gseries:
------------------------------------------------------------------------*/
gseries::TeX := s -> generate::TeX(gseries::print(s)):

/*------------------------------------------------------------------------
    Overload "has" for gseries:
------------------------------------------------------------------------*/
gseries::has := (s,f) -> has([extop(s)], f):

/*------------------------------------------------------------------------
    Overload "diff" for series.
    Note that the result makes no sense mathematically;
    a small term that has been neglected when constructing the
    series may have a large derivative. Therefore, we do not
    differentiate the O-term (should we?)
------------------------------------------------------------------------*/
gseries::diff:=
proc(s, x)
begin
  case args(0)
    of 1 do
      return(s)
    of 2 do
      return(series(diff(expr(s), x), gseries::indet(s) = gseries::point(s)))
    otherwise
      return(diff(diff(s, x), args(3..args(0))))
  end_case
end_proc:

/*------------------------------------------------------------------------
    Overload "int" for series.
------------------------------------------------------------------------*/

gseries::int:= proc(s, x)  // compute int(s, x) with a gseries s
local l, L, Oterm, var, p, furtherargs, xx, order;
begin
  if domtype(s) <> Series::gseries then
     return(FAIL);
  end_if:
  if type(x) <> DOM_IDENT and
     type(x) <> "_index" and
     type(x) <> "_equal" then
     return(FAIL);
  end_if:

  L:= extop(s, 1);    // = [[c1,expr1], [c2, expr2], ...]
  Oterm:= extop(s, 2);// the expression inside the O-term 
  var:= extop(s, 3);  // expansion variable
  p:= extop(s, 4);    // expansion point

  furtherargs:= args(3 .. args(0));

  if domtype(x) = DOM_IDENT and
     has(p, x) then
       // x occurs in the expansion point --> no chance
       return(FAIL);
  end_if;

  // the following code holds if the series variable var and
  // the integration variable xx coincide or do not coincide!
  if type(x) = "_equal" then
     xx:= op(x, 1);
  else
     xx:= x;
  end_if:
  L:= map(extop(s, 1), proc(l) begin
            if not has(l[2], xx) then
              [int(l[1], x, furtherargs), l[2]]
            elif not has(l[1], xx) then
              [l[1], int(l[2], x, furtherargs)] 
            else
              [1, int(l[1]*l[2], x, furtherargs)]
            end_if;
          end_proc);
  if hastype(L, "int") then
     if domtype(x) = "_equal" and
        op(x, 1) = var then
        return(  _plus(l[1]*l[2] $ l in L)
               + hold(int)(O(Oterm), x, furtherargs)
              );
     else
        return(extsubsop(s, 1 = L,
                            2 = int(Oterm, x, furtherargs),
                            4 = p));
     end_if:
  end_if;
  // Here, all integrals in L could be evaluated
  order:= nops(L);
  L:= _plus(l[1]*l[2] $ l in L);
  if type(x) = "_equal" and
     op(x, 1) = var then
     return(L + hold(int)(O(Oterm), x, furtherargs));
  else
     return(asympt(L, var = p, order));
  end_if:
end_proc:

/*------------------------------------------------------------------------
    Functions which are not implemented:
------------------------------------------------------------------------*/
gseries::rectform:= FAIL:

/*------------------------------------------------------------------------
    Interface methods for extracting the series variable and the 
    series point:
------------------------------------------------------------------------*/
gseries::indet:= s -> extop(s,3):
gseries::point:= s -> extop(s,4):

unalias( gseries ):

