//**********************************************************//
alias(Puiseux = Series::Puiseux):
alias(gseries = Series::gseries):

//**********************************************************//
// Series::series(f, x, n, dir): expand the expression f into a series
// around the point x = 0 with order n in direction dir, if specified.
//
// x: identifier
// f: arithmetical expression in x
// n: nonnegative integer
// dir: (optional): Real, Left, or Right
//
// no argument checking is performed; this is done by the interface
// function series
//
// CHANGED (jngerhar) 25.02.01:
// - rewritten the code for converting O-terms
// - inserted check for the trivial case where f does not contain x
//
Series::series :=
proc(f, x, n=ORDER, dir=Undirected, opt= series::defaultOptions)
   local tf, p, opf0, Opoints, s;
begin
  
  userinfo(2, "computing expansion of", f);
  
  // trivial case: f does not depend on x
  if iszero(f) then // O(x^n)
    return(Puiseux::zero(x, n, dir))
  elif not has(f, x) then // f + O(x^n)
    return(Puiseux::const(f, x, n, dir));
  end_if;

  // handle O-terms
   if domtype(f) = O then
     // check whether the expansion of the O-term is around x = 0
     Opoints := O::points(f);
     if not has(Opoints, x) then
       // series variable does not occur in the O-term
       // ==> make it a coefficient
       return(Puiseux::const(f, x, n, dir));
     elif not has(Opoints, x = 0) then
       Series::error("O-term has a different expansion point");
     end_if;

     // convert the O-term into a series
     s := Series::series(op(f, 1), args(2..args(0)));
     if not contains({Puiseux, gseries}, domtype(s)) then
       Series::error("cannot convert O-term into a series");
     end_if;

     if nops(Opoints) = 1 then
       // univariate O-term around x = 0
       // ==> just take lowest order term
       if s::dom::iszero(s) then
         return(s)
       elif domtype(s) = Puiseux and not has(lcoeff(s), x) then
         return(Puiseux::truncate(s, Puiseux::ldegree(s)))
       else // Series::gseries or has(lcoeff(s), x)
         if opt[UseGseries] then
           return(gseries::create([], lmonomial(s), x = 0))
         else
           return(FAIL)
         end_if
       end_if
     else
       // multivariate O-term ==>
       // convert coeffs into O-terms again
       return(map(s, O, op(Opoints)));
     end_if
   end_if; // domtype(f) = O

   if testtype(f, Type::PolyExpr(x, Type::AnyType)) then
     // f is a polynomial in x
     Puiseux::frompoly(poly(f, [x]), x, n, dir)
   else // recursively descend into the expression

      if MAXEFFORT < 10 then
        return(FAIL)
      end_if;
     
      tf := type(f); // "exp" or "tan" or ...

      // first catch functions with variable number of arguments 
      if tf = "_plus" then
        Series::plus(f, x, n, dir, opt)
      elif tf="_mult" then
        Series::mult(f, x, n, dir, opt)
      elif tf="_power" then
        Series::power(op(f), x, n, dir, opt);
      elif tf="int" then 
        if ((domtype(x) = DOM_IDENT and x = op(f, 2))) or
           ((type(x) = "_equal") and op(x, 1) = op(f, 2)) then
          return(int(series(op(f, 1), x, n, dir, opt), op(f, 2..nops(f))))
        end_if;
        if not has([op(f, 2..nops(f))], op(x, 1)) then
          return(int(series(op(f, 1), x, n, dir, opt), op(f, 2..nops(f))));
        end_if;
        Series::unknown(args())

      // specfuncs with one or two args
      elif tf="psi" and nops(f) = 1 then
        psi::series(op(f), 0, x, n, dir, opt)
      elif tf="dirac" and nops(f) = 1 then
        dirac::series(op(f), 0, x, n, dir, opt)
      elif tf="zeta" and nops(f) = 1 then
        zeta::series(op(f), 0, x, n, dir, opt)

      // functional expressions make no sense
      elif tf = "_fconcat" or tf = "_fnest" then
        FAIL
      else
         // if the 0th operand of f has a series slot, then
         // call it
         opf0 := op(f, 0);
         // the eval is necessary since DOM_VARs are evaluated with
         // depth 1 only and the object that opf0 refers to may change
         // after the following slot command due to loadproc effects
         if domtype(eval(opf0)) = DOM_FUNC_ENV then
            p := slot(eval(opf0), "series");
            if contains({DOM_PROC,DOM_EXEC}, domtype(p)) then
              return(p(op(f), x, n, dir, opt))
            end_if;
         end_if;

         // last resort: try formal Taylor expansion via diff
         if traperror((p := Series::unknown(args()))) = 0 then
            p
         else
            FAIL
         end_if
      end_if;
   end_if; 
end_proc:


// if an expression is not evaluable at x=0, try a few iterations
// of l'Hospital, if applicable
Series::applyLHospital :=
proc(f, x, maxsteps=10)
  local num, den;
begin
  [num, den] := [numer, denom](f);
  while maxsteps>0
    and not iszero(num)
    and not iszero(den)
    and iszero(subs(den, x=0,EvalChanges))
    and iszero(subs(num, x=0, EvalChanges)) do
    num := diff(num, x);
    den := diff(den, x);
    f := normal(num/den);
    [num, den] := [numer, denom](f);
    maxsteps := maxsteps - 1;
  end_while;
  f;
end_proc:

//**********************************************************//
// Series::unknown(f, x, n, dir): compute the formal Taylor expansion
//   of the expression f around x = 0 of order n by using the Taylor
//   formula and diff. The direction dir is ignored.
//
// x: identifier
// f: arithmetical expression in x
// n: nonnegative integer
//
// no argument checking is performed; this is done by the interface
// function series
//
Series::unknown :=
proc(f,x,n,dir = Undirected)
  local g,fa,i,g0,l, h;
begin
  userinfo(3, "Series::unknown called with argument ".
           expr2text(f));
  fa:=1;
  g:=normal(f);
  userinfo(8, "Normalizing gives ".expr2text(g));
  if traperror((g0:=subs(g,x=0,EvalChanges))) <> 0 or
    contains({FAIL, undefined}, g0) then
    return(FAIL);
  end_if;
  if has(g0, infinity) then
    // we have probably evaluated g0 at a pole, so no Taylor expansion
    // exists
    return(FAIL)
  end_if;
  if n = 0 then return(Puiseux::zero(x, 0, dir)) end_if;
  userinfo(5, "Function value is ".expr2text(g0));
  l:=[g0];
  for i from 1 to n-1 do
    g:=diff(g,x);
    userinfo(5, expr2text(i)."th derivative is ".expr2text(g));
    g := rewrite(g, D);
    userinfo(15, "Converting diff to D gave ".expr2text(g));
    fa:=fa*i;
    if traperror((g0 := subs(g, x=0, EvalChanges))) <> 0 or
      contains({FAIL, undefined}, g0) then
      if traperror((h := Series::applyLHospital(g, x))) <> 0 then
        return(FAIL)
      end_if;
      userinfo(10, "after application of l'Hospital's rule: ".expr2text(h));
      if traperror((g0 := subs(h, x=0, EvalChanges))) <> 0 or
        contains({FAIL, undefined}, g0) then
        return(FAIL);
      end_if;
    end_if;
    l := append(l, g0/fa)
  end_for;
  Puiseux::normal(Puiseux::create(1,0,n,l,x,0,dir), 0) // quick normalization
end_proc:


/*********************************************************************/
Series::plus:=
proc(f: "_plus", x, n = ORDER, dir = Undirected, opt = series::defaultOptions)
local L, i;
  save MAXEFFORT;
begin
  userinfo(3, "Computing series of the sum ". expr2text(f));
  //--------------------------------------------------------
  // Walter 14.2.02: replaced the code above by the following.
  // First, convert all terms in the sum to series objects.
   // Series::series produces either Puiseux or gseries:
  //--------------------------------------------------------
  MAXEFFORT:= MAXEFFORT/nops(f)/2;
  L:= map([op(f)], Series::series, x, n, dir, opt);
  if has(L, FAIL) then
    return(FAIL)
  end_if;
  //--------------------------------------------------------
   // If both Puiseux and gseries are present in the sum,
   // we cannot just add them: Puiseux *always* has a
   // BigOh term. (In contrast to gseries, Puiseux does 
   // not keep track of exact objects).
   // This may cause problems, e.g.
   //  1 - erf(x) --> 
   //   (1 + O(x^6)) - (1 - 1/x/sqrt(PI)/exp(x^2)) + O(1/x^2/exp(x^2))
   //   = O(x^6) + 1/x/sqrt(PI)/exp(x^2) + O(1/x^2/exp(x^2))
   //   = O(x^6);
   // Workaround: recompute Puiseux terms via Series::gseries.
   // Do not use Series::gseries::convert, because this would
   // keep the BigOh!
  //--------------------------------------------------------
  if {Series::Puiseux, Series::gseries} minus
    {op(map(L, domtype))} = {} then
    for i from 1 to nops(L) do
      if domtype(L[i]) <> Series::gseries then
        // gseries cannot handle Undirected or Real
        if dir <> Right and dir <> Left then
          dir:= null()
        end_if;
        L[i]:= Series::gseries(op(f, i), x, n, dir);
        if L[i] = FAIL then
          return(FAIL)
        end_if;
      end_if;
    end_for;
  end_if;
  //--------------------------------------------------------
  // Now, all terms in L should all be either of type
  // Series::Puiseux or Series::gseries. Doublecheck:
  //--------------------------------------------------------
  if nops({op(map(L, domtype))}) <> 1 then
    return(FAIL)
  end_if;
  //--------------------------------------------------------
  // call the overloaded _plus methods:
  //--------------------------------------------------------
  _plus(op(L));
end_proc:


/**********************************************************/
Series::mult:=
proc(f: "_mult", x, n = ORDER, dir = Undirected, opt = series::defaultOptions)
  local s,i,k,j;
  save MAXEFFORT;
begin
  userinfo(3, "Computing series of the product ".
           expr2text(f));
  ([f,k,s]):=split(f,has,x); j:=0;

   // 
   // splits the product into
   // f :  expression in x  
   // k :  scalar factor
   // s :  unknown terms
   //
   if f=1 then // k + O(x^n)
      Puiseux::const(k, x, n, dir)
   else
      if type(f)<>"_mult" then
         s:=Series::series(f,args(2..args(0)))
      else
         ([j,f,s]):=split(f,proc() begin args(1)=x or type(args(1))="_power"
		and op(args(1),1)=x and type(op(args(1),2))=DOM_INT end_proc );
         //
         // j :  x, power in x
         // f :  expressions in x
         // s :  power in x with integer exponents
         //
         if j=1 then j:=0 elif j=x then j:=1 else j:=op(j,2) end_if;
         if type(f)<>"_mult" then s:=Series::series(f,args(2..args(0)))
      else
       MAXEFFORT:= MAXEFFORT/nops(f);
	    s:=Series::series(op(f,1),args(2..args(0)));
	    for i from 2 to nops(f) do
	       s:=s*Series::series(op(f,i),args(2..args(0)))
	    end_for;
      end_if
      end_if;
      // return k*s*x^j
      if domtype(s)=Puiseux then
         Puiseux::scalmult(s,k,j)
      else
         s::dom::scalmult(s,k,x^j)
      end_if
   end_if
end_proc:


/*********************************************************/
Series::power :=
proc(a, b, x, n, dir, opt)
  local s, result;
begin
  userinfo(3, "Computing series of a power");
  case domtype(b)
    of DOM_INT do
    of DOM_RAT do
      s:= _power(Series::series(a, args(3..args(0))), b, a, n);
      return(s)
    otherwise
      if iszero(a) then
        return(Puiseux::zero(x, n, dir))
      else
        result:= Series::series(hold(exp)(b*simplify(ln(a))),
                                args(3..args(0)));
        if type(result) = Series::gseries and nterms(result)=1 and
          extop(result, 2) = 0 then
          extsubsop(result, 1= [[1, a^b]])
        else
          result
        end_if
      end_if
  end_case
end_proc:


//**********************************************************
// Series::check_point(p)
//
// check that p is of the form x or x = x0, where x is an identifier
// and x0 is an arithmetical expression not containing x or infinity,
// or x0 = infinity, -infinity, or complexInfinity. Returns the list [x, x0].
//
// CHANGED (jngerhar) 25.02.01: new function
//
// To do: allow x to be an indexed identifier ???
//
Series::check_point :=
proc(p)
  local x, x0;
begin
   if domtype(p) = DOM_IDENT or type(p) = "_index" then
      // default expansion point 0, if not given
      x0 := 0;
       x := p;
   elif type(p) <> "_equal" or
   not contains({"_index", DOM_IDENT}, type(op(p, 1))) then 
      Series::error("invalid series variable")
   else 
      x0 := op(p, 2);
      x := op(p, 1);

      if not testtype(x0, Type::Arithmetical) then
         Series::error("invalid expansion point")
      elif has(x0, x) then
         Series::error("expansion point depends on series variable")
/* ------------------------------
      elif has(x0, infinity) and not contains({infinity, -infinity}, x0) then
         // this case is handled outside check_point by rewriting
         // series(f(x), x = I*infinity) to series(f(I*x), x = infinity)
------------------------------ */
      end_if;
   end_if;

   [x, x0]
end_proc:


//**********************************************************
// Series::algebraic(f, x, n, dir)
//
// compute all non-zero series solutions around x = 0 of an algebraic equation
// and returns them as a set
//
// f: RootOf(g(y, x), y), y an identifier different from x
// x: identifier
// n: nonnegative integer
// dir: Left, Right, Real, or Undirected
//
Series::algebraic :=
proc(f/* : RootOf */, x, n, dir: DOM_IDENT, opt: DOM_TABLE)
  local a, g, i, k, m, v, y, myconvert: DOM_PROC,
    mapconvert: DOM_PROC, solutions;
begin

  assert(args(0) = 5);
  
  userinfo(2, "Computing all non-zero series solutions of ".expr2text(f));

  [g, y] := [op(f)];
  g := poly(g, [y]);
  assert(domtype(g) = DOM_POLY);
  if y = x then
    error("cannot expand w.r.t. RootOf variable");
  end_if;

  // replace g by its squarefree part
  g := _mult(op(Factored::factors(polylib::sqrfree(g))));
  userinfo(3, "squarefree part is ".expr2text(g));

  // divide by y if possible to make the constant coefficient non-zero
  if iszero(coeff(g, y, 0)) then
    g := g / poly(y, [y]);
  end_if;    

  m := degree(g);
  if m = 0 then // shortcut: equation of degree zero
    return({})
  end_if;

  // expand all non-zero coefficients into Puiseux series around 0
  a := map(array(0..m, [coeff(g, y, i) $ i = 0..m]),
           z -> if iszero(z) then 0
                else Series::series(z, x, n, dir, opt) end_if);
  if map({op(a)} minus {0}, domtype) <> {Puiseux} or
     map({op(a)} minus {0}, extop, 1) <> {0} then // some coeffs have flag = 1
    error("cannot expand coefficients of ".expr2text(f)." into Puiseux series");
  end_if;

  if iszero(a[m]) then // a[m] = O(..)
    error("order too small")
  end_if;
  
  // divide all non-zero coefficients by the leading one
  a := map(a, z -> if z = 0 then 0
                   else Puiseux::condense(z/a[m]) end_if);

  if m = 1 then // shortcut: equation of degree 1
    return({-a[0]})
  end_if;

  // perform a change of variable y = y*x^k such that afterwards,
  // the valuation m*k of the leading coefficient is less than the
  // valuation of all other coefficients and k is maximal, and
  // divide all coefficients by x^(m*k)
  v := map(a, Series::valuation);
  k := min(v[i]/(m - i) $ i = 0..(m - 1));
  if domtype(k) = DOM_INT then
    k := k - 1;
  else
    k := floor(k);
  end_if;
  assert(domtype(k) = DOM_INT);
  for i from 0 to m - 1 do
    if a[i] <> 0 then
      a[i] := Puiseux::scalmult(a[i], 1, (i - m)*k);
    end_if
  end_for;
  // now a[m] = 1 and valuation(a[i]) > 0 for 0 <= i < m

  // convert a list of pairs [[c1, e1], [c2, e2], ..., [0, er]]
  // as returned by Series::_algebraic into the Series::Puiseux
  // c1*x^(e1+k) + c2*x^(e2+k) + ... + O(x^(er+k))
  myconvert :=
  proc(l)
    local r, d, i, s;
  begin
    r := nops(l);
    assert(r > 0);
    assert(testtype(l[r][2], Type::Rational));

    if r = 1 then // O(x^(er+k))
      return(Puiseux::zero(x, l[1][2] + k, dir))
    end_if;

    // determine branching order
    d := lcm(op(map(l, denom@op, 2)));

    s := [l[1][1]];
    for i from 2 to r - 1 do
      s := s . [0 $ (l[i][2] - l[i - 1][2])*d - 1, l[i][1]]
    end_for;

    Puiseux::create(d, (l[1][2] + k)*d, (l[r][2] + k)*d, s, x, 0, dir)
  end_proc:


  // mapconvert - applies myconvert to all elements of the set S
  mapconvert:=
  proc(S)
  begin
    case type(S)
      of DOM_SET do
        return(map(S, myconvert))
      of "_union" do
        return(map(S, mapconvert))
      of "function" do
        assert(op(S, 0) = `#ImageSet`);
        return(Dom::ImageSet(myconvert(op(S, 1)), op(S, 2), op(S, 3)))
    end_case;
    error("unexpected type of set ".expr2text(type(S)))
  end_proc;

  // call our work horse
  solutions := Series::_algebraic(a, -infinity, x, n, dir, opt);

  // convert the set of lists of pairs, as returned by Series::_algebraic,
  // into a set of Series::Puiseux and return the result

  mapconvert(solutions)
end_proc:


//**********************************************************
// auxilliary method
//
// Series::valuation(s) - determine the order of the lowest term in s
//
// s - Series::Puiseux or 0
//
// returns a rational number of infinity
//
Series::valuation := proc(s) begin
    if s = 0 then infinity
    elif Puiseux::ldegree(s) <> FAIL then
      Puiseux::ldegree(s)
    else
      Puiseux::order(s)
    end_if
end_proc:


//**********************************************************
// Series::_algebraic(a, w, x, n, dir, options, rof)
//
// a - array a[0],a[1],...,a[m]=1 of series expansions in x
//     some of the a[i] may be 0
//     valuation(a[i]) > 0 for 0 <= i < m
// w - rational number or -infinity
// x - identifier
// n - nonnegative integer
// dir: Left, Right, Real, or Undirected
// options - table
// rof: if present, specifies a "RootOf", and coefficient arithmetic
//      has to take place modulo the corresponding polynomial
//
// returns the set of all series solutions of the algebraic equation
//
//   y^n + a[n-1]*y^(n-1) + ... + a[0] = 0
//
// with valuation(a) > w
//
// each element of the solution set is stored in the form
//
//   [[c1, e1], [c2, e2], ..., [0, er]],
//
// representing the series c1*x^e1 + c2*x^e2 + ... + O(x^er)
//
// special coding: since a Dom::ImageSet only accepts arithmetical
// expressions, 
// we do not use it until myconvert in the main program has converted the
// nested lists into series

Series::_algebraic :=
proc(a, w, x, n, dir, options, rof = null())
  local C, F, S, aa, c, e, eq, i, j, l, lterms, m, p, solutions, u, v, tmp;
begin
  m := nops(a) - 1;

  // reduce all coefficients modulo rof, if present
  if args(0) > 6 then
    assert(domtype(rof) = RootOf);
    (if a[i] <> 0 then
       a[i] := map(a[i], divide, op(rof, 1), [op(rof, 2)], Rem)
     end_if) $ i = 0..m
  end_if;

//  // normalize all coefficients to have the same branch order b
//  b := ilcm(map(op(a), z -> if z = 0 then 1 else extop(z, 2) end_if));
//  (if a[i] <> 0 then a[i] := Puiseux::multbo(a[i], b/extop(a[i], 2)) end_if)
//   $ i = 0..m;
  
  // determine the slopes of the Newton polygon
  v := map(a, Series::valuation);
  e := array(0..m);
  for j from m downto 1 do
    u := infinity;
    for i from j - 1 downto 0 do
      if (v[i] - v[j])/(j - i) < u then
        u := (v[i] - v[j])/(j - i);
        l := i;
      end_if;
    end_for;
    if iszero(a[l]) then
      l := 0;
    end_if;
    (e[j] := u) $ j = (l + 1)..j;
    j := l + 1;
  end_for;
  e[0] := infinity;
  // now e[i] is the valuation corresponding to the slope of the
  // Newton polygon between i-1 and i,
  // infinity = e[0] >= e[1] >= ... >= e[m] > 0

  lterms := {};
  c := genident("c");
  for i from 1 to m do
    assert(testtype(e[i], Type::Rational));
    if e[i] <= w then
      break;
    end_if;
    if e[i - 1] > e[i] then
      // plug c*x^e[i] into the algebraic equation and solve for c by
      // setting the coefficient of x^(v[i-1]+(i-1)*e[i]) to 0
      eq := poly(_plus(
        if a[j] = 0 then null() else
          c^j*coeff(a[j], v[i - 1] + (i - 1 - j)*e[i])
        end_if
        $ j = 0..m), [c]);
      if eq = FAIL then // insufficient precision; recursion stops here
        lterms := lterms union {[0, e[i]]};
      elif args(0) > 6 then
        // calculate in an algebraic extension specified by rof

        // find all non-zero solutions
        eq := eq / poly(c^ldegree(eq), [c]);

        if degree(eq) = 1 then
          // C := -coeff(eq, 0)/coeff(eq, 1) modulo rof
          C := divide(
            -coeff(eq, 0)*op(gcdex(coeff(eq, 1), op(rof, 1), op(rof, 2)), 2),
            op(rof, 1), [op(rof, 2)], Rem);
  
          lterms := lterms union {[C, e[i]]}
        else // degree >= 2
          // this only works if the RootOf is univariate with rational coeffs
          if not testtype(op(rof, 1),
                          Type::PolyExpr(op(rof, 2), Type::Rational)) then
            userinfo(3, "warning: RootOf with non-rational coefficients encountered; cannot compute higher order terms");
            lterms := lterms union {[0, e[i]]};
          else
            // convert eq into a polynomial with coefficient domain equal
            // to the algebraic extension of Q generated by rof
            F := Dom::AlgebraicExtension(Dom::Rational, op(rof));
            eq := poly(eq, [c], F);

            // find only in-field roots
            userinfo(3, "warning: possible loss of solution");
            eq := solve(eq, c, MaxDegree = 1);
            // Warning: this does not yield all roots, so we may
            // lose some branches here!
            if type(eq) = "_union" then
              eq := [op(eq)];
            else
              eq := [eq];
            end_if;
            for S in eq do
              if type(S) = DOM_SET and nops(S) > 0 then
                lterms := lterms union map(S, z -> [expr(z), e[i]]);
              else
                userinfo(3, "warning: cannot compute higher order terms");
                lterms := lterms union {[0, e[i]]};
              end_if
            end_for
          end_if
        end_if;
      else // args(0) <= 6
        // find all non-zero solutions; RootOfs of degree 2 are solved below
        eq := solve(eq / poly(c^ldegree(eq), [c]), c, MaxDegree = 1);
        if type(eq) = "_union" then
          eq := [op(eq)];
        else
          eq := [eq];
        end_if;
        for S in eq do
          case domtype(S)
          of DOM_SET do
            lterms := lterms union map(S, z -> [z, e[i]]);
            break;
          of RootOf do
            // change RootOf variable to genident("z")
            S := subs(S, op(S, 2) = genident("z"));
            lterms := lterms union {[op(S, 2), e[i], S]};
            break;
          otherwise // hold(solve)(...), piecewise(...), ...
            userinfo(3, "warning: cannot compute higher order terms");
            lterms := lterms union {[0, e[i]]};
          end_case
        end_for
      end_if;
    end_if
  end_for;

  solutions := {};
  for p in lterms do
    if p[1] = 0 then
      solutions := solutions union {[p]}       
    else
      // perform change of variable y = y + p[1]*x^p[2]
      // aa[i] = sum(binomial(j, i)*p[1]^(j - i)*x^(p[2]*(j - i))*a[j],
      //             j = i..m)
      aa := array(0..m):
      aa[m] := a[m];
      for i from 0 to m - 1 do
        aa[i] := Puiseux::_plus(
          if a[j] = 0 then null() else
            Puiseux::scalmult(a[j], binomial(j, i)*p[1]^(j - i), p[2]*(j - i))
          end_if
          $ j = i..m);
        if aa[i] <> 0 and domtype(aa[i]) <> Puiseux then
          aa[i] := Puiseux(aa[i], x, n, dir);
        end_if;
      end_for;
      // recursively determine all solutions of the equation
      // sum(aa[i]*y^i) = 0 of valuation > p[2]
      // and prepend p[1]*x^p[2] to each of them
      if nops(p) = 2 then
        S := Series::_algebraic(aa, p[2], args(3..args(0)));
        if type(S) = "_union" then
          solutions := solutions union map(S, map, z -> _concat([p], z));
        else
          solutions := solutions union map(S, z -> _concat([p], z));
        end_if
      else // nops(p) = 3
        // special treatment of RootOfs

        // recursively determine all solutions of the equation
        // sum(aa[i]*y^i) = 0 of valuation > p[2]
        S := Series::_algebraic(aa, p[2], x, n, dir, options, p[3]);
        assert(domtype(S) = DOM_SET);

        // prepend p[1]*x^p[2] to each of them
        S := map(S, z -> _concat([[p[1], p[2]]], z));

        if degree(op(p[3], 1), p[1]) = 2 then
          // now we explicitly solve RootOfs of degree 2
          C := solve(op(p[3], 1), p[1]);
          /* Beware: there may be only one (double) solution C[1]
          solutions := solutions union
            map(S, subs, p[1] = C[1]) union
            map(S, subs, p[1] = C[2])
          */
          solutions:= _union(solutions, map(S, subs, p[1] = tmp) $ tmp in C);
        else
          solutions := solutions union
            op(map(S, `#ImageSet`, p[1], p[3]));
        end_if
      end_if
    end_if;
  end_for;

  solutions
end_proc:




//**********************************************************
// functions for "arithmetic" with the four directions
// Left, Right, Real and Undirected
//
Series::sign := proc(dir) begin
  case dir
  of Left do -1; break
  of Right do 1; break
  of Real do  0; break
  otherwise dir;
  end_case
end_proc:


//**********************************************************
Series::invert := proc(dir) begin
  case dir
  of Left do Right; break
  of Right do Left; break
  otherwise    dir;
  end_case
end_proc:


//**********************************************************
Series::min := proc(dir1, dir2)
  local s;
begin
  s := {dir1, dir2};
  if s = {Left, Right} then FAIL
  elif contains(s, Left) then Left
  elif contains(s, Right) then Right
  elif contains(s, Real) then Real
  elif s = {Undirected} then Undirected
  else FAIL
  end_if
end_proc:


//**********************************************************
// routines to recall the error message of the last error


Series::error :=
proc(s)
begin
  context(hold(error)(s))
end_proc:

// print the O term?
Series::printO :=
proc(tf)
begin
  if args(0)=0 then
    return(TRUE);
  end;
  
  if tf in {TRUE, FALSE} then
    sysassign(Series::printO(), tf);
  else
    error("illegal argument ".expr2text(tf));
  end;
end:


//**********************************************************
unalias(Puiseux):
unalias(gseries):
//**********************************************************
