//   

// 06/2003, bij, stefanw
//
// simplify::Simplifier
//   returns a simplifier procedure
//
// local (in this file) defined help procedures for
//
//
//
//

alias(ExpectedSimplificationPerStep = 0.98):



// returns a procedure, a simplifier reacting to given options
Simplify::Simplifier:=
proc()
  option escape;
  local OPTIONS, OPTIONT, PROC;
begin

  // set default options for simplifier created without options
  // !! however, each option can be set at any call to the created simplifier

  OPTIONS:= table(hold(Discard)     = Simplify::exclude,
                  hold(Goal)        = 0,  // default goal
                  hold(StopIf)      = FALSE,
                  hold(IsSimple)    = Simplify::isSimple,
                  hold(Seconds)     = infinity,
                  hold(Steps)       = 100,
                  hold(OutputType)  = "Best",
                  hold(SelectRules) = Simplify::selectRules,
                  hold(RuleBase)    = Simplify,
                  hold(ApplyRule)   = Rule::apply,
                  hold(Strategy)    = "Default",
                  hold(Valuation)   = Simplify::defaultValuation,
                  hold(FinalValuation) = Simplify::defaultFinalValuation,
                  hold(LocalStepPriority) = Simplify::localStepPriority,
                  hold(All)         = FALSE,
                  hold(AllValuation)= FALSE,
                  hold(ShowSteps) = FALSE,
                  hold(IgnoreAnalyticConstraints)      = FALSE,
                  hold(Remember)    = TRUE,
                  hold(MaxInfoDepth) = 0,
                  hold(KernelSteps) = 100,
                  // undocumented option AllOperands
                  // if we have an expression a1 + a2 + a3 
                  // and know an equivalent expressions bi for each ai, we substitute some ai by bi and some not, in all possible ways:
                  // a1 + a2 + a3, a1 + a2 + b3, a1 + b2 + a3, a1 + b2 + b3, b1 + a2 + a3, etc.
                  // with AllOperands = FALSE, this behavor may be switched off: we just substitute the operand we are just working on. 
                  // e.g., if we are working on the second term a2 and find out that b2 is equivalent, we only generate a1 + b2 + a3
                  hold(AllOperands) = FALSE
                  );
                 

  OPTIONT:= table(hold(Discard)     = Type::Union(Type::Function, DOM_BOOL),
                  hold(Goal)        = Type::AnyType,
                  hold(StopIf)      = Type::AnyType,
                  hold(IsSimple)    = Type::AnyType,
                  hold(Seconds)     = Type::Union(Type::Positive, stdlib::Infinity),
                  hold(Steps)       = Type::Union(Type::PosInt, stdlib::Infinity),
                  hold(OutputType)  = DOM_STRING, //
                  hold(SelectRules) = Type::Function,
                  hold(RuleBase)    = Type::Union(Type::ListOf(DOM_DOMAIN),
                                                  DOM_DOMAIN),
                  hold(ApplyRule)   = Type::Function,
                  hold(Strategy)    = DOM_STRING,
                  hold(Valuation)   = Type::Function,
                  hold(FinalValuation) = Type::Function,
                  hold(LocalStepPriority) = Type::Function,
                  hold(All)         = DOM_BOOL,
                  hold(AllValuation)= DOM_BOOL,
		  hold(ShowSteps) = DOM_BOOL,
                  hold(IgnoreAnalyticConstraints)      = DOM_BOOL,
                  hold(Remember)    = DOM_BOOL,
                  hold(MaxInfoDepth) = Type::Union(Type::NonNegInt, stdlib::Infinity),
                  hold(KernelSteps) = Type::AnyType,
                  hold(AllOperands) = DOM_BOOL
                  );
                

  // combine default options and user defined options
  OPTIONS:= prog::getOptions(1, [args()],  OPTIONS, TRUE, OPTIONT)[1];


  ///////////////////////////////////////////////////////
  //  begin of  S I M P L I F I E R  ////////////////////
  ///////////////////////////////////////////////////////
  PROC :=
   proc(EXPR)                // expression to simplify
     name Simplify;
     local SFUNC,
     valuation,
     finalValuation,
     paireval,
     localStepPriority,
     queueinsert,
     getQueueMinimalVal,
     proofout,
     exclude,
     selectRules,      // procedure, that select rules for given pattern
     ruleBase,         // collection of rules, used as first argument of select rules
     apply,            // procedure, that applies given rule to given expression
     strategy,
     expressions: DOM_TABLE, // expressions[EXPR]: list of all expressions
                            // equivalent to EXPR
     exprnops: DOM_INT, // the first exprnops expressions in expressions[EXPR]
                        // were known before, all otheers have just been obtained
     proofs: DOM_TABLE,      // proofs[EXPR][i]: a proof that
                            // expressions[EXPR][i] is equivalent to EXPR
                            // such proof can be recognized by its proof type
                            // (first list entry).
                            // There are the following kinds of proofs:
                            // a: ["axiom"] (i=1)
                            // b: ["global", k, rule]
                            //    rule has been applied
                            //    to  the k-th equivalent expression as a whole
                            // c: ["subexpr", k, [i1, ..,im ]]
                            //    (where m = nops(EXPR[k]))
                            //    in the k-th equivalent exprssion to EXPR,
                            //    the j-th operand has been replaced by the
                            //    ij-th expression equivalent to it
                            // d: ["subsum", k, subexpr, rule]
                            //    apply rule to subexpr, where subexpr
                            //    is a subsum(or -product) of the k-th
                            //    equivalent expression to EXPR

     RESULTS: DOM_TABLE,     // the same, but with valuation attached to each
                            // expression
     simplifier,
     result,
     i,
     evalSubexpression,
     createSimplifier: DOM_PROC,
     NEXTVAL,
     SIMPLIFIERS: DOM_TABLE, // SIMPLIFIERS[EXPR]: simplification method for EXPR
     OPTS:        DOM_TABLE,
     maxSteps, steps, isSimple, stopIf, stop: DOM_BOOL, maxTime, TIME, failed, matches;

     option escape;

   begin
     // set default options by global default options and options given to top call
     // get given options
     OPTS:= prog::getOptions(2, [args()],  OPTIONS, TRUE, OPTIONT)[1];

     valuation:=   OPTS[hold(Valuation)];
     selectRules:= OPTS[hold(SelectRules)];
     ruleBase:=    OPTS[hold(RuleBase)];
     apply:=       OPTS[hold(ApplyRule)];
     strategy:=    OPTS[hold(Strategy)];
     maxTime:=     OPTS[hold(Seconds)];
     stopIf:=      OPTS[hold(StopIf)];
     isSimple:=    OPTS[hold(IsSimple)];
     if maxTime <> infinity then
       maxSteps:=  infinity
     else
       maxSteps:=  OPTS[hold(Steps)]
     end_if;
     exclude:=     OPTS[hold(Discard)];
     localStepPriority:= OPTS[hold(LocalStepPriority)];

     paireval:= Simplify::paireval;
     
     SFUNC:= (X, Y) -> op(X, 2) < op(Y, 2) or
             op(X, 2) = op(Y, 2) and sysorder(op(X, 1), op(Y, 1));

     if type(ruleBase) = DOM_DOMAIN then
       ruleBase:= [ruleBase]
     end_if;

     if OPTS[IgnoreAnalyticConstraints] then
       if strategy = "Default" then
          strategy:= "IgnoreAnalyticConstraints"
       end_if;
       ruleBase:= append(ruleBase, Simplify::IgnoreAnalyticConstraints)
     end_if;

     
     RESULTS:= expressions:= proofs:= NEXTVAL:= SIMPLIFIERS:= table();

     // local methods
     // queueinsert: inserts [TYPE, CONTENT] with priority VAL into the
     // queue QUEUE and return sthe new QUEUE
     // to avoid queue entries with equal priority, VAL is increased by a
     // small amount if necessary
     
     queueinsert:=
     proc(QUEUE, VAL, TYPE, CONTENT)
     begin
       assert(contains({DOM_FLOAT, DOM_RAT, DOM_INT}, domtype(VAL)));
       if iszero(VAL) then
         VAL:= 1e-9
       end_if;
       while contains(QUEUE, VAL) do
         VAL := VAL * 1.001
       end_while;
       QUEUE[VAL] := [TYPE, CONTENT];
       QUEUE
     end_proc;


     getQueueMinimalVal:=
     proc(QUEUE)
     begin
       if nops(QUEUE) > 1 then
         stdlib::min(map(op(QUEUE), op, 1))
       elif nops(QUEUE) = 0 then
         1e10
       else
         op(QUEUE, [1, 1])
       end_if
     end_proc;
     
    //
    // proofout(out, EXPR) - use the table proof to generate
    //                       a proof for out from EXPR
    // out is the result, EXPR the simplified input expression

    proofout:=
    proc(out, EXPR)
      name Simplify::proofout;
      local i, j, proofout_recursive, ruleprint, makeLemma;
    begin
      i:= contains(expressions[EXPR], out);
      assert(i>0);
      
      proofout_recursive:=
      proc(i, EXPR)
        local laststep;
        save PRETTYPRINT;
      begin
        PRETTYPRINT:= FALSE;
        
        laststep:= proofs[EXPR][i];

        case laststep[1]
          of "axiom" do
            assert(i=1);
            assert(expressions[EXPR][1] = EXPR);
            "Input was \n  ".expr2text(EXPR)." \n";
            break;
          of "global" do
            proofout_recursive(laststep[2], EXPR).
            "Applying the rule\n  ".ruleprint(laststep[3])."\ngives \n  ".
            expr2text(expressions[EXPR][i])." \n";
            break;
          of "subexpr" do
            proofout_recursive(laststep[2], EXPR).
            _concat(makeLemma(laststep[3][j],
                                                 op(expressions[EXPR]
                                                    [laststep[2]], j))
                    $j=1..nops(laststep[3])).
            "\nsubstituting gives \n  ".expr2text(expressions[EXPR][i])." \n";
            break
          of "subsum" do
            proofout_recursive(laststep[2], EXPR).
            "Applying rule\n  ".ruleprint(laststep[4])."\nto subexpression\n  ".
            expr2text(laststep[3])."\ngives \n  ".
            expr2text(expressions[EXPR][i])." \n";
            break;
         end_case;
      end_proc;

      
      ruleprint:= 
      proc(R: Rule)
        local procout;
      begin
        procout:=
        proc(f)
        begin
          if type(f) = DOM_FUNC_ENV then 
            if f::print <> FAIL then 
              return(f::print)
            else
              f:= op(f, 1)
            end_if
          end_if;
          if type(f) <> DOM_PROC then
            expr2text(f)
          elif op(f, 6) <> NIL then    
            expr2text(op(f, 6))
          else 
            expr2text(f)
          end_if   
        end_proc;
        
        if Rule::result(R) <> FAIL then
          expr2text(Rule::pattern(R))." -> ".expr2text(Rule::result(R))
        else   
          procout(op(R, 1))
        end_if;  
      end_proc;
      
      
      // creates a lemma of the form:
      // subexpression is equivalent to expressions[subexpression][newindex]
      makeLemma:=
      proc(newindex, subexpression)
      begin
        if newindex = 1 then
          // no lemma is necessary for the trivial statement
          // that an expression is equal to itself
          return("")
        end_if;

        "Lemma: \n".
        expr2text(subexpression = expressions[subexpression][newindex]).
        "\n".
        proofout_recursive(newindex, subexpression).
        "End of lemma \n"
      end_proc;

      stdlib::Exposed(proofout_recursive(i, EXPR)."END OF PROOF \n")

    end_proc;



    // local method evalSubexpression
    // evalSubexpression(ex, i, totalval)
    //
    // determine the priority of doing the first simplification step
    // on the i-th subexpression of ex
    // totalval = valuation(ex)


    evalSubexpression:=
      proc(ex, i, totalval)
        local valuat;
      begin
        valuat:= valuation(op(ex, i));
        // upper and lower bound
        if valuat < 0.1*totalval then
           valuat:= 0.1*totalval
        elif valuat > totalval then
           // should not happen
           valuat:= totalval
        end_if;
        totalval - (1 - ExpectedSimplificationPerStep) * valuat;
      end_proc;



    ///////////////////////////////////////////////////////
    // internal utility function
    ///////////////////////////////////////////////////////
    // createSimplifier(EXPR)
    //
    // returns a procedure that
    //
    // appends result to RESULTS[EXPR]
    // sets              NEXTVAL[EXPR] to next valuation in queue
    //
    // returns
    // - null() if some internal step has been carried out
    //          that has not produced any new equivalent result
    // - FAIL   if the internal queue is empty and no further
    //          equivalent results can be produced in the future
    // - TRUE   if a new equivalent result has been produced
    //

    createSimplifier:=
    proc(EXPR)
      name Simplify::createSimplifier;
      option escape;
      local QUEUE, // queue of tasks to do; see below
            MINVAL, // minimum complexity of expression equivalent to EXPR
            RET, RET1, //
            RULES, //
            SUBPTR: DOM_TABLE, // SUBPTR[expression] is a list of integers
                               // SUBPTR[expression][i] = j means:
                               // replacing the 1st to jth equivalent expression
                               // for the ith operand has already been tried
            SUBEXPRS, //
            currentval, beenthere;
    begin
      userinfo(20, "Creating simplifier for ".expr2text(EXPR));

      // initializations
      MINVAL:= valuation(EXPR);

      RESULTS[EXPR]:= [[EXPR, MINVAL]];
      expressions[EXPR]:= [EXPR];
      proofs[EXPR]:= [["axiom"]];
      //QUEUE:= [[0, EXPR, ExpectedSimplificationPerStep*MINVAL]];
      /*%*/ QUEUE := table(ExpectedSimplificationPerStep*MINVAL = [0, EXPR]);
      //# QUEUE := adt::Heap();
      //# QUEUE::insert(ExpectedSimplificationPerStep*MINVAL, [0, EXPR]);

      beenthere:=[];

      // QUEUE is a table of tasks that we have to do
      // Types of tasks:
      // 0 - the queue element is an expression, and
      //     we have to find all applicable rules
      //
      // 1 - the queue element is a pair [expression, rule]
      //     we have to apply the rule to the expression
      //
      // 2 - the queue element is a list [i, expression]
      //     Let subexpression:= op(expression, i)
      //     we have to produce another expression equivalent
      //     to subexpression ---
      //     either by calling the simplifier or by getting the next
      //     unused result ---
      //     and replace the i-th operand by that expression
      //     and all other operands by expressions equivalent
      //     to them, in all possible ways
      //     SUBPTR[i] is the current position in RESULTS[expression]
      //
      // 3 - the queue element is a list [expression, rule, addargs]
      //     where addargs is a list
      //     we have to apply rule to the expression; if this succeeds
      //     and produces result,
      //     we obtain a new expression op(expression, 0)(result, op(addargs))




      /*************************************************************
        return value:
        procedure simplifyOneStep
        arguments:
        depth  - nonnegative integer: number of calls to simplifyOneStep that
                 lie below the current one in the dynamical execution stack.


       ************************************************************/
      proc(depth = 0: DOM_INT)
        name Simplify::simplifyOneStep;
        local ptr, L, xpr, ex, chooseSubsets: DOM_PROC,
        k, v, c, t, uinfo, strstep,
        operandsToSimplify: DOM_LIST,
        VAL, TYPE, CONTENT; // first element in queue, to do next
      begin
        if depth <= OPTIONS[hold(MaxInfoDepth)] then
          uinfo:= userinfo
        else
          uinfo:= null()
        end_if;

        if EXPR::dom::simplify = FAIL and
          EXPR::dom::Simplify = FAIL and
        not contains({DOM_EXPR, DOM_SET, DOM_LIST}, domtype(EXPR))
        then
          uinfo(Text, 10, EXPR, " is not an expression, do not simplify");
          return(FAIL)
        end_if;

        if domtype(EXPR) = DOM_EXPR and
          type((ex:= eval(op(EXPR, 0)))) = DOM_FUNC_ENV and
          ex::dontSimplify = TRUE
          then
          uinfo(Text, 10, EXPR, " must not be simplified");
          return(FAIL)
        end_if;

        
        if nops(QUEUE) = 0 then
          uinfo(Text, 10, "Queue for ", EXPR, " is empty");
          return(FAIL)
        end_if;

        VAL:= getQueueMinimalVal(QUEUE);
        [TYPE, CONTENT]:= QUEUE[VAL];
        delete QUEUE[VAL]; // remove "minimal" element
       
        if OPTS[hold(ShowSteps)] = TRUE then
          if ( depth>0 ) then
            strstep := "";
          else
            strstep := strprint( NoNL, steps ).":";
          end_if:
          while length(strstep)<4 do strstep := " ".strstep; end_while;
          if TYPE in {0,1} then
            fprint(Unquoted, 0, "Memory: ", (bytes()[1] div 10000)/100.0, " MB    Time: ", time());
            fprint(Unquoted, 0, strstep, " ", " " $ i = 1..depth, TYPE, ", ", CONTENT, " (", VAL, ")")
          end_if;
        end_if;

        if exclude(VAL, MINVAL, depth) then
          // all elements with higher valuation
          // may also be discarded
          return(FAIL)
        end_if;

        if depth > 0 then
          uinfo(Text, 5, "Depth is ", depth)
        end_if;
        uinfo(Text, 10, "Working on rule ", [TYPE, CONTENT], " ", [VAL]);
        uinfo(NoNL, 5, Simplify::queueprint(QUEUE), "");

        if TYPE = 1 then
          uinfo(Text, 5, "Queue element is: ", VAL = [TYPE, Rule::print(CONTENT[1]), CONTENT[2]]);
        else
          uinfo(Text, 5, "Queue element is: ", VAL = [TYPE, CONTENT]);
        end_if;

        case TYPE

          of 0 do
            // TYPE 0:
            // we have to
            // (a) generate all rules for CONTENT; the resulting
            // tasks are stored in RULES
            // (b) create the task to simplify every operands of CONTENT
            // these are stored in SUBEXPRS

            
            beenthere:= append(beenthere, CONTENT);

            if isSimple(CONTENT, depth) then
              // nothing to do
              return(FAIL)
            end_if;
            
            // select rule base
            RULES:= selectRules(ruleBase, CONTENT, bool(depth = 0), strategy);

            assert(testtype(RULES, Type::ListOf(Rule)));

            // since we have to apply many rules, use a remember
            // mechanism for combinat::choose

            chooseSubsets:=
            proc(k)
              option remember;
            begin
              {op(combinat::choose({op(CONTENT)}, k))}
            end_proc;

            RULES:= map(RULES,
                        proc(rule)
                          local k, n, oper;
                        begin
                          if Rule::result(rule) <> FAIL and // procedures not!
                             (k:= nops(Rule::pattern(rule))) < (n:= nops(CONTENT)) and
                            contains(Simplify::comm intersect Simplify::ass,
                                     op(Rule::pattern(rule), 0)) then
                            // partial sums/products/.. that have
                            // exactly as many operands as the pattern
                            if k > n then
                              // do nothing
                              return(null())
                            else
                              oper:= op(CONTENT, 0);
                              // form all k-element subsets of the set of
                              // operands of the expression
                              // for each such subset S, generate the
                              // task to simplify the partial expression
                              // consisting of the elements of S
                              // note that we assume that no operand can
                              // appear twice in the expression!

                              op(map
                                 (
                                  chooseSubsets(k),
                                  proc(S)
                                  begin
                                    [paireval(Rule::priority
                                              (rule, strategy, valuation),
                                              VAL),
                                     3, [rule, oper(op(S)),
                                         [op({op(CONTENT)} minus S) ]]
                                     ]
                                  end_proc
                                  ))
                            end_if
                          else
                            [paireval(Rule::priority(rule, strategy, valuation),
                                      VAL),
                             1,
                             [rule, CONTENT]]

                          end_if
                        end_proc);

            if contains({DOM_EXPR, DOM_SET, DOM_LIST}, domtype(CONTENT))
              or CONTENT::dom::operandsToSimplify <> FAIL then

              // determine those indices i for which we want to recurse
              // on op(CONTENT, i)
              if (v:= CONTENT::dom::operandsToSimplify(CONTENT)) <> FAIL
                or
                domtype(CONTENT) = DOM_EXPR and
                domtype((t:= eval(op(CONTENT, 0)))) = DOM_FUNC_ENV and
                (v:= t::operandsToSimplify) <> FAIL then
                operandsToSimplify:= v
              else
                operandsToSimplify:= [$1..nops(CONTENT)]
              end_if;

             // assert(type(operandsToSimplify) = DOM_LIST); 
              
              SUBEXPRS:=
              map
              (operandsToSimplify,
               proc(i: DOM_INT)
                 local EXPR;
               begin
                 EXPR := op(CONTENT, i);
                 if ( EXPR::dom::simplify = FAIL and
                     not contains({DOM_EXPR, DOM_SET, DOM_LIST},
                                  domtype(EXPR) ) ) then
                   return( null() );
                 end_if;

               [evalSubexpression(CONTENT, i, VAL),
                2,
                [i, CONTENT]]
               end_proc);

              // SUBPTR[i] = j means: the 1st to jth equivalent expression
              // to op(QUELEM[2], i) have already been used
              SUBPTR[CONTENT]:= [1 $ nops(CONTENT)];

            else

              SUBEXPRS:= []

            end_if;

            uinfo(Text, 10, "Appending ", nops(RULES),
                  " rule-application tasks and ", nops(SUBEXPRS),
                  " subexpression-tasks");


            for k in RULES do
              QUEUE:= queueinsert(QUEUE, op(k))
            end_for;
            for k in SUBEXPRS do
              QUEUE:= queueinsert(QUEUE, op(k))
            end_for;


            //if nops(QUEUE) > 0 then
            //  NEXTVAL[EXPR] := QUEUE[1][3];
            //else
            //  NEXTVAL[EXPR] := infinity
            //end_if;
            VAL:= getQueueMinimalVal(QUEUE);

            NEXTVAL[EXPR] := VAL;

            if OPTS[hold(ShowSteps)] = TRUE then
              strstep := "";
              while length(strstep)<4 do
                strstep := " ".strstep;
              end_while;
              fprint( Unquoted, 0, " " $ i = 1..depth+5, "+ ",
                     nops(RULES)+nops(SUBEXPRS), " new rules" );
            end_if;

            // only internal step
            uinfo(Text, 10, "Internal step finished for ", EXPR);
            return()

          of 1 do
            // TYPE 1
            // apply the rule CONTENT[1] to the expression CONTENT[2]
            
            if traperror( ( RET1:= apply(op(CONTENT)) ) ,
                         MaxSteps = OPTS[KernelSteps](CONTENT[1], CONTENT[2]))
              <>0  then
              RET1 := FAIL
            end_if;


            if RET1 <> FAIL then
              RET := Simplify::normalize( RET1 )
            else
              RET:= FAIL
            end_if;

            if RET <> FAIL and contains(expressions[EXPR], RET) = 0 then

              matches:= matches + 1;
              expressions[EXPR]:= append(expressions[EXPR], RET);
              assert(contains(expressions[EXPR], CONTENT[2]) > 0);
              if OPTS[OutputType] = "Proof" then 
                proofs[EXPR] := append(proofs[EXPR],
                                     ["global",
                                      contains(expressions[EXPR], CONTENT[2]),
                                      CONTENT[1]
                                      ]
                                    ):
              end_if;
              currentval:= valuation(RET);


              if currentval < MINVAL then
                MINVAL:= currentval
              end_if;

              RESULTS[EXPR]:= append(RESULTS[EXPR], [RET, currentval]);



              // insert the non-normalized result, too, if it is different
              // however, do not attempt further simplifications of the
              // non-normalized result
              if RET <> RET1 and contains(expressions[EXPR], RET1) = 0 then
                expressions[EXPR]:= append(expressions[EXPR], RET1);
                if OPTS[OutputType] = "Proof" then 
                  proofs[EXPR] := append(proofs[EXPR],
                                       ["global",
                                        contains(expressions[EXPR], CONTENT[2]),
                                        CONTENT[1]
                                        ]
                                       )
                end_if;
                RESULTS[EXPR]:= append(RESULTS[EXPR], [RET1, valuation(RET1)]);
              end_if;


              if not exclude(currentval, MINVAL, depth) then
                // QUEUE insert
                if exclude = FALSE or not isSimple( RET, depth ) then
                  QUEUE:= queueinsert(QUEUE, currentval, 0, RET)
                else
                  QUEUE := table();
                end_if;
              end_if;

              if nops(QUEUE) > 0 then
                VAL:= getQueueMinimalVal(QUEUE);
                NEXTVAL[EXPR] := VAL
              else
                NEXTVAL[EXPR] := 1e10
              end_if;

              assert(RET <> FAIL);
              uinfo(Text, 10, "Obtained equivalent expression ", RET,
                       " for ", EXPR);
              return(TRUE)
            elif  RET <> RET1 and contains(expressions[EXPR], RET1) = 0 then
              // RET was known, but the non-normalized RET1 is new
              // insert RET1 into the list of results
              // but do not create the task
              // to simplify it further
              matches:= matches + 1;
              expressions[EXPR]:= append(expressions[EXPR], RET1);
              if OPTS[OutputType] = "Proof" then 
                proofs[EXPR] := append(proofs[EXPR],
                                     ["global",
                                      contains(expressions[EXPR], CONTENT[2]),
                                      CONTENT[1]
                                      ]
                                       )
              end_if;
              RESULTS[EXPR]:= append(RESULTS[EXPR], [RET1, valuation(RET1)]);

              if nops(QUEUE) > 0 then
                VAL:= getQueueMinimalVal(QUEUE);
                NEXTVAL[EXPR] := VAL
              else
                NEXTVAL[EXPR] := 1e10
              end_if;
              uinfo(10, "Rule produced new result ".expr2text(RET1).
                    " the normalized form of which had already been known");
              return()
            else

              if nops(QUEUE) > 0 then
                VAL:= getQueueMinimalVal(QUEUE);
                NEXTVAL[EXPR] := VAL
              else
                NEXTVAL[EXPR] := 1e10
              end_if;

              failed:= failed + 1;
              uinfo(10,
                       if RET = FAIL then
                           "Rule did not match"
                       else
                           "Rule produced already known result"
                       end_if
                       );
              return()
            end_if;

                  // NOT REACHED
            assert(FALSE)

          of 2 do
            // TYPE 2
            // do one simplification step on a subexpression
            ptr:= CONTENT[1];
            xpr:= CONTENT[2];
            ex:= op(xpr, ptr);


            if not contains(SIMPLIFIERS, ex) then
              SIMPLIFIERS[ex]:= createSimplifier(ex)
            end_if;

            if SUBPTR[CONTENT[2]][ptr] >= nops(RESULTS[ex]) then
              // get another expression equivalent to ex


              RET:=SIMPLIFIERS[ex](depth + 1);

              // if RET=FAIL, do nothing: do not call
              // the simplifier again
              if RET = null() then
                // nothing has been produced; call the
                // simplifier again later

                assert(type(CONTENT[1]) = DOM_INT);

                v :=
                localStepPriority(valuation(xpr), valuation(ex), NEXTVAL[ex],
                                  depth);

               
                
                QUEUE:= queueinsert(QUEUE, v, 2, CONTENT);


              elif RET <> FAIL then
                assert(RET = TRUE);
                SUBPTR[CONTENT[2]][ptr]:= SUBPTR[CONTENT[2]][ptr] + 1;
              end_if
            else
              // use known result
              SUBPTR[CONTENT[2]][ptr] := SUBPTR[CONTENT[2]][ptr] + 1;

              RET:= TRUE; // dummy value
            end_if;

            if RET <> FAIL and RET <> null() then
              assert(RET = TRUE);
              // we have to replace each subexpression by
              // an equivalent expression, in all possible ways
              // therefore, we create a set L of lists
              // [n1, ..., nk] denoting the task to replace, for each i,
              // the i-th operand by the n_i-th expression equivalent to
              // it
              // ptr is number of the operand for which we just
              // created another equivalent expression, we always
              // use that newly created expression

              if OPTS[hold(AllOperands)] then 
                L:= combinat::cartesianProduct
                ({$1..SUBPTR[CONTENT[2]][i]} $i=1..ptr-1,
                 {SUBPTR[CONTENT[2]][ptr]},
                 {$1..SUBPTR[CONTENT[2]][i]} $i=ptr+1..nops(CONTENT[2]))
              else 
                L:= {[1 $ptr-1, SUBPTR[CONTENT[2]][ptr], 1 $(nops(CONTENT[2])-ptr)]}
              end_if;
              
              L:= map
                  (L,
                   proc(list)
                     local result, valuat, indexlist;

                   begin
                     indexlist:= select([$1..nops(list)],
                                        i -> contains(expressions,
                                                      op(xpr, i)));
                     if domtype(xpr) = DOM_SET then
                       // avoid subsop for sets !!
                       result:={op(subsop
                                  ([op(xpr)],
                                   i = expressions[op(xpr, i)][list[i]]
                                   $i in indexlist))}
                     else
                       result:= subs(xpr,
                                     [op(xpr, i) = expressions[op(xpr, i)][list[i]]
                                           $i in indexlist], 
                                     EvalChanges
                                    )
                     end_if;
                     result:= Simplify::normalize(result);
                     assert(contains(expressions, EXPR));
                     assert(contains(RESULTS, EXPR));
                     if contains(expressions[EXPR], result) = 0 then
                       expressions[EXPR]:= append(expressions[EXPR],
                                                  result);
                       if OPTS[OutputType] = "Proof" then 
                         proofs[EXPR]:= append(proofs[EXPR],
                                             ["subexpr",
                                              contains(expressions[EXPR],
                                                       CONTENT[2]),
                                              list
                                              ]);
                       end_if;
                       valuat:= valuation(result);
                       RESULTS[EXPR]:= append(RESULTS[EXPR],
                                              [result, valuat]);

                       [ExpectedSimplificationPerStep * valuat, 0, result] //
                     else
                       null()
                     end_if
                   end_proc
                   );


              // now L contains everything we want to do
              // with the newly generated expressions

              // we also want to try simplifying
              // the i-th subexpression another time


              assert(type(CONTENT[1]) = DOM_INT);

              v :=
              localStepPriority(valuation(xpr), valuation(ex), NEXTVAL[ex],
                                depth);

              QUEUE:= queueinsert(QUEUE, v, 2, CONTENT);

              uinfo(Text, 10, "Obtained ", nops(L),
                    "new results in subexpression step");
              // insert all new ... from L
              for k in L do
                [v, t, c] := k;
                QUEUE:= queueinsert(QUEUE, v, t, c)
              end_for;

             
              if nops(L) = 0 then
                RET:= null()
              else
                assert(RET=TRUE);
              end_if;
            end_if;

            if nops(QUEUE) > 0 then
              VAL:= getQueueMinimalVal(QUEUE);
              NEXTVAL[EXPR] := VAL
            else
              NEXTVAL[EXPR] := 1e10
            end_if;

            if RET= FAIL then
              return()
            end_if;

            uinfo(Text, 10, "Finished subexpression-step for ", EXPR);
            return(RET)

          of 3 do
            // TYPE 3
            // do a simplification step on a subsum/subproduct
            
            RET:= apply(CONTENT[1], CONTENT[2]);

            if RET <> FAIL and
              // add the additional operands
              (RET:= eval(op(CONTENT[2], 0)(RET, op(CONTENT[3])));
               contains(expressions[EXPR], RET) = 0) then

              matches:= matches + 1;

              expressions[EXPR]:= append(expressions[EXPR], RET);
               
              if OPTS[OutputType] = "Proof" then 
                k:= contains(expressions[EXPR],
                           op(CONTENT[2], 0)
                           (op(CONTENT[2]), op(CONTENT[3]))
                          );
                if k=0 then
                    k:= contains(expressions[EXPR],
                           eval(op(CONTENT[2], 0)(op(CONTENT[2]),
                                                  op(CONTENT[3]))))
                end_if;
                assert(k>0);       

                proofs[EXPR]:=append(proofs[EXPR],
                                   ["subsum",
                                    k,
                                    CONTENT[2],
                                    CONTENT[1]
                                    ]
                                   );
              end_if;
              currentval:= valuation(RET);



              if currentval < MINVAL then
                MINVAL:= currentval
              end_if;

              RESULTS[EXPR]:= RESULTS[EXPR]. [[RET, currentval]];

              if not exclude(currentval, MINVAL, depth) then
                QUEUE:= queueinsert(QUEUE, currentval, 0, RET)
              end_if;

              VAL:= getQueueMinimalVal(QUEUE);
              NEXTVAL[EXPR] := VAL;

              assert(RET <> FAIL);
              uinfo(Text, 10, "Obtained equivalent expression ", RET,
                       " for ", EXPR);
              return(TRUE)

            else
              if nops(QUEUE) > 0 then
                VAL:= getQueueMinimalVal(QUEUE);
                NEXTVAL[EXPR] := VAL;
              else
                NEXTVAL[EXPR] := 1e10
              end_if;

              failed:= failed + 1;
              uinfo(10, "Rule did not match");
              return()

            end_if;

        end_case;
      end_proc    // returned procedure simplifyOneStep

    end_proc;   // createSimplifier

    ////////////////////////////////////////////////////////////////////////
    // main program
                                                       
    steps:= 0;
    if maxTime <> infinity then
      TIME:= time();
      maxTime:= 1000*maxTime + TIME
    else
      TIME:= 0
    end_if;
    failed:= 0;
    matches := 0;

    simplifier:= createSimplifier(EXPR);
    exprnops:= 1; // the first operand of expressions[EXPR], namely
                  // EXPR itself, is known                                     

                                                       
    SIMPLIFIERS[EXPR]:= simplifier;

    if stopIf(EXPR) then
      // stop at once
      stop:= TRUE;
      OPTS[hold(Goal)]:= EXPR
    else
      stop:= FALSE
    end_if;
                                                       
    while not stop and steps < maxSteps and (TIME:= time()) < maxTime do
      steps:= steps + 1;
      userinfo(3, "**************************************************\nStarting step ".expr2text(steps));
      if maxTime = infinity then
        result:= simplifier()
      else
        case
            traperror((result:= simplifier()), ceil((maxTime - TIME)/1000))
          of 0 do
            break;
          of 1320 do
            userinfo(1, "Step ".expr2text(steps)." interrupted (timeout)");
            result:= null(); // nothing has been produced
            break;
          otherwise
            lasterror()
        end_case;
      end_if;
      if result = FAIL then
        userinfo(2, "Simplification finished because of empty queue after ".
                    expr2text(steps)." steps");
        break
        // elif result <> null() then

      end_if;

      if result = TRUE then
        // obtained new equivalent expression
        
        for i from exprnops+1 to nops(expressions[EXPR]) do
          if expressions[EXPR][i] = OPTS[hold(Goal)] then
            userinfo(1, "Goal reached after ".expr2text(steps)." steps");
            stop:= TRUE;
            break
          elif stopIf(expressions[EXPR][i]) then
            userinfo(1, "Stopping criterion met after ".expr2text(steps).
                     " steps");
            OPTS[hold(Goal)]:= expressions[EXPR][i];
            stop:= TRUE;
            break
          end_if
        end_for;

        // adjust pointer
        exprnops:= nops(expressions[EXPR]);
      end_if;
    end_while;


    userinfo(5, "Failed :".expr2text(failed)."; Matches: ".expr2text(matches));


    if OPTS[OutputType] = "NumberOfSteps" then
      return(steps)
    end_if;

    result := sort(RESULTS[EXPR], SFUNC);


    if (finalValuation:= OPTS[hold(FinalValuation)]) <>
      Simplify::defaultFinalValuation then

      result:= map(result, l -> [l[1], finalValuation(op(l))]);
      result := sort(result, SFUNC)
    end_if;


    if (i:= contains(map(result, op, 1), OPTS[hold(Goal)])) > 0 then
      result:= [result[i], op(result, 1..i-1), op(result, i+1..nops(result))]
    end_if;




    if OPTS[hold(All)] = TRUE then
      return(map(result, op, 1))
    elif OPTS[hold(AllValuation)] = TRUE then
      return(result)
    elif OPTS[OutputType] = "Best" then // default
      return(op(result, [1, 1]))
    elif OPTS[OutputType] = "Proof" then
      return(proofout(op(result, [1, 1]), EXPR))
    else
      error("Unknown output type")
    end_if
  end_proc;

  // call of REM to use prog::remember depending on options
  if OPTIONS[hold(Remember)] = TRUE then
    PROC := prog::remember(PROC,
                             property::depends,
                             hold(PreventRecursion),
                             x->x)
     // dont catch errors !!
     // Because e.g. if in Simplify(1/X), X simplifies to zero, we
     // want the error to be thrown over and over again
  end_if;

  ///////////////////////////////////////////////////////
  //  end of created  S I M P L I F I E R  //////////////
  ///////////////////////////////////////////////////////
  
  return(PROC);

end_proc:

//
Simplify::queueprint:=
proc(queue)
  local qulist, j;
begin
  qulist:= sort([op(queue)],(X,Y) -> op(X, 1) < op(Y, 1) );
  fprint(Unquoted, 0, "\n***** QUEUE:");
  for j from 1 to min(8, nops(qulist)) do 
  fprint(0, op(qulist, j))
  end_for;
  fprint(Unquoted, 0, "Total: ".expr2text(nops(qulist))." elements in queue");
  fprint(Unquoted, 0, "*****")
end_proc:

// must return a list of procedures/rules
Simplify::selectRules:=
proc(Ruleset : Type::Union(DOM_LIST, DOM_DOMAIN), expression, global = FALSE,
     strat = "")
  local ret, result: DOM_LIST, i, rulebase;
begin

  result:= [];
 
  if type(Ruleset) = DOM_DOMAIN then
    Ruleset:= [Ruleset]
  end_if;

  for i from 1 to nops(Ruleset) do
    rulebase:= Ruleset[i];
    assert(type(rulebase) = DOM_DOMAIN);

    result:=
    result.
    (if global = TRUE and contains(rulebase, "Global") then
       ret:= slot(rulebase, "Global");
       if domtype(ret) = DOM_LIST then
         ret
       else
         assert(domtype(ret) = DOM_PROC  or domtype(ret) = DOM_FUNC_ENV);
         ret(expression)
       end_if
     elif not global and contains(rulebase, "Local") then
     ret:= slot(rulebase, "Local");
     if domtype(ret) = DOM_LIST then
       ret
     else
       assert(domtype(ret) = DOM_PROC or domtype(ret) = DOM_FUNC_ENV);
       ret(expression)
     end_if
   end_if)
    .
    (if contains(rulebase, type(expression)) then
       ret:= slot(rulebase, type(expression));
       if domtype(ret) = DOM_LIST then
         ret
       else
         assert(domtype(ret) = DOM_PROC  or domtype(ret) = DOM_FUNC_ENV);
         ret(expression)
       end_if
     else
     []
     end_if)
    .
    (if contains(rulebase, "All") then
       ret:= slot(rulebase, "All");
       if domtype(ret) = DOM_LIST then
         ret
       else
         assert(domtype(ret) = DOM_PROC  or domtype(ret) = DOM_FUNC_ENV);
         ret(expression)
       end_if
     end_if)

  end_for;
  result
end_proc:

Simplify::defaultValuation:=
proc(EXPR)
begin
  float(Simplify::indetComplexity(EXPR) + Simplify::complexity(EXPR)/2
        + Simplify::pairPenalty(EXPR) )
end_proc:

Simplify::defaultFinalValuation:=
proc(EXPR, valuation)
begin
   valuation
end_proc:


/*
    Simplify::localStepPriority(v, sub, nextsub, depth)

    determines the priority of applying another simplification step to
    a subexpression if the expression has valuation v, the
    subexpression has valuation sub, the priority of the next element in
    the local queue of the subexpression is nextsub, and the depth of
    the subexpression in the global expression is depth 
*/
Simplify::localStepPriority:=
proc(v: Type::Real, sub: Type::Real, nextsub: Type::Real, depth:
     DOM_INT): DOM_FLOAT
begin
  if iszero(v) then
    1.0
  else
    max(0.1*v,
        1.2*(1+1/(2*depth+1))*float(v - sub + nextsub),
        1.2*(1+1/(2*depth+1))*float(v + (nextsub - sub)*(sub/v)^(1/2))
        )
  end_if
end_proc:


Simplify::paireval:=
proc(rulePriority, valuation): DOM_FLOAT
begin
  float(rulePriority * valuation)
end_proc:

// checks if an expression is simple enough to do no more simplification-steps
Simplify::isSimple:=
proc( EXPR, depth )
begin
  case type(EXPR)
  // a boolean expression is never simple since 0 < 0 should become FALSE
  // and x < 0 might become TRUE or FALSE depending on the properties of x
    of "_less" do
    of  "_leequal" do
    of "_equal" do
    of  "_unequal" do
      return(FALSE)
    of DOM_IDENT do
    of DOM_INT do
    of DOM_RAT do
    of DOM_COMPLEX do
    of DOM_BOOL do
      return(TRUE)
    of "_plus" do
    of "_mult" do
      if map({op(EXPR)}, type) subset
      {DOM_IDENT, DOM_COMPLEX, DOM_INT, DOM_RAT } then
        return(TRUE)
      end_if;
      break
    of "_power" do
      if type(op(EXPR, 1)) = DOM_IDENT and testtype(op(EXPR, 2), Type::Numeric)
        then
        return(TRUE)
      end_if;
      break
  end_case;

  FALSE
end_proc:

// simplify::exclude(currentval, MINVAL)
// currentval: complexity of current expression
// MINVAL: minimum complexity among equivalent expressions
// depth: depth in expression tree

Simplify::exclude:=
proc(currentval, MINVAL, depth=0)
begin
  currentval >= 10 + MINVAL * (2 + 12 * 1.5^(-depth))
end_proc:


// operators regarded as associative and commutative
Simplify::ass:= {hold(_mult), hold(_plus), hold(_and), hold(_or),
                     hold(_union), hold(_intersect)}:

Simplify::comm:= {hold(_mult), hold(_plus), hold(_and), hold(_or),
                     hold(_union), hold(_intersect)}:

Simplify::normalize :=
proc( EXPR )
  local res, normalizeTrigonometric, inds;
begin

  if contains({"_less", "_leequal", "_equal", "_unequal"}, type(EXPR)) then
    if domtype(op(EXPR, 1)) = piecewise or domtype(op(EXPR, 2)) = piecewise
      then
      return(slot(piecewise, type(EXPR))(op(EXPR)))
    end_if
  end_if;

  inds:= indets(EXPR, All);

  if contains(inds, hold(log)) then
    EXPR:= misc::maprec(EXPR, {"_power"} = proc(pow) 
                                           begin 
                                            if type(op(pow, 1)) = DOM_INT and hastype(op(pow, 2), "log") then
                                              Simplify::intPowHasLog(pow)
                                            else 
                                              pow 
                                            end_if
                                           end_proc)
  end_if;

  if inds intersect
    { hold(sin), hold(cos), hold(tan), hold(cot) } = {} then
    return( EXPR )
  end_if;

  normalizeTrigonometric :=
  proc( LIST )
    local i, t1, t2, expr, cosc, sinc;
  begin
    expr := op( LIST, [1,1,1] );
    t1 := split( LIST, X -> ( ( contains( { "sin", "cos", "tan", "cot" }, type( op(X,1) ) ) and op(X,[1,1])=expr ) ) );
    sinc := cosc := 0;
    for i in op(t1,1) do
      case type( op(i,1) )
        of "sin" do sinc := sinc + op(i,2); break;
        of "cos" do cosc := cosc + op(i,2); break;
        of "tan" do sinc := sinc + op(i,2); cosc := cosc - op(i,2); break;
        of "cot" do sinc := sinc - op(i,2); cosc := cosc + op(i,2); break;
      end_case;
    end_for;
    
    t2 := 1;
    if sign( sinc*cosc )=-1 then
      if sign( sinc ) = 1 then
        i := min( sinc, -cosc );
        t2 := t2 * _power( tan(expr), i );
        sinc := sinc - i;
        cosc := cosc + i;
      else
        i := min( -sinc, cosc );
        t2 := t2 * _power( cot(expr), i );
        sinc := sinc + i;
        cosc := cosc - i;
      end;
    end_if;
    
    t2 := t2 * _power( sin(expr), sinc );
    t2 := t2 * _power( cos(expr), cosc );

    return( [ [ t2, 1 ], op(op(t1,2)) ] );
  end;

  case type( EXPR )
    of "_mult" do
      res := null();
      EXPR := map([op(EXPR)], X-> if testtype(X,"_power") and testtype( op(X,2), Type::Integer) then [op(X,1),op(X,2)] else [X,1] end_if);
      while nops(EXPR)>0 do
        if contains( { "sin", "cos", "tan", "cot" }, type( op(EXPR, [1,1] ) ) ) then
          EXPR := normalizeTrigonometric( EXPR );
        end;
        res := res * op(EXPR, [1,1])^op(EXPR, [1,2]);
        delete EXPR[1]
      end_while;
      return(res)
    of "_plus" do
      return( _plus( map( op(EXPR), Simplify::normalize) ) );
    of "_power" do
      return(Simplify::normalize(op(EXPR,1))^op(EXPR,2))
  end_case;
  return( EXPR )
end_proc:
