/*--
        solvelib::isolate  --  isolate an identifier to the lhs
--*/


solvelib::isolate :=
proc(lhside, Rhs, x, options: DOM_TABLE, reclevel = 1 : DOM_INT, known={}):
  Type::Set
  local Lhs,
  maxreclevel: DOM_INT,
  l, i, j, a, b, s,
  ar : DOM_IDENT,
  lev,
  vars,
  newvars,
  newlhs,
  newrhs,
  solutions,
  pw,
  u : DOM_IDENT,
  v : DOM_IDENT,
  p, q,
  roots: DOM_BOOL,
  getfractpow : DOM_PROC,
  fractionalPowerHeuristicMult: DOM_PROC, 
  fractionalPowerHeuristicPlus: DOM_PROC, 
  applyHeuristic: DOM_PROC,
  sinify: DOM_PROC,
  cosify: DOM_PROC,
  combinePower: DOM_PROC,
  combineLog: DOM_PROC,
  isolatePowerLog: DOM_PROC,
  checkSols:DOM_PROC,
  tryRationalize: DOM_PROC,
  R2toC: DOM_PROC,
  n, k, opt,
  Ln: DOM_PROC,
  Arcsin: DOM_PROC,
  Arccos: DOM_PROC,
  Arctan: DOM_PROC;

  save MAXEFFORT;


begin


// l o c a l   m e t h o d s



// local method getfractpow
// checks whether the expresion a is a fractional power or a product
// in which exactly one fractional power occurs

// if no, FAIL is returned
// if yes, the fractional exponent is returned, 
// such that raising a to the denominator of it would remove the fractional power 
// the cofactor is returned, too
// negative powers are indicated by returning a negative exponent

  getfractpow :=
  proc(a)
    local
    frc,  // factors that are fractional powers
    other,  // other factors
    dummy, 
    result;
  begin
    case type(a)
      of "_power" do
        if type(op(a,2))=DOM_RAT then
          return([op(a, 2), 1])
        else
          return(FAIL)
        end_if;
      of "_mult" do
        [frc, other, dummy]:= split([op(a)],
                                    fac -> type(fac)="_power" and
                                    type(op(fac ,2)) = DOM_RAT and
                                    has(fac, x)
                                    );
        assert(dummy = []);
        if nops(frc) = 0 then
          return(FAIL)
        end_if;
        // return the lcm of the denominators (second op)
        // of the exponents (second op)
        // of the elements of frc
        result:= [lcm(op(map(frc, op, [2, 1]))) / lcm(op(map(frc, op, [2, 2]))), _mult(op(other))];
        return(result)
    end_case;
    FAIL
  end_proc;

  
  tryRationalize:=
  proc(L, R, x, options)
    local a, u, f, sol, types;
  begin
     // check whether we want to solve L=R by rationalizing 
      a:= rationalize(L-R, StopOn = (X -> not has(X, x)), FindRelations = ["_power"]);
      u:= op(a, 2);
      if nops(u) = 1 and not has(op(a, 1), x) then
        u:= op(u, 1);
        assert(type(u) = "_equal");
        sol:= solvelib::isolate(op(a, 1), 0, op(u, 1), options, reclevel+1, known);
        if has(sol, hold(solve)) then return(FAIL) end_if;
        return(solvelib::preImage(op(u, 2), x, sol, options))
      end_if;  

      // arcsin / arccos - heuristic
      a:= rationalize(L, StopOn = (X -> not has(X, x)), FindRelations = ["_power"]);
      u:= op(a, 2);
      if nops(u) > 0 and (types:= map(map(u, op, 2), type)) minus {"arcsin", "arccos", "arctan"} = {} then
        // check whether the input is polynomial in arcxyz, with small integer  coefficients
        f:= poly(op(a, 1), map([op(u)], op, 1), Dom::Integer);
        if f<>FAIL and degree(f) = 1 and max(map([coeff(f)], abs)) < 10 then
          if types = {"arctan"} then
            sol:= solvelib::isolate(subs(expand(tan(op(f, 1))), u, EvalChanges), tan(R), x, options, reclevel+1, known)
          else   
            sol:= solvelib::isolate(subs(expand(sin(op(f, 1))), u, EvalChanges), sin(R), x, options, reclevel+1, known)
          end_if;  
          if type(sol) = "solve" then
            return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
          else  
            return(solvelib::checkSolutions(sol, L= R, x, options))
          end_if;  
        end_if
      end_if;
      FAIL
  end_proc:
  
  
  /* 
  
    local method fractionalPowerHeuristicPlus
    find out whether exactly one summand is a fractional power
    or a product of fractional powers
    if yes, let the exponent be p/q; raise that summand and all others to the power of q, such
    that no fractional power occurs any more
    i.e. s1+s2+..+si^(p/a[i])+..+sn = Rhs becomes
    si^p = (Rhs -(s1+s2+..+s(i-1)+s(i+1)+...+sn))^a[i]
    note that this increases the number of solutions !!

*/

fractionalPowerHeuristicPlus:=
proc()
  local a, i, l, q, s, u, X, S, opt, solveFractionalEquation: DOM_PROC;
begin
   
   // solve L = R for x, where L is of the form C*f(x)^(q/s) and both sides may depend on x
   solveFractionalEquation:=
   proc(L, q, s, R, l)
     local a, u;
     save MAXEFFORT;
   begin
     
      a := L^abs(s) - R^abs(s);
      // from the solutions of
      // si^p = (Rhs -(s1+s2+..+s(i-1)+s(i+1)+...+sn))^a[i] ,
      // select those for which si has small polar angle

      MAXEFFORT:= MAXEFFORT/2;
      u:= solvelib::isolate(a, 0, x, options, reclevel+1, known);
      if not has(u, hold(solve)) then
        if options[IgnoreAnalyticConstraints] then
          return(u)
        end_if;

        if type(u) = DOM_SET and freeIndets(u) = {} then
          return(solvelib::checkSolutions(u, Lhs=Rhs, x, options))
        end_if;  
          
        u:= solvelib::checkAngle(u, l*R, x, q/s,
                                      options, maxreclevel-reclevel);

        if type(u) <> "solve" then
          return(u)
        end_if 
      end_if;
      FAIL     
   end_proc;
  
   a := map([op(Lhs)], getfractpow);
      if nops(select(a, _unequal, FAIL)) = 1 then
        // get the index of the uniquely determined non-FAIL entry
        i := contains(map(a, type), DOM_LIST);
        [s, l] := a[i]; // exponent, cofactor
        if type(s) <> DOM_RAT then
          return(FAIL)
        end_if;  
        [q, s]:= [op(s)]; // numerator and denominator of the exponent
        newrhs := - _plus(op(Lhs, 1..i-1), op(Lhs, i+1..nops(Lhs))) + Rhs;
        if abs(s) > 100 then
          return(FAIL)
        end_if;
        
        return(solveFractionalEquation(op(Lhs, i), q, s, newrhs, l))
       
      else  // second method: check whether the only irrational subexpression is a fractional power  
      
        u:= indets(Lhs, RatExpr) minus  {x};
        u:= select(u, has, x);
        if nops(u) = 1 and type((l:= op(u, 1))) = "_power" and type((q:= op(l, 2))) = DOM_RAT then 
          // solve for l
          X:= genident("l");
          a:= subs(Lhs, l = X);
          opt:= options;
          opt[IgnoreSpecialCases]:= TRUE;
          // we will check the solutions anyway
          S:= solvelib::isolate(a, Rhs, X, opt, reclevel+1, known);
          if type(S) = DOM_SET and nops(S) = 1 then 
            // our equation Lhs = Rhs is equivalent to l = op(S, 1); where both l and op(S, 1) depend on x
            [q, s]:= [op(q)]; 
            return(solveFractionalEquation(l, q, s, op(S, 1), 1))
          end_if;  
        end_if   
      end_if;
      FAIL 
end_proc;


  // handle products with fractional powers by taking the inverse
      // power of both sides:
      // f(x) * g(x)^(1/n) = C becomes
      // f(x)^n * g(x) = C^n
      // only those solutions x0 with C/f(x0) having small polar angle are valid

      fractionalPowerHeuristicMult:=
      proc()
        local a, s, l, q, u;
      begin
        a:= getfractpow(Lhs);
        if a <> FAIL then
          [s, l]:= a;
          if type(s) = DOM_RAT then
            [q, s]:= [op(s)]
          else
            return(FAIL)
          end_if;
          u:= solvelib::isolate(simplify(Lhs^abs(s)), simplify(Rhs^abs(s)), x, options,
          reclevel+1, known);
          
          if type(u) <> "solve" then
            if options[IgnoreAnalyticConstraints] then
              return(u)
            end_if;
            
            if type(u) = DOM_SET and freeIndets(u) = {} then
              return(solvelib::checkSolutions(u, Lhs=Rhs, x, options))
            end_if;
            
            u:= solvelib::checkAngle(u, Rhs/l, x, q/s,
            options, maxreclevel-reclevel);
              
            if not has(u, hold(solve)) then
              return(u)
            end_if 
          end_if;
        end_if;
        FAIL 
      end_proc;


/* applyHeuristic(Lhs, Rhs)
*
 * Solves Lhs(x) = Rhs(x) heuristically by applying the same function to
 * both sides of the equation, or returns FAIL if the heuristic did not
 * succeed.
 */
  applyHeuristic :=
  proc(Lhs, Rhs)
    local expa, expb, basa, basb, g, result, s, lhs_f, lhs_c, rhs_f, rhs_c,
    powerheuristic: DOM_PROC;
    
  begin
    
    powerheuristic:=
    proc(Lhs: "_power", Rhs)
      local equ;
    begin
      if options[IgnoreAnalyticConstraints] then
        if type(op(Lhs, 1)) = DOM_INT and op(Lhs, 1) > 0 then
          equ:= op(Lhs, 2) - simplify(log(op(Lhs, 1), Rhs))
        else
          if iszero(op(Lhs, 1)) then
            equ:= piecewise([op(Lhs, 2) = 0, Rhs - 1],
                            [op(Lhs, 2) > 0, Rhs]);
            return(solve(equ, x, options))
          end_if;
          equ:= op(Lhs, 2)*ln(op(Lhs, 1)) - ln(Rhs)
        end_if;
        result:= solvelib::isolate(equ, 0, x, options, reclevel+1, known);
          
      elif not options[Real] then
        if not has(op(Lhs, 1), x) then
            result:= piecewise([op(Lhs, 1) = 0, solve(Rhs - 1 = 0 and op(Lhs, 2) = 0, x, options) union
                                            solve(Rhs = 0 and op(Lhs, 2) > 0, x, options)],
                               [op(Lhs, 1) <> 0, solvelib::preImage(op(Lhs, 2)*ln(op(Lhs, 1)) - ln(Rhs), x, 2*I*PI*Z_, options)]
                              )
        else
            result:= solvelib::preImage(op(Lhs, 2)*ln(op(Lhs, 1)) - ln(Rhs), x, 2*I*PI*Z_, options)
        end_if
      else
        return(FAIL)
      end_if;
      if has(result, hold(solve)) then
        return(FAIL)
      else
        return(result)
      end_if
    end_proc;
  
    
    
    if type(Lhs) = "sin" then 
      if type(Rhs) = "cos" then
        Rhs:= sin(op(Rhs, 1) + PI/2)
      elif type(-Rhs) = "cos" then
        Rhs:= -sin(op(-Rhs, 1) + PI/2)
      end_if  
    elif type(Lhs) = "cos" then 
      if type(Rhs) = "sin" then
        Lhs:= sin(op(Lhs, 1) + PI/2)
      elif type(-Rhs) = "sin" then  
        // multiply both sides by -1, and exchange
        [Lhs, Rhs]:= [-Rhs, -Lhs];
        // now we are in the first case, second subcase: Lhs = sin(..), -Rhs = cos(..)
        Rhs:= hold(sin)(-op(-Rhs, 1) - PI/2)
      end_if;  
    end_if;  
    
    if type(Lhs) = type(Rhs) then
      case type(Lhs)
        of "_power" do
          basa:= op(Lhs, 1);
          expa := op(Lhs, 2);
          basb:= op(Rhs, 1);
          expb := op(Rhs, 2);
          if type(expa) = DOM_RAT and type(expb) = DOM_RAT and
            op(expa, 1) = 1 and op(expb, 1) = 1 and
            (g := igcd(op(expa, 2), op(expb, 2))) > 1
            then
            /* a^(1/2) = b^(1/2)  <=>  a = b
* and similar cases
           */
            result:= solvelib::isolate(Lhs^g-Rhs^g, 0, x,
                                       options, reclevel+1, known);
            if options[Real] and not options[IgnoreAnalyticConstraints] then
              return(solvelib::solve_intersect
                     (result,
                      solve(Lhs^g >= 0, x, options),
                      solve(Rhs^g >= 0, x, options)
                      ))
            else
              return(result)
            end_if
          end_if;
          if options[IgnoreAnalyticConstraints] and (has(expa, x) or has(expb, x)) then
            // take logarithms
            return(simplify(solvelib::isolate(expa*ln(basa) - expb*ln(basb), 0, x,
                                              options, reclevel+1, known),
                            IgnoreAnalyticConstraints
                            )
                   )
          end_if;
          break
        of "exp" do
          if options[Real] or options[IgnoreAnalyticConstraints] then
            /* real case:
             * exp(a) = exp(b)  <=>  a = b
             */
            return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                     options, reclevel+1, known));
          else
            /* general case:
             * exp(a) = exp(b)  <=>  a in b + 2*I*PI*Z_
             */
            return(solvelib::preImage(op(Lhs)-op(Rhs), x, 2*I*PI*Z_,
                                      options));
          end_if;
        of "cos" do
          if options[IgnoreAnalyticConstraints] then
            return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                     options, reclevel+1, known))
          end_if;
          return(_union(solvelib::preImage(op(Lhs)-op(Rhs), x, 2*PI*Z_,
                                           options),
                        solvelib::preImage(op(Lhs)+op(Rhs), x, 2*PI*Z_,
                                           options)));
        of "sin" do
          if options[IgnoreAnalyticConstraints] then
            return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                     options, reclevel+1, known))
          end_if;
          return(_union(solvelib::preImage(op(Lhs)-op(Rhs), x, 2*PI*Z_,
                                           options),
                        solvelib::preImage(op(Lhs)+op(Rhs), x, PI+2*PI*Z_,
                                           options)));
        of "tan" do
          if options[IgnoreAnalyticConstraints] then
            return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                     options, reclevel+1, known))
          end_if;
          return(solvelib::preImage(op(Lhs)-op(Rhs), x, PI*Z_,
                                    options));
        of "arccos" do
        of "arcsin" do
          /* injective functions where
           * f(a) = f(b)  <=>  a = b
           */
          return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                   options, reclevel+1, known))
        of "ln" do 
          return(solvelib::isolate(op(Lhs)-op(Rhs), 0, x,
                                   options, reclevel+1, known)
                 minus {0}
                )
        of "log" do
          // simple case: equal bases
          if op(Lhs, 1) = op(Rhs, 1) then
            return(solvelib::isolate(op(Lhs, 2)-op(Rhs, 2), 0, x,
                                   options, reclevel+1, known)
                   minus {0}
                  )
          end_if;
          break
      end_case;
    end_if;
  
  /* division heuristic
   */
    if (type(Lhs) = "_mult" or type(Rhs) = "_mult") and
      has(Lhs, x) and has(Rhs, x) and is(Rhs = 0) = FALSE
      then
      MAXEFFORT:= MAXEFFORT/2;
      result := solvelib::isolate(Lhs/Rhs, 1, x, options, reclevel+1, known);
      if not has(result, hold(solve)) then
        return(result)
      end_if;
    end_if;


    if type(Lhs) = "_power" and has(op(Lhs, 2), x) then
      result:= powerheuristic(Lhs, Rhs);
      if result <> FAIL then
        return(result)
      end_if
    elif type(Rhs) = "_power" and has(op(Rhs, 2), x) then
      result:= powerheuristic(Rhs, Lhs);
      if result <> FAIL then
        return(result)
      end_if
    end_if;
    

  
  /* additional heuristic for
   *   c1*exp(f1(x)) = c2*exp(f2(x))
   * (real case)
   */
    if options[Real] then
      case type(Lhs)
        of "_mult" do
          s := split(Lhs, t->bool(has(t, x) and type(t) = "exp"));
          if type(s[1]) = "exp" then
            lhs_f := op(s[1]);
          else
            lhs_f := _plus(map(op(s[1]), op));
          end_if;
          lhs_c := s[2];
          break;
        of "exp" do
          lhs_f := op(Lhs);
          lhs_c := 1;
          break;
        otherwise
          return(FAIL);
      end_case;
      case type(Rhs)
        of "_mult" do
          s := split(Rhs, t->bool(has(t, x) and type(t) = "exp"));
          if type(s[1]) = "exp" then
            rhs_f := op(s[1]);
          else
            rhs_f := _plus(map(op(s[1]), op));
          end_if;
          rhs_c := s[2];
          break;
        of "exp" do
          rhs_f := op(Rhs);
          rhs_c := 1;
          break;
        otherwise
          return(FAIL);
      end_case;
      if not has(lhs_c, x) and not has(rhs_c, x) then
        if sign(lhs_c) = sign(rhs_c) then
          return(solvelib::isolate(lhs_f-rhs_f+ln(abs(lhs_c))-ln(abs(rhs_c)), 0,
                                   x, options, reclevel+1, known));
        else
          return({});
        end_if;
      end_if;
    end_if;

    return(FAIL);
  end_proc;


/* sinify(ex, x)
 *
 * Tries to express all non-constant cos-/tan-/cot-subexpressions of ex(x)
 * in *simple* terms of sin, or returns FAIL if nothing changed or the
 * result still contains non-constant cos-/tan-/cot-subexpressions.
 */
  sinify :=
  proc(ex, x)
    local l, s;
  begin
    l := misc::maprec(ex,
                      {"tan", "cot"} =
                      proc(t)
                      begin
                        if has(t, x) then
                          rewrite(t, sincos);
                        else
                          return(t);
                        end_if;
                      end_proc);
    if has(l, cos) and has(l, sin) then
      l := misc::maprec
      (expand(l, MaxExponent = 5),
       {"_mult"} =
       proc(t)
         local a, b, c, m;
       begin
         /* rewrite
          *   cos(x)*sin(x)
          * as
          *   sin(2*x)/2
          */
         if has(t, x) then
           a := genident();
           b := genident();
           c := genident();
           m := match(t, a*cos(b)*sin(c), hold(Const)={cos, sin});
           if m <> FAIL then
             [a, b, c] := map([a, b, c], foo->subs(foo, op(m)));
             if is(b = c, Goal = TRUE) then
               return(a*sin(2*b)/2)
             end_if;
           end_if;
         end_if;
         return(t);
       end_proc,
       {"_power"} =
       proc(t)
         local a, b;
       begin
             /* rewrite
              *   cos(x)^2
              * as
              *   1-sin(x)^2
              */
         a := op(t, 1);
         b := op(t, 2);
         if has(t, x) and type(a) = "cos" then
           if b = 2 then
             return(1-sin(op(a))^2);
           elif b = -2 then
             return(1/(1-sin(op(a))^2));
           end_if;
         end_if;
         return(t);
       end_proc);
      s := split(l, has, x)[1];
      if s = null() or not has(s, cos) then
        return(l);
      end_if;
    end_if;
    return(FAIL);
  end_proc;


/* cosify(ex, x)
 *
 * Tries to express all non-constant sin-/tan-/cot-subexpressions of ex(x)
 * in *simple* terms of cos, or returns FAIL if nothing changed or the
 * result still contains non-constant sin-/tan-/cot-subexpressions.
 */
  cosify :=
  proc(ex, x)
    local l, s;
  begin
    l := misc::maprec(ex,
                      {"tan", "cot"} =
                      proc(t)
                      begin
                        if has(t, x) then
                          rewrite(t, sincos)
                        else
                          return(t)
                        end_if
                      end_proc);
    if has(l, cos) and has(l, sin) then
      l := misc::maprec
      (expand(l, MaxExponent = 5),
      /*
       {"_mult"} =
       proc(t)
         local a, b, c, m;
       begin
         /* rewrite
          *   cos(x)*sin(x)
          * as
          *   cos(PI/2-2*x)/2
          */
         if has(t, x) then
           a := genident();
           b := genident();
           c := genident();
           m := match(t, a*cos(b)*sin(c), hold(Const)={cos, sin});
           if m <> FAIL then
             [a, b, c] := map([a, b, c], foo->subs(foo, op(m)));
             if is (b = c, Goal = TRUE) then
               return(a*cos(PI/2-2*b)/2);
             end_if;
           end_if;
         end_if;
         return(t);
       end_proc,
       */
       {"_power"} =
       proc(t)
         local a, b;
       begin
             /* rewrite
              *   sin(x)^2
              * as
              *   1-cos(x)^2
              */
         a := op(t, 1);
         b := op(t, 2);
         if has(t, x) and type(a) = "sin" then
           if b = 2 then
             return(1-cos(op(a))^2);
           elif b = -2 then
             return(1/(1-cos(op(a))^2));
           end_if;
         end_if;
         return(t);
       end_proc);
      s := split(l, has, x)[1];
      if s = null() or not has(s, sin) then
        return(l);
      end_if;
    end_if;
    return(FAIL);
  end_proc;


combinePower :=
proc(ex, x, changeflag)
  local pows, nonpows, changed, t, a, b, ca, cb, i, j;
begin
  assert(options[Real] = TRUE);
  changed := changeflag;

  case type(ex)
    of "_mult" do
      [pows, nonpows, t] := split([op(ex)], has, x);
      pows               := map(pows, proc(s)
                                        local t, ct;
                                      begin
                                        [t, ct] := combinePower(s, x,
                                                                changed);
                                        changed := changed or ct;
                                        case type(t)
                                          of "_power" do
                                            return([op(t)]);
                                          of "exp" do
                                            return([exp(1), op(t)]);
                                          otherwise
                                            return([t, 1]);
                                          end_case;
                                        end_proc);

      /* rewrite
       *   a^b*a^c
       * as
       *   a^(b+c)
       */
      for i from 1 to nops(pows)-1 do
        for j from i+1 to nops(pows) do
          if pows[j] <> [1, 1] and
             is(pows[i][1] = pows[j][1], Goal = TRUE)
          then
            pows[i][2] := pows[i][2]+pows[j][2];
            pows[j]    := [1, 1];
            changed    := TRUE;
          end_if;
        end_for;
      end_for;

      /* rewrite
       *   a^c*b^c
       * as
       *   (a*b)^c
       */
      for i from 1 to nops(pows)-1 do
        for j from i+1 to nops(pows) do
          if pows[j] <> [1, 1] and
             is(pows[i][2] = pows[j][2], Goal = TRUE)
          then
            pows[i][1] := pows[i][1]*pows[j][1];
            pows[j]    := [1, 1];
            changed    := TRUE;
          end_if;
        end_for;
      end_for;

      /* rewrite
       *   a^c*b^(-c)
       * as
       *   (a/b)^c
       */
      for i from 1 to nops(pows)-1 do
        for j from i+1 to nops(pows) do
          if pows[j] <> [1, 1] and
             is(pows[i][2] = -pows[j][2], Goal = TRUE)
          then
            pows[i][1] := pows[i][1]/pows[j][1];
            pows[j]    := [1, 1];
            changed    := TRUE;
          end_if;
        end_for;
      end_for;

      return([_mult(op(map(pows, foo->_power(op(foo)))),
                    op(nonpows)),
              changed]);

    of "_power" do
      /* rewrite
       *   (a^b)^c
       * as
       *   a^(b*c)
       */
      [a, ca] := combinePower(op(ex, 1), x, changed);
      [b, cb] := combinePower(op(ex, 2), x, changed);
      if type(a) = "_power" then
        return([op(a, 1)^(op(a, 2)*b), TRUE]);
      else
        return([a^b, changed or ca or cb]);
      end_if;

    of "exp" do
      [b, cb] := combinePower(op(ex), x, changed);
      return([exp(b), changed or cb]);

    otherwise
      return([ex, changed]);
  end_case;
end_proc;


combineLog :=
proc(ex, x, changeflag, use_x)
  local splitproc, logs, nonlogs, preimgs, multops, changed,
        t, u, v, w, a, b, i, j;
begin
  splitproc :=
  proc(ops, splittype)
    local t, i;
  begin
    /* step 1
     * ======
     * split ops into three parts:
     *   1st: expressions of type splittype
     *        raised to a (small) negative power
     *   2nd: expressions of type splittype containing x
     *        raised to a (small) positive power
     *   3rd: everything else
     */
    t := split(ops, proc(s)
                      local i;
                    begin
                      case type(s)
                        of "_power" do
                          if type(op(s, 1)) = splittype then
                            if op(s, 2) in {i $ i = 1..10} and
                               (not use_x or has(s, x))
                            then
                              return(FALSE);
                            elif op(s, 2) in {-i $ i = 1..10} then
                              return(TRUE);
                            end_if;
                          end_if;
                          break;
                        of splittype do
                          if not use_x or has(s, x) then
                            return(FALSE);
                          end_if;
                          break;
                      end_case;
                      return(UNKNOWN);
                    end_proc);
    /* step 2
     * ======
     * rewrite powers as sequences
     */
    t[1] := map(t[1], foo->(op(foo, 1) $ i = 1..-op(foo, 2)));
    t[2] := map(t[2], proc(s)
                        local i;
                      begin
                        if type(s) = "_power" then
                          return(op(s, 1) $ i = 1..op(s, 2));
                        else
                          /* type(s) = splittype */
                          return(s);
                        end_if;
                      end_proc);
    return([t, nops(t[1]), nops(t[2])]);
  end_proc;

  assert(options[Real] = TRUE);
  changed := changeflag;

  if use_x then
    /* rewrite
     *   ln(exp(f(x)))
     * as
     *   f(x)
     */
    t := misc::maprec(ex, {"ln"} = proc(s)
                            local a;
                          begin
                            a := op(s);
                            if type(a) = "exp" and has(a, x) then
                              return(op(a));
                            else
                              return(s);
                            end_if;
                          end_proc);
    if ex <> t then
      ex      := t;
      changed := TRUE;
    end_if;
  end_if;

  case type(ex)
    of "_plus" do
      /* rewrite
       *   log(a, b(x))+log(a, c(x))
       * as
       *   log(a, b(x)*c(x))
       */
      logs    := [];
      nonlogs := [];
      preimgs := [];
      for t in ex do
        [u, v, w] := combineLog(t, x, changed, use_x);
        if v = {} then
          return([ex, v, changed]);
        else
          preimgs := append(preimgs, v);
          if w then
            changed := w;
          end_if;
        end_if;
        if type(u) in {"log", "ln"} and (not use_x or has(u, x)) then
          if type(u) = "log" then
            a := op(u, 1);
            b := op(u, 2);
          else
            /* type(u) = "ln" */
            a := exp(1);
            b := op(u);
          end_if;
          for i from 1 to nops(logs) do
            if testeq(logs[i][1], a) = TRUE then
              logs[i][2] := logs[i][2]*b;
              changed := TRUE;
              break;
            end_if;
          end_for;
          if i > nops(logs) then
            logs := append(logs, [a, b]);
          end_if;
        else
          nonlogs := append(nonlogs, u);
        end_if;
      end_for;
      return([_plus(op(map(logs, foo->log(op(foo)))), op(nonlogs)),
              _intersect(op(preimgs)),
              changed]);

    of "_mult" do
      /* rewrite
       *   log(a, b(x))/log(a, c)
       * as
       *   ln(b(x))/ln(c)
       */
      [t, u, v] := splitproc([op(ex)], "log");
      for i from 1 to u do
        for j from 1 to v do
          if type(t[2][j]) = "log" and op(t[1][i], 1) = op(t[2][j], 1) then
            t[1][i] := ln(op(t[1][i], 2));
            t[2][j] := ln(op(t[2][j], 2));
            changed := TRUE;
            break;
          end_if;
        end_for;
      end_for;
      multops := [1/t[1][i] $ i = 1..u,
                  t[2][i] $ i = 1..v,
                  op(t[3])];

      /* rewrite
       *   ln(b(x))/ln(c)
       * as
       *   log(c, b(x))
       */
      [t, u, v] := splitproc(multops, "ln");
      multops := [log(op(t[1][i]), op(t[2][i])) $ i = 1..min(u, v),
                  1/t[1][i] $ i = v+1..max(u, v),
                  t[2][i] $ i = u+1..max(u, v),
                  op(t[3])];
      if min(u, v) > 0 then
        changed := TRUE;
      end_if;

      assert(nops(multops) > 0);
      if nops(multops) = 1 then
        /* type(_mult(op(multops))) <> "_mult"
         * --> jump to the correct branch
         */
        return(combineLog(multops[1], x, changed, use_x));
      end_if;

      /* rewrite
       *   c*log(a, b(x))
       * as
       *   log(a, b(x)^c)
       */
      t := split(multops,
                 foo->bool(type(foo) in {"log", "ln"} and
                           (not use_x or has(foo, x))));
      if nops(t[1]) > 0 then
        u := t[1][1];
        if type(u) = "log" then
          a := op(u, 1);
          b := op(u, 2);
        else
          /* type(u) = "ln" */
          a := exp(1);
          b := op(u);
        end_if;
        if not use_x or nops(t[2]) > 0 then
          changed := TRUE;
        end_if;
        if use_x then
          return([_mult(log(a, b^_mult(op(t[2]))),
                        t[1][i] $ i = 2..nops(t[1])),
                  solvelib::preImage(b, x,
                                     Dom::Interval(0, infinity),
                                     Real),
                  changed]);
        else
          return([log(a, b^_mult(t[1][i] $ i = 2..nops(t[1]), op(t[2]))),
                  solvelib::preImage(b, x,
                                     Dom::Interval(0, infinity),
                                     Real),
                  changed]);
        end_if;
      else
        /* nothing to do */
        return([_mult(op(multops)), R_, changed]);
      end_if;

    of "log" do
      return([ex,
              solvelib::preImage(op(ex, 2), x,
                                 Dom::Interval(0, infinity),
                                 Real),
              changed]);
    of "ln" do
      return([ex,
              solvelib::preImage(op(ex), x,
                                 Dom::Interval(0, infinity),
                                 Real),
              changed]);

    otherwise
      return([ex, R_, changed]);
  end_case;
end_proc;


isolatePowerLog :=
proc(Lhs, Rhs, x)
  local simplifyPowerLog, ex, preimg, changeflag, s;
  save MAXEFFORT;
begin
  simplifyPowerLog :=
  proc(foo)
    local l;
  begin
    if type(foo) = DOM_SET then
      l:= foo;
      l := misc::maprec(l, {"log"} = proc(t)
                             begin
                               if not has(t, x) then
                                 return(rewrite(t, ln));
                               end_if;
                               return(t);
                             end_proc,
                             {"_power"} = proc(t)
                               local a, b;
                             begin
                               if not has(t, x) then
                                 a := op(t, 1);
                                 b := op(t, 2);
                                 if is(a > 0) = TRUE then
                                   return(hold(exp)(b*ln(a)));
                                 end_if;
                               end_if;
                               return(t);
                             end_proc);
      l:= map(l, combine);
      l := map(l, simplify);
      l := map(l, fred->(combineLog(fred, x, FALSE, FALSE)[1]));
      return(l);
    end_if;
    return(foo);
  end_proc;


  if map(indets(Lhs, RatExpr), op, 0) minus
    {hold(exp), hold(log), hold(ln), hold(_power), FAIL} <> {} then
    // quick exit
    return(FAIL)
  end_if;
  
  if options[Real] then
    if options[IgnoreAnalyticConstraints] then
      ex := combine(Lhs, IgnoreAnalyticConstraints)
    else
      ex := combine(Lhs)
    end_if;
    if ex <> Lhs then
      MAXEFFORT:= MAXEFFORT/2;
      s := solvelib::isolate(ex, Rhs, x, options, reclevel+1, known);
      if type(s) <> "solve" then
        return(simplifyPowerLog(s));
      end_if;
    end_if;
    MAXEFFORT:= MAXEFFORT/3;
    [ex, changeflag]         := combinePower(Lhs, x, FALSE);
    [ex, preimg, changeflag] := combineLog(ex, x, changeflag, TRUE);
    if preimg = {} then
      return({});
    elif changeflag then
      MAXEFFORT:= MAXEFFORT/2;
      s := solvelib::isolate(ex, Rhs, x, options, reclevel+1, known);
      if type(s) <> "solve" then
        return(solvelib::solve_intersect
               (simplifyPowerLog(s), preimg));
      end_if;
    end_if;
  end_if;
  return(FAIL);
end_proc;

Ln:=
proc(a)
begin
  if type(a) = "exp" then
    op(a, 1)
  else
    ln(a)
  end_if
end_proc;
  
Arcsin:=
proc(a)
begin
  case type(a)
    of "sin" do
      return(op(a, 1))
    of "cos" do
      // cos(a) = sin(a+PI/2)
      return(op(a, 1) + PI/2)
    otherwise
      arcsin(a)
  end_case
end_proc;


Arccos:=
proc(a)
begin
  case type(a)
    of "cos" do
      return(op(a, 1))
    of "sin" do
      return(op(a, 1) - PI/2)
    otherwise
      arccos(a)
  end_case
end_proc;


Arctan:=
proc(a)
begin
  if type(a) = "tan" then
    op(a, 1)
  else
    arctan(a)
  end_if
end_proc;


//////////////////////////////////////////////////////////////////
// b e g i n   o f   m a i n   p r o g r a m
//
/////////////////////////////////////////////////////////////////

  Lhs:=lhside;
  assert(not has(Rhs, x));

  lev:=options[MaxDegree];

  pw:=if options[IgnoreSpecialCases] then
        solvelib::ignoreSpecialCases
      else
        piecewise::new
      end_if;


  maxreclevel:= options[MaxRecLevel];
  if reclevel>= maxreclevel or MAXEFFORT <= 100 or
    (not testtype(Lhs, Type::RatExpr(x)) and
    MAXEFFORT <= length(Lhs)) then
    userinfo(2, "Maximum recursion level exceeded in solvelib::isolate");
    return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
  end_if;




  // isolate has been called recursively, and the current arguments have
  // already been there
  if contains(known, [Lhs, Rhs, x]) then
    // determine simplest result in call stack
    known:= [op(known)];
    s:= map(known, l -> length(l[1]));
    l:= known[contains(s, min(s))];
    return(hold(solve)(l[1] = l[2], l[3], solvelib::nonDefaultOptions(options)))
  else
    known:= known union {[Lhs, Rhs, x]}
  end_if;

  userinfo(3,"trying to isolate ".expr2text(x)." in ".expr2text(Lhs=Rhs));
  userinfo(5,"Recursion level is ".expr2text(reclevel));
  if Lhs = x then
    return({Rhs})
  elif not has(Lhs, x) then
    if options[IgnoreAnalyticConstraints] then
      if Lhs = Rhs then
        if options[Real] then
          return(R_)
        else
          return(C_)
        end_if
      else
        return({})
      end_if
    elif options[Real] then
      return(pw([Lhs = Rhs, R_], [Lhs <> Rhs, {}]))  
    else
      return(pw([Lhs = Rhs, C_], [Lhs <> Rhs, {}]));
    end_if;
  end_if;

  // case analysis, depending on the type of left hand side
  // solve f(x) = C for x, for different types of f


  // first try overloading

  if domtype(Lhs) = DOM_EXPR and
    type((a := eval(op(Lhs, 0)))) = DOM_FUNC_ENV then
    if a::isolate <> FAIL then
      return(a::isolate(Lhs, Rhs, x, options, reclevel, known))
    end_if
  end_if;

  if type(Lhs) = piecewise then
    if not has(piecewise::conditions(Lhs), x) then
      return(piecewise::extmap(Lhs, solvelib::isolate, Rhs, x, options, reclevel+1, known))
    else  
      return(piecewise::solve(piecewise::extmap(Lhs, _subtract, Rhs), x, options))
    end_if  
  end_if;
  
  if type(Rhs) = piecewise then
    return(piecewise::extmap(Rhs, R -> solvelib::isolate(Lhs, R, x, options, reclevel+1, known)))
  end_if;

  // give up if piecewise occurs in any other way, and in other hopeless cases
  if has(map(misc::subExpressions(Lhs, {"limit", "int", "sum", piecewise}), op, 2), x) then
    return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
  end_if;

  // try solving numerically if x is real
  if freeIndets(Lhs-Rhs) minus {x} = {} and
    type((l:= getprop(x, "Targets" = {Dom::Interval}))) = Dom::Interval then
    if traperror
      ((l:= numeric::hasroot(Lhs-Rhs,
                               x, l::dom::left(l), l::dom::right(l))),
       MaxSteps = 3
       ) = 0
      and l = FALSE then
      return({})
    end_if
  end_if;


  // special case for IgnoreAnalyticConstraints: replace surd by _power, although this is
  // mathematically wrong in general
 
  if options[IgnoreAnalyticConstraints] then
    Lhs:= misc::maprec(Lhs, {"surd"} =
                       proc(a)
                       begin
                         // a= surd(x, n) = x^(1/n)
                         op(a, 1)^(1/op(a, 2))
                       end_proc
                       ):
  end_if;



  /* try to recognize a polynomial */
  if testtype(Lhs,Type::PolyExpr(x)) then
    if traperror((l:= poly(Lhs-Rhs, [x]))) = 0 then
      return(solvelib::solve_poly(l, x, options))
    end_if
  end_if;


 /*

   Try to find a common subexpression. If Lhs = p(s(x)), we solve
   p(u) = Rhs f. u;  then, fr each such u, s(x)=u fr x

  */

   s:= misc::greatestCommonExpression(Lhs, x);
   assert(s <> FAIL);
   if s<>x then
     MAXEFFORT:= MAXEFFORT*3/4;
     u:= genident("YINTERN");
     p:= subsex(Lhs, s=u);
     if is(s in R_, Goal = TRUE) then
       assume(u in R_)
     end_if;
     q:= solvelib::isolate(p, Rhs, u, options, reclevel+1, known);
     if type(q) = DOM_SET and nops(q) = 1 then
       return(solvelib::isolate(s, op(q, 1), x, options, reclevel+1,
                                 known))
     elif not has(q, hold(solve)) then
       return(solvelib::preImage(s, x, q, options))
     else
       return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
     end_if;
   end_if;





  case type(Lhs)
    of "_plus" do

      if iszero(Rhs) and nops(Lhs) = 2 then
        // handle f(x)+g(x)=0 by trying to solve f(x)=-g(x) heuristically

        if type(op(Lhs,1))="_mult" then
          a:= applyHeuristic(op(Lhs, 2), -op(Lhs, 1))
        else
          a:= applyHeuristic(op(Lhs, 1), -op(Lhs, 2))
        end_if;

        if a <> FAIL then
          return(a)
        end_if;

      end_if; // nops(Lhs)=2
      

      // extract content
      
      if traperror((a:= content(Lhs))) = 0 and a <> FAIL then 
        if iszero(a) then
          return(piecewise([Rhs = 0, C_], [Rhs<>0, {}]))
        else  
          Lhs:= Lhs/a;
          Rhs:= Rhs/a
        end_if  
      end_if;        
     

      if ((u:= tryRationalize(Lhs, Rhs, x, options))) <> FAIL then
        return(u)
      end_if;  

      if ((u:= fractionalPowerHeuristicPlus())) <> FAIL then
        return(u)
      end_if;  

    

       // normalization heuristic
      // do *not* call normal because it would cancel common
      // factors of the numerator and denominator
/*
      a:= lcm(denom(op(Lhs, i)) $i=1..nops(Lhs));
      a:= expr(factor(a));
      if a<>1 then
        if type(a) = "_mult" then
          l:= split(a, has, x)
        elif has(a, x) then
          l:= [a, 1]
        else
          l:= [1, a]
        end_if;
        if type(l[2]) <> DOM_INT then
          u:= factor(l[2]);
          if nops(u) > 1 then
            assumeAlso(_and(op(u, 2*i) <>0 $i=1..nops(u) div 2))
          end_if;
        end_if;

      
        u:= map(Lhs, normal@_mult, l[1]*l[2]) / l[1];
        if u <> Lhs then
          a:= solvelib::isolate(u, Rhs*l[2], x, options, reclevel+1,
                                   known);
          if not has(a, hold(solve)) then
            return(a)
          end_if
        end_if;
      end_if;
*/      
      
      // apply linear shifts to arguments of ln if possible
      vars:= select(indets(Lhs, PolyExpr), u -> has(u, x));
      if nops((vars:= select(vars, V-> (type(V) = "ln")))) = 1
        and type((a:= op(vars, [1, 1]))) = "_plus" and
        ((l:= Type::Linear(a, [x]))) <> FALSE and not iszero(l[2])
        then
        // substitute y = l[1]*x + l[2], i.e. x = (y-l[2])/l[1]
        u:= genident();
        s:= piecewise([l[1]=0,
                       solvelib::isolate(subs(Lhs, l[1]=0, EvalChanges), Rhs, x,
                                         options, reclevel+1, known)],
                      [l[1]<>0,
                       solvelib::preImage(l[1]*x+l[2], x,
                       solvelib::isolate(subs(Lhs, x=(u-l[2])/l[1], EvalChanges), Rhs,
                                         u, options, reclevel+1, known),
                                          options)]
                      );
        if not has(s, hold(solve)) then
          return(s)
        end_if;
      end_if;
      

      
      if nops(Lhs) <= 2 and has(Lhs, {hold(exp), hold(ln)}) then
        l:= solvelib::matchplus(Lhs, Rhs, x, options);
        if l <> FAIL then
          return(l)
        end_if;
      end_if;
      
      

      if iszero(Rhs) then
        /* try to factor the left hand side, but only over Q */
        u := misc::maprec(Lhs,
               {"log"} = proc(t)
                 name rewriteLog;
               begin
                 if has(t, x) then
                   return(rewrite(t, ln));
                 else
                   return(t);
                 end_if;
               end_proc,
               {"tan", "cot"} = proc(t)
                 name rewriteTanAndCot;
               begin
                 if has(t, x) then
                   return(rewrite(t, sincos));
                 else
                   return(t);
                 end_if;
               end_proc);
        if traperror((l := expr(factor(numer(u)))), MaxSteps = 2) <> 0 then
          break
        end_if;
        if type(l)="_mult" then
          if options[IgnoreAnalyticConstraints] or not options[hold(DiscontCheck)] then
            MAXEFFORT:= MAXEFFORT/nops(l);
            return(_union(solvelib::isolate(op(l,i),0,x,options, reclevel+1,
                                            known)
                          $i=1..nops(l)
                          )
                   )
          else  
            MAXEFFORT:= MAXEFFORT/nops(l)/2;
            s:= [FAIL $ nops(l)]; // to initialize list
            for i from 1 to nops(l) do 
               s[i]:= solvelib::isolate(op(l,i),0,x,options, reclevel+1,
                                            known);
               if type(s[i]) = "solve" then
                    // give up
                    return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
               end_if;
            end_for;
            l:= _union(op(s));
            return(solvelib::solve_minus(l,
                                         discont(u, x, hold(Undefined), options)
                                         )
                   )
           
          end_if
        end_if
      end_if; // iszero(Rhs)

      s:= select(indets(Lhs, PolyExpr), X -> type(X) = "ln" and has(X, x));
      if nops(s) > 0 then
        if (options[IgnoreAnalyticConstraints] or not options[Real]) then
          l:= combine(Lhs, ln, if options[IgnoreAnalyticConstraints] then IgnoreAnalyticConstraints end);
          if l<>Lhs then
            // MAXEFFORT:= 1/2*MAXEFFORT;
            return(solvelib::isolate(l,Rhs,x,options, reclevel+1, known))
          end_if;
        end_if;
        
        if not options[Real] then
          // heuristic for sums of several logarithms
          if nops(s) >= 2 and degree(poly(Lhs, [op(s)])) = 1 then
            // Lhs is a linear combination of logarithms
            l:= exp(Lhs);
            l:= misc::maprec(l, 
            {"exp"} = proc(a: "exp")
                      begin
                      a:= exp::expand(a);
                      if type(a) = "exp" then
                        a:= exp::simplify(a)
                      end_if;
                      a
                      end_proc
            );
            if not contains(map(indets(l, RatExpr), type), "exp") then
              u:= exp(Rhs);
              if type(u) = "exp" then
                u:= exp::simplify(u)
              end_if;
              MAXEFFORT:= MAXEFFORT/2;
              s:= solvelib::isolate(l, u, x, options, reclevel+1, known);
              if not has(s, hold(solve)) then
                return(solvelib::checkSolutions(s, Lhs=Rhs, x, options))
              else
                return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
              end_if;  
            end_if;
          end_if;  
        end_if
      end_if;

      if has(Lhs,log) and (options[IgnoreAnalyticConstraints] or not options[Real]) then
        l:= combine(Lhs, log, if options[IgnoreAnalyticConstraints] then IgnoreAnalyticConstraints end);
        if l<>Lhs then
          // MAXEFFORT:= 3/4*MAXEFFORT;
          return(solvelib::isolate(l,Rhs,x,options, reclevel+1, known))
        end_if;
        l:= rewrite(Lhs, ln);
        if l<>Lhs then
          return(solvelib::isolate(l,Rhs,x,options, reclevel+1, known))
        end_if
      end_if;
      
      if has(Lhs, {hold(sin), hold(cos)}) then
        // rewrite all tan/cot
        Lhs:= subs(Lhs, [hold(tan) = (X -> sin(X)/cos(X)), hold(cot) = (X -> cos(X)/sin(X))], EvalChanges);
      end_if;  
      
      if traperror((l := sinify(Lhs, x)), MaxSteps = 2) = 0 and
        l <> FAIL and not contains(known, [l, Rhs, x]) then
        return(solvelib::isolate(l, Rhs, x, options, reclevel+1, known))
      end_if;
      if traperror((l := cosify(Lhs, x)), MaxSteps = 2) = 0 and
        l <> FAIL and not contains(known, [l, Rhs, x]) then
        return(solvelib::isolate(l, Rhs, x, options, reclevel+1, known))
      end_if;

      if MAXEFFORT > 500 then
        s := split(Lhs, proc(t)
                        begin
                          case type(t)
                            of "_mult" do
                              if nops(t) = 2 and
                                not has(t, {cos, sin, tan, cot}) and
                                (type(op(t, 1)) = "_power" and
                                 type(op(t, 2)) in {DOM_INT, DOM_RAT} and 
                                 abs(op(t, 2)) < Pref::autoExpansionLimit()
                                 or
                                 type(op(t, 2)) = "_power" and
                                 type(op(t, 1)) in {DOM_INT, DOM_RAT})
                                then
                                return(TRUE);
                              else
                                return(FALSE);
                              end_if;
                     of "_power" do
                   return(TRUE);
                     otherwise
                   return(FALSE);
                   end_case;
                   end_proc);
        if type(s[1]) = "_plus" and not has(s[1], {hold(Re), hold(Im)}) then
          l := expr(factor(s[1]));
          case type(l)
            of "_plus" do
              if iszero(Rhs) then
                MAXEFFORT:= MAXEFFORT/6;
                a := [op(s[1], 2..nops(s[1])), 
                      s[2]
                     ];
                b :=
                solvelib::isolate(op(s[1], 1), 0, x,
                                  options, reclevel+1, known)
                intersect
                solvelib::isolate(_plus(op(a)), 0, x,
                                  options, reclevel+1, known);
                if not has(b, hold(solve)) then
                  q := solvelib::isolate
                  (
                   _plus(op(map(a, foo->foo/op(s[1], 1)))),
                   -1,
                   x,
                   options, reclevel+1, known)
                  union
                  eval(b);
                  
                  if not has(q, hold(solve)) then
                    return(q)
                  end_if;
                end_if;
                // break and spend the other half of the "effort"
                MAXEFFORT:= MAXEFFORT*3
              end_if;
              break;
            of "_mult" do
              MAXEFFORT:= MAXEFFORT/2;
              q:= solvelib::isolate(l+s[2], Rhs, x,
                                    options, reclevel+1, known);
              if not has(q, hold(solve)) then
                return(q)
              end_if;
              // spend the other half of the effort
              break
          end_case;
        end_if;
      end_if;

      l := isolatePowerLog(Lhs, Rhs, x);
      if l <> FAIL then
        return(l);
      end_if;

      break;
    of "_mult" do

      if iszero(Rhs) then /* one factor must be zero */
        userinfo(10, "Solving each factor separately");
        if options[IgnoreAnalyticConstraints] then
          MAXEFFORT:= MAXEFFORT/(nops(Lhs)+1);
          solutions:= _union(solvelib::isolate(op(Lhs,i),0,x,options,
                                         reclevel+1, known)
                             $ i=1..nops(Lhs)
                             );
          options[NoWarning]:= TRUE;
          return(solvelib::checkSolutions(solutions, Lhs=Rhs, x, options))
        end_if;
        MAXEFFORT:= MAXEFFORT/(nops(Lhs)+2);
        l:= _union(solvelib::isolate(op(Lhs,i),0,x,options,
                                         reclevel+1, known)
                       $ i=1..nops(Lhs));
        if not has(l, hold(solve)) then
           if options[hold(DiscontCheck)] then
             return(solvelib::solve_minus
                         (l, discont(Lhs, x, Undefined, options),
                         options))
           else
             return(l)
           end_if
        else
           // give up
           return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))           
        end_if;
      end_if;


      // handle products with negative powers by multiplying both sides
      // of the equation:
      // f(x) * g(x)^(-k) = C is equivalent to
      // f(x) = C* g(x)^k and g(x) <> 0, which in turn is equivalent to
      // f(x) - C*g(x)^k = 0

      a:= split(Lhs, fctor ->
                type(fctor) ="_power" and
                testtype(op(fctor, 2), Type::Negative) = TRUE);
      if a[1] <> 1 then
        if options[IgnoreAnalyticConstraints] then
          MAXEFFORT:= MAXEFFORT/2;
          return(
                 solvelib::isolate(a[2] - Rhs/a[1], 0, x, options,
                                   reclevel+1, known )
                 )
        end_if;
        MAXEFFORT:= MAXEFFORT/3;
        p:= solvelib::isolate(a[2] - Rhs/a[1], 0, x, options,
                              reclevel+1, known );
        if type(p) = DOM_SET then
          return(select(p, u -> (traperror(evalAt(a[1], x=u)) = 0) ))
        end_if;
        if type(p) = "solve" then
          // give up
          return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))           
        end_if;
        q:= solvelib::isolate(1/a[1], 0, x, options,
                                  reclevel+1, known);
        // remove branches where the input was not defined
        if type(q) = piecewise then
          q:= piecewise::selectExpressions(q, _unequal, C_)
        end_if;
        return(solvelib::solve_minus(p, q, options));
      end_if;


      if nops(select([op(Lhs)], testtype, "exp")) > 1 then
        Lhs:= combine(Lhs, exp);
        if type(Lhs) <> "_mult" then
          return(solvelib::isolate(Lhs, Rhs, x, options, reclevel+1, known))
        end_if
      end_if;
      
      if (u:= tryRationalize(Lhs, Rhs, x, options)) <> FAIL then
        return(u)
      end_if;  
      
      u:= fractionalPowerHeuristicMult();
      if u <> FAIL then 
        return(u)
      end_if;  
      
    

      
      if nops(Lhs) <= 2 and has(Lhs, {hold(exp), hold(ln)}) then
        l:= solvelib::matchmult(Lhs, Rhs, x, options);
        if l <> FAIL then
          return(l);
        end_if;
      end_if;

      if has(Lhs, hold(ln)) and has(Lhs, hold(log)) then
        Lhs:= rewrite(Lhs, ln)
      end_if;

      /* solve a(x)*b(x) = c by solving 1/a(x)-b(x)/c = 0
       */
      if nops(Lhs) = 2 and is(Rhs <> 0, Goal = TRUE) then
        a := op(Lhs, 1);
        b := op(Lhs, 2);
        if has(a, x) and not has(a, [hold(sin), hold(cos)]) and
           type(a) <> "_power" and not iszero(a) and
           has(b, x) and not has(b, [hold(sin), hold(cos)]) and
           type(b) <> "_power"
        then
          MAXEFFORT:= MAXEFFORT/2;
          l := solvelib::isolate(a^(-1)-b/Rhs, 0, x,
                                 options, reclevel+1, known);
          if not has(l, hold(solve)) then
            return(l);
          end_if;
        end_if;
      end_if;

      l := sinify(Lhs, x);
      if l <> FAIL then
        return(solvelib::isolate(l, Rhs, x, options, reclevel+1, known));
      end_if;
      l := cosify(Lhs, x);
      if l <> FAIL then
        return(solvelib::isolate(l, Rhs, x, options, reclevel+1, known));
      end_if;

      l := isolatePowerLog(Lhs, Rhs, x);
      if l <> FAIL then
        return(l);
      end_if;

      break
    of "_power" do
      if type(Rhs) = "_power" then
        l:= applyHeuristic(Lhs, Rhs);
        if l <> FAIL then
          return(l)
        end_if;
      end_if;
      a := op(Lhs, 1);
      b := op(Lhs, 2);
      if options[Real] then
        if not has(b, x) then
          if options[IgnoreAnalyticConstraints] and type(b) <> DOM_INT then
            return(solvelib::isolate(a, Rhs^(1/b), x, options,
                                     reclevel+1, known))
          end_if;
          /* solving a(x)^b = c for x */
          return(solvelib::preImage(a, x,
                   pw([Rhs > 0,
                       pw([b in 2*Z_ minus {0},
                           {Rhs^(1/b), -Rhs^(1/b)}],
                          [not b in 2*Z_,
                           {Rhs^(1/b)}],
                          [b = 0 and Rhs = 1,
                           R_],
                          [b = 0 and Rhs <> 1,
                           {}])],
                      [Rhs < 0,
                       pw([b in 2*Z_+1,
                           {-(-1/Rhs)^(-1/b)}],
                          [not b in 2*Z_+1,
                           {}])],
                      [Rhs = 0,
                       pw([b > 0,
                           {0}],
                          [b <= 0,
                           {}])]),
                   options));
        elif not has(a, x) then
          /* solving a^b(x) = c for x */
          if options[IgnoreAnalyticConstraints] then
            if (type(a) = DOM_INT or type(a) = DOM_RAT) and a>0 then
              return(solvelib::isolate(b, simplify(log(a, Rhs)), x, options,
                                       reclevel+1))
            end_if;
            return(solvelib::isolate(rewrite(Lhs, exp), Rhs, x,
                                     options, reclevel+1, known)
                   )
          end_if;
          return(pw([a > 0 and a <> 1,
                     solvelib::preImage(b, x,
                       pw([Rhs > 0,
                           if testtype(a, Type::Numeric) then
                             {simplify(log(a, Rhs))}
                           else  
                             {ln(Rhs)/ln(a)}
                           end_if
                             ],
                          [Rhs <= 0,
                           {}]),
                       options)],
                    [a = 0,
                     solvelib::preImage(b, x,
                       pw([Rhs = 0,
                           Dom::Interval(0, infinity)],
                          [Rhs = 1,
                           {0}],
                          [Rhs <> 0 and Rhs <> 1,
                           {}]),
                       options)],
                    [a = 1,
                     pw([Rhs = 1,
                         R_],
                        [Rhs <> 1,
                         {}])],
                    [a < 0, /* FIXME: needs to be implemented */
                     hold(solve)(Lhs = Rhs, x,
                                 solvelib::nonDefaultOptions(options))]));
        elif testeq(a, b) = TRUE then
          /* solving f(x)^f(x) = c for x */
          k := genident();
          return(pw([Rhs >= exp(-exp(-1)),
                     solvelib::preImage(b, x,
                       pw([Rhs <= 1,
                           {exp(lambertW(k, ln(Rhs))) $ k = -1..0}],
                          [Rhs > 1,
                           {exp(lambertW(0, ln(Rhs)))}]),
                       options)],
                    [Rhs < exp(exp(-1)),
                     {}]));
        else
          return(solvelib::isolate(rewrite(Lhs, exp), Rhs, x,
                                   options, reclevel+1, known));
        end_if;
      else // not options[Real]
        if not has(b,x) then
          /* solving a(x)^b = c for x */
          if iszero(Rhs) then
            return(pw([b > 0,
                       solvelib::preImage(a, x, {0}, options)],
                      [not b > 0,
                       {}]));
          elif type(b) = DOM_INT then
            /* integer exponent */
            if b<0 then
              if options[IgnoreAnalyticConstraints] then
                if not is(Rhs = 0, Goal = TRUE) then
                  return(solvelib::isolate(a^(-b), 1/Rhs, x,
                                         options, reclevel+1, known)
                         )
                else
                  return({})
                end_if;
              else // not IgnoreAnalyticConstraints
                return(
                       pw([Rhs = 0,
                       solvelib::isolate(Lhs, 0, x,
                                         options, reclevel+1, known)],
                      [Rhs <> 0,
                       solvelib::isolate(a^(-b), 1/Rhs, x,
                                         options, reclevel+1, known)]
                          )
                       )
              end_if;
            end_if;
            assert(b>=0);
            return(pw([Rhs = 0,
                       solvelib::isolate(Lhs, 0, x,
                                         options, reclevel+1, known)],
                     
                      [Rhs <> 0,
                       solvelib::preImage(a, x,
                         solvelib::solve_eq(x^b-Rhs, x, options),
                         options)])
                   );
          elif options[IgnoreAnalyticConstraints] then
            // a(x)^b = c is considered equivalent to a(x) = c^(1/b)
            return(solvelib::isolate(a, Rhs^(1/b), x, options,
                                     reclevel+1, known))
          elif type(b) = DOM_RAT and (op(b, 1) = 1 or op(b, 1) = -1) then
            /* rational exponent */
            p := op(b, 1);
            q := op(b, 2);
            v := genident();
            u := solvelib::checkAngle(
                   solvelib::solve_eq(v^p-Rhs, v, options), x, x, p/q,
                   options, maxreclevel-reclevel);
            return(solvelib::preImage(a, x, Dom::ImageSet(v^q, v, u), options));
          else
            /* general case:
             * Any solution must be of the form
             *   a = c^(1/b)*exp(-2*I*PI*k/b)
             * for some integer k. These k satisfy
             *   -PI < Im(ln(c)/b-2*I*PI*k/b) <= PI.
             */
            k := genident();
            q:= pw([Rhs <> 0 and b <> 0,
                    solvelib::preImage
                    (a, x,
                     Dom::ImageSet
                     (Rhs^(1/b)*exp(-2*I*PI*k/b), k,
                      pw([Re(1/b) > 0,
                          _intersect
                          (
                           Dom::Interval
                           (
                            [(Im(ln(Rhs)/b)-PI)/2/PI/Re(1/b)],
                            (Im(ln(Rhs)/b)+PI)/2/PI/Re(1/b)),
                           Z_)],
                         [Re(1/b) < 0,
                          _intersect
                          (
                           Dom::Interval
                           (
                            (Im(ln(Rhs)/b)+PI)/2/PI/Re(1/b),
                            [(Im(ln(Rhs)/b)-PI)/2/PI/Re(1/b)]),
                           Z_)],
                         [Re(1/b) = 0 and Im(ln(Rhs)/b) in
                          Dom::Interval(-PI, [PI]),
                          Z_],
                         [Re(1/b) = 0 and not Im(ln(Rhs)/b) in
                          Dom::Interval(-PI, [PI]),
                          {}])),
                     options)],
                   [Rhs = 0,
                    solvelib::isolate(Lhs, 0, x,
                                      options, reclevel+1, known)],
                   [b = 0 and Rhs = 1,
                    C_],
                   [b = 0 and Rhs <> 1,
                    {}]);
            return(q)
          end_if;
        elif not has(a, x) then
          /* solving a^b(x) = c for x */
          if contains({DOM_INT, DOM_RAT}, type(a)) and a > 0 then
            /* integer/rational base:
             * Express the solution in terms of log.
             * a^b = c  <=>  b in log(a, c) + 2*I*PI*Z_/ln(a)
             */
            assert(a <> 1);
            if options[IgnoreAnalyticConstraints] then
              if iszero(Rhs) then
                return({})
              else
                return(solvelib::isolate(b, simplify(log(a, Rhs)), x, 
                                         options, reclevel+1, known))
              end_if;
            end_if;
            return(pw([Rhs <> 0,
                       solvelib::preImage(b, x, simplify(log(a, Rhs))+ 2*I*PI*Z_/ln(a),
                                          options)],
                      [Rhs = 0,
                       {}]));
          else
            /* general case */
            return(pw([a <> 0,
                       solvelib::isolate(rewrite(Lhs, exp), Rhs, x,
                                         options, reclevel+1, known)],
                      [a = 0,
                       solvelib::preImage(b, x,
                         pw([Rhs = 0,
                             Dom::Interval(0, infinity)],
                            [Rhs = 1,
                             {0}],
                            [Rhs <> 0 and Rhs <> 1,
                             {}]),
                         options)]));
          end_if;
        elif testeq(a, b) = TRUE then
          /* solving f(x)^f(x) = c for x */
          k := genident();
          if options[IgnoreAnalyticConstraints] then
            if iszero(Rhs) then
              return({})
            else
              return(solvelib::preImage
                     (b, x,
                      solvelib::isolate(x*ln(x), ln(Rhs), x,
                                        options, reclevel+1, known))
                     )
            end_if;
            /* Alternative: return a piecewise:
            return(pw([Rhs <> 0,
                       solvelib::preImage
                       (b, x,
                        solvelib::isolate(x*ln(x), ln(Rhs), x,
                                          options, reclevel+1, known),
                        options)
                       ],
                      [Rhs = 0, {}])
                   )
             */
          end_if;
          return(pw([Rhs <> 0,
                     solvelib::Union(
                       solvelib::preImage(b, x,
                         solvelib::isolate(x*ln(x), ln(Rhs)+2*I*PI*k, x,
                                           options, reclevel+1, known),
                         options),
                       k, Z_)],
                    [Rhs = 0,
                     {}]));
        elif iszero(Rhs) then
          /* solving a(x)^b(x) = 0 for x */
          MAXEFFORT := MAXEFFORT/3;
          return(solvelib::solve_intersect(
                   solvelib::isolate(a, 0, x, options, reclevel+1, known),
                   solve(b > 0, x, options)));
        else
          return(solvelib::isolate(rewrite(Lhs, exp), Rhs, x,
                                   options, reclevel+1, known));
        end_if;
      end_if;
      /* not reached */
      assert(FALSE);
    of "log" do
      if not has(op(Lhs, 1), x) and
        (options[Real] or options[IgnoreAnalyticConstraints] or is(Rhs in R_, Goal =TRUE)) then
        return(solvelib::isolate(op(Lhs, 2), simplify(op(Lhs, 1)^Rhs), x,
                               options, reclevel+1, known)
               )
      end_if;
      if op(Lhs, 1) = x then
        // take care that log(x, ...) is undefined unless x>0
        assume(x in R_);
        return(solvelib::solve_intersect
               (
                solvelib::isolate(rewrite(Lhs, ln), Rhs, x,
                                  options, reclevel+1, known),
                Dom::Interval(0, infinity)
                )
               )
      end_if;
      // last resort: rewrite by ln
      return(solvelib::isolate(rewrite(Lhs, ln), Rhs, x,
                               options, reclevel+1, known));
    of "ln" do
      if options[Real] or options[IgnoreAnalyticConstraints] then
        l := solvelib::isolate(op(Lhs), exp(Rhs), x,
                               options, reclevel+1, known);
        if type(l) = DOM_SET then
          return(map(l, simplify));
        else
          return(l);
        end_if
      elif type(Rhs) = "ln" then
        return(
               solvelib::isolate(op(Lhs), op(Rhs, 1), x,
                                 options, reclevel+1, known)
               )
      else
        return(pw([Im(Rhs) in Dom::Interval(-PI, [PI]),
                   solvelib::isolate(op(Lhs), exp(Rhs), x,
                                     options, reclevel+1, known)],
                  [not Im(Rhs) in Dom::Interval(-PI, [PI]),
                   {}]));
      end_if;
    of "exp" do
      if options[IgnoreAnalyticConstraints] then
        if iszero(Rhs) then
          return({})
        else
          s:= solvelib::isolate(op(Lhs), Ln(Rhs), x,
                                     options, reclevel+1, known);
          if has(s, hold(solve)) then
             return(s)
          else
             return(simplify(s))        
          end_if
        end_if;
        /* an alternative would be:
          return(pw([Rhs <> 0,
                   solvelib::isolate(op(Lhs), ln(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs = 0, {}])
                 )
        */
      end_if;
      if options[Real] or is(op(Lhs) in R_, Goal = TRUE) then
        /* real case:
         * exp(a) = b  <=>  a = ln(b)
         */
        return(pw([Rhs > 0,
                   solvelib::isolate(op(Lhs), Ln(Rhs), x,
                                     options, reclevel+1, known)],
                  [not Rhs > 0,
                   {}])
               )
      else
        /* general case:
         * exp(a) = b  <=>  a in ln(b) + 2*I*PI*Z_
         */
        return(pw([Rhs <> 0,
                   solvelib::Union(solvelib::isolate(op(Lhs), Ln(Rhs) + 2*#k*I*PI, x, options, reclevel+1, known), #k, Z_)
                   ],
                  [Rhs = 0,
                   {}])
               )
      end_if;
    of "lambertW" do
      assert(nops(Lhs) = 2);
      /* Solving lambertW(a, b) = c for either a or b.
       */
      if options[IgnoreAnalyticConstraints] then
        // ignore a, and make W(b) = c into
        // b = c*exp(c)
        return(solvelib::isolate(op(Lhs, 2), Rhs*exp(Rhs), x,
                                 options, reclevel+1, known))
      end_if;
        
      
      /*
       * Otherwise, only the special case where c is real is implemented.
       */
      if options[Real] or is(Rhs in R_) = TRUE then
        a := op(Lhs, 1);
        b := op(Lhs, 2);
        if not has(b, x) then
          /* solving for a */
          return(pw([Rhs = -1 and b = -exp(-1),
                     solvelib::preImage(a, x, {0, -1}, options)],
                    [Rhs > -1 and b = Rhs*exp(Rhs),
                     solvelib::isolate(a, 0, x,
                                       options, reclevel+1, known)],
                    [Rhs < -1 and b = Rhs*exp(Rhs),
                     solvelib::isolate(a, -1, x,
                                       options, reclevel+1, known)],
                    [b <> Rhs*exp(Rhs),
                     {}]));
        elif not has(a, x) then
          /* solving for b */
          return(pw([a = 0 and Rhs >= -1 or a = -1 and Rhs <= -1,
                     solvelib::isolate(b, Rhs*exp(Rhs), x,
                                       options, reclevel+1, known)],
                    [not (a = 0 and Rhs >= -1 or a = -1 and Rhs <= -1),
                     {}]));
        end_if;
      end_if;
      /* c is not real or both a and b contain x. */
      return(hold(solve)(Lhs = Rhs, x,
                         solvelib::nonDefaultOptions(options)));
    of "wrightOmega" do
      if options[IgnoreAnalyticConstraints] then
        if iszero(Rhs) then
          return({})
        else
          return(
                 solvelib::isolate(op(Lhs), Rhs+ln(Rhs), x,
                                   options, reclevel+1, known)
                 )
        end_if
      elif options[Real] then
        return(pw([Rhs > 0,
                   solvelib::isolate(op(Lhs), Rhs+ln(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs <= 0,
                   {}]));
      else
        return(pw([not Rhs <= -1 and Rhs <> 0,
                   solvelib::isolate(op(Lhs), Rhs+ln(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs < -1,
                   solvelib::isolate(op(Lhs), Rhs+ln(Rhs)-2*PI*I, x,
                                     options, reclevel+1, known)],
                  [Rhs = -1,
                   solvelib::preImage(op(Lhs), x, {-1+I*PI, -1-I*PI},
                                      options)],
                  [Rhs = 0,
                   {}]));
      end_if;
    of "cos" do
      if options[IgnoreAnalyticConstraints] then
        return(solvelib::isolate(op(Lhs), Arccos(Rhs), x, options, reclevel+1,
                                  known))
      end_if;
      if Rhs = 0 then
        /* _union could work better here. */
        return(solvelib::preImage(op(Lhs), x, PI/2+PI*Z_, options));
      elif Rhs = -1 then
        /* _union does not work here. */
        return(solvelib::preImage(op(Lhs), x, PI+2*PI*Z_, options));
      elif is(op(Lhs) in Dom::Interval([0, PI]), Goal = TRUE) then
        // intersection with getprop(..) is done later
        return(solvelib::isolate(op(Lhs), arccos(Rhs), x, options, reclevel+1,
                                  known))
      else
        /* cos(a) = b  <=>  a in {arccos(b), -arccos(b)} + 2*PI*Z_
         */
        MAXEFFORT := MAXEFFORT/3;
        l := _union(solvelib::preImage(op(Lhs), x,
                      Arccos(Rhs)+2*PI*Z_, options),
                    solvelib::preImage(op(Lhs), x,
                      -Arccos(Rhs)+2*PI*Z_, options));
        if options[Real] then
          return(pw([Rhs in Dom::Interval([-1], [1]), l],
                    [not Rhs in Dom::Interval([-1], [1]), {}]));
        else
          return(l);
        end_if;
      end_if;
    of "sin" do
      if options[IgnoreAnalyticConstraints] then
        return(solvelib::isolate(op(Lhs), Arcsin(Rhs), x,
                                 options, reclevel+1, known))
      end_if;
      if Rhs = 0 then
        /* _union could work better here. */
        return(solvelib::preImage(op(Lhs), x, PI*Z_, options));
      elif Rhs = -1 then
        /* _union does not work here. */
        return(solvelib::preImage(op(Lhs), x, -PI/2+2*PI*Z_, options));
      elif is(op(Lhs) in Dom::Interval([-PI/2, PI/2]), Goal = TRUE) then
        // intersection with getprop(..) is done later
        return(solvelib::isolate(op(Lhs), arcsin(Rhs), x, options, reclevel+1,
                                 known))
      else
        /* sin(a) = b  <=>  a in {arcsin(b), PI-arcsin(b)} + 2*PI*Z_
         */
        MAXEFFORT := MAXEFFORT/3;
        l := _union(solvelib::preImage(op(Lhs), x,
                      Arcsin(Rhs)+2*PI*Z_, options),
                    solvelib::preImage(op(Lhs), x,
                      PI-Arcsin(Rhs)+2*PI*Z_, options));
        if options[Real] then
          return(pw([Rhs in Dom::Interval([-1], [1]), l],
                    [not Rhs in Dom::Interval([-1], [1]), {}]));
        else
          return(l);
        end_if;
      end_if;
    of "tan" do
      if options[IgnoreAnalyticConstraints] then
        if traperror((l:= Arctan(Rhs))) <> 0 then
          return({})
        end_if;
        return(solvelib::isolate(op(Lhs), l, x, options, reclevel+1,
                                  known))
      end_if;
      
      /* tan(a) = b  <=>  a in arctan(b) + PI*Z_
       */
      if options[Real] then
        l := solvelib::preImage(op(Lhs), x, Arctan(Rhs)+PI*Z_, options);
        /* simplify arctan(expression containing radicals) */
        case type(l)
          of DOM_SET do
          of Dom::ImageSet do
            return(map(l, eval@radsimp))
          otherwise
            return(l)
        end_case;
      else
        return(pw([Rhs <> I and Rhs <> -I,
                   solvelib::preImage(op(Lhs), x, Arctan(Rhs)+PI*Z_,
                                      options)],
                  [Rhs = I or Rhs = -I,
                   {}]));
      end_if;
    of "cot" do
      if options[IgnoreAnalyticConstraints] then
        if traperror((l:= arccot(Rhs))) <> 0 then
          return({})
        end_if;
        return(solvelib::isolate(op(Lhs), l, x, options, reclevel+1,
                                  known))
      end_if;
      
      /* cot(a) = b  <=>  a in arccot(b) + PI*Z_
       */
      if options[Real] then
        return(solvelib::preImage(op(Lhs), x, arccot(Rhs)+PI*Z_, options));
      else
        return(pw([Rhs <> I and Rhs <> -I,
                   solvelib::preImage(op(Lhs), x, arccot(Rhs)+PI*Z_,
                                      options)],
                  [Rhs = I or Rhs = -I,
                   {}]));
      end_if;
    of "arccos" do
      l := solvelib::isolate(op(Lhs), cos(Rhs), x,
                             options, reclevel+1, known);
      if options[IgnoreAnalyticConstraints] then
        return(l)
      end_if;
      if options[Real] then
        return(pw([Rhs >= 0 and Rhs <= PI, l],
                  [Rhs < 0 or Rhs > PI, {}]));
      else
        return(pw([Re(Rhs) > 0 and Re(Rhs) <= PI or Re(Rhs) = 0 and Im(Rhs) >= 0, l],
                  [Re(Rhs) < 0 or Re(Rhs) > PI or Re(Rhs) = 0 and Im(Rhs) < 0, {}]));
      end_if;
    of "arcsin" do
      l := solvelib::isolate(op(Lhs), sin(Rhs), x,
                             options, reclevel+1, known);
      if options[IgnoreAnalyticConstraints] then
        return(l)
      end_if;
      if options[Real] then
        return(pw([Rhs >= -PI/2 and Rhs <= PI/2, l],
                  [Rhs < -PI/2 or Rhs > PI/2, {}]));
      else
        return(pw([Re(Rhs) >= -PI/2 and Re(Rhs) <= PI/2, l],
                  [Re(Rhs) < -PI/2 or Re(Rhs) > PI/2, {}]));
      end_if;
    of "arctan" do
      u:= op(Lhs, 1);
      if options[IgnoreAnalyticConstraints] then
        if traperror((l:= tan(Rhs))) <> 0 then
          return({})
        else
          return(
                 solvelib::isolate(u, l, x,
                                   options, reclevel+1, known)
                 )
        end_if
      end_if;
      return(pw([not (Rhs+PI/2)/PI in Z_,
                 solvelib::isolate(u, tan(Rhs), x,
                                   options, reclevel+1, known)],
                [(Rhs+PI/2)/PI in Z_,
                 {}]));
    of "arccot" do
      u:= op(Lhs, 1);
      if options[IgnoreAnalyticConstraints] then
        if traperror((l:= cot(Rhs))) <> 0 then
          return({})
        else
          return(
                 solvelib::isolate(u, l, x,
                                   options, reclevel+1, known)
                 )
        end_if
      end_if;
      
      return(pw([not Rhs/PI in Z_,
                 solvelib::isolate(u, cot(Rhs), x,
                                   options, reclevel+1, known)],
                [Rhs/PI in Z_,
                 {}]));
    of "cosh" do
      if options[Real] or options[IgnoreAnalyticConstraints] or
        is(op(Lhs) in R_, Goal = TRUE) then
        /* real case:
         * cosh(a) = b  <=>  a in {arccosh(b), -arccosh(b)} intersect R_
         */
        MAXEFFORT := MAXEFFORT/3;
        l := _union(solvelib::isolate(op(Lhs), arccosh(Rhs), x,
                                      options, reclevel+1, known),
                    solvelib::isolate(op(Lhs), -arccosh(Rhs), x,
                                      options, reclevel+1, known));
        if options[IgnoreAnalyticConstraints] or is(op(Lhs) in R_, Goal = TRUE) then
          return(l)
        else
          /* arccosh(b) is real iff b >= 1. */
          return(pw([Rhs >= 1, l],
                    [Rhs < 1, {}]));
        end_if;
      else
        /* general case:
         * cosh(a) = b  <=>  a in {arccosh(b), -arccosh(b)} + 2*I*PI*Z_
         */
        MAXEFFORT := MAXEFFORT/3;
        return(_union(solvelib::preImage(op(Lhs), x,
                        arccosh(Rhs)+2*I*PI*Z_, options),
                      solvelib::preImage(op(Lhs), x,
                        -arccosh(Rhs)+2*I*PI*Z_, options)));
      end_if;
    of "sinh" do
      if options[Real] or options[IgnoreAnalyticConstraints] or
        is(op(Lhs), Type::Real) = TRUE then
        /* real case:
         * sinh(a) = b  <=>  a in {arcsinh(b)} intersect R_
         */
        l := solvelib::isolate(op(Lhs), arcsinh(Rhs), x,
                               options, reclevel+1, known);
        if options[IgnoreAnalyticConstraints] or is(op(Lhs) in R_, Goal = TRUE) then
          return(l)
        else
          /* arcsinh(b) is real iff b is real. */
          return(pw([Rhs in R_, l],
                    [not Rhs in R_, {}]));
        end_if;
      else
        /* general case:
         * sinh(a) = b  <=>  a in arcsinh(b) + I*PI*Z_
         */
        return(solvelib::preImage(op(Lhs), x, arcsinh(Rhs)+I*PI*Z_,
                                  options));
      end_if;
    of "tanh" do
      if options[IgnoreAnalyticConstraints] then
        if Rhs = -1 or Rhs = 1 then
          return({})
        else
          return(solvelib::isolate(op(Lhs), arctanh(Rhs), x,
                                     options, reclevel+1, known))
        end_if
      end_if;
      if options[Real] then
        /* arctanh(b) is real iff -1 < b < 1. */
        return(pw([Rhs > -1 and Rhs < 1,
                   solvelib::isolate(op(Lhs), arctanh(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs <= -1 or Rhs >= 1,
                   {}]));
      else
        return(solvelib::isolate(rewrite(Lhs, sincos), Rhs, x,
                                 options, reclevel+1, known));
      end_if;
    of "coth" do
      if options[IgnoreAnalyticConstraints] then
        if Rhs = -1 or Rhs = 1 then
          return({})
        else
          return(solvelib::isolate(op(Lhs), arccoth(Rhs), x,
                                     options, reclevel+1, known))
        end_if
      end_if;
      if options[Real] then
        /* arccoth(b) is real iff b < -1 or b > 1. */
        return(pw([Rhs < -1 or Rhs > 1,
                   solvelib::isolate(op(Lhs), arccoth(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs >= -1 and Rhs <= 1,
                   {}]));
      else
        return(solvelib::isolate(rewrite(Lhs, sincos), Rhs, x,
                                 options, reclevel+1, known));
      end_if;
    of "arccosh" do
      return(solvelib::isolate(op(Lhs), cosh(Rhs), x,
                               options, reclevel+1, known));
    of "arcsinh" do
      return(solvelib::isolate(op(Lhs), sinh(Rhs), x,
                               options, reclevel+1, known));
    of "arctanh" do
      if options[Real] or options[IgnoreAnalyticConstraints] then
        return(solvelib::isolate(op(Lhs), tanh(Rhs), x,
                                 options, reclevel+1, known));
      else
        ar := genident("k");
        return(pw([not Rhs in Dom::ImageSet(I*PI*(2*ar+1)/2, ar, Z_),
                   solvelib::isolate(op(Lhs), tanh(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs in Dom::ImageSet(I*PI*(2*ar+1)/2, ar, Z_),
                   {}]));
      end_if;
    of "arccoth" do
      if options[IgnoreAnalyticConstraints] then
        if iszero(Rhs) then
          return({})
        else
          return(solvelib::isolate(op(Lhs), coth(Rhs), x,
                                     options, reclevel+1, known
                                     )
                )
        end_if                                                    
      elif options[Real] then
        return(pw([Rhs <> 0,
                   solvelib::isolate(op(Lhs), coth(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs = 0,
                   {}]));
      else
        ar := genident("k");
        return(pw([not Rhs in Dom::ImageSet(I*PI*ar, ar, Z_),
                   solvelib::isolate(op(Lhs), coth(Rhs), x,
                                     options, reclevel+1, known)],
                  [Rhs in Dom::ImageSet(I*PI*ar, ar, Z_),
                   {}]));
      end_if;
    of "ellipticF" do
      assert(nops(Lhs) = 2);
      a := op(Lhs, 1);
      b := op(Lhs, 2);
      if not has(b, x) then
        if options[Real] then
          return(pw([b <= 1,
                     solvelib::isolate(a, jacobiAM(Rhs, b), x,
                                       options, reclevel+1, known)],
                    [b > 1,
                     /* FIXME: needs further research */
                     hold(solve)(Lhs = Rhs, x,
                                 solvelib::nonDefaultOptions(options))]));
        else
          return(solvelib::isolate(a, jacobiAM(Rhs, b), x,
                                   options, reclevel+1, known));
        end_if;
      else
        return(hold(solve)(Lhs = Rhs, x,
                           solvelib::nonDefaultOptions(options)));
      end_if;
    of "jacobiAM" do
      assert(nops(Lhs) = 2);
      a := op(Lhs, 1);
      b := op(Lhs, 2);
      if not has(b, x) and is(b <= 1) = TRUE and is(Rhs in R_) = TRUE then
        return(solvelib::isolate(a, ellipticF(Rhs, b), x,
                                 options, reclevel+1, known));
      else
        return(hold(solve)(Lhs = Rhs, x,
                           solvelib::nonDefaultOptions(options)));
      end_if;
    of "arg" do
      assert(nops(Lhs) = 1);
      l:= op(Lhs, 1);
      if options[Real] then
        return(pw([Rhs = 0,
                   solvelib::preImage(l, x,
                     Dom::Interval([0], infinity), options)],
                  [Rhs = PI,
                   solvelib::preImage(l, x,
                     Dom::Interval(-infinity, 0), options)],
                  [Rhs <> 0 and Rhs <> PI,
                   {}]));
      else
        return(pw([Rhs = 0, /* note: arg(0) = 0 */
                   solvelib::preImage(l, x,
                     Dom::Interval([0], infinity), options)],
                  [Rhs > -PI and Rhs <> 0 and Rhs <= PI,
                   solvelib::preImage(l, x,
                     exp(I*Rhs)*Dom::Interval(0, infinity), options)],
                  [not Rhs > -PI or not Rhs <= PI,
                   {}]));
      end_if;
    of "abs" do
      if options[Real] or is(op(Lhs), Type::Real) = TRUE then
        /* real case:
         * abs(a) = b  <=>  a in {-b, -b}
         */
        MAXEFFORT := MAXEFFORT/3;
        l := _union(solvelib::isolate(op(Lhs), Rhs, x,
                                      options, reclevel+1, known),
                    solvelib::isolate(-op(Lhs), Rhs, x,
                                      options, reclevel+1, known));
        if options[Real] then
          return(pw([Rhs >= 0, l], [Rhs < 0, {}]));
        else
          return(pw([Rhs >= 0, l], [not Rhs >= 0, {}]));
        end_if;
      else
        /* general case:
         * abs(a) = b  <=>  a in {b*exp(2*PI*I*alpha) | alpha in [0, 1]}
         */
        ar := genident("alpha");
        return(pw([Rhs >= 0,
                   solvelib::preImage(op(Lhs), x,
                     Dom::ImageSet(Rhs*exp(2*PI*I*ar), ar,
                                   Dom::Interval([0], [1])), options)],
                  [not Rhs >= 0,
                   {}]));
      end_if;
    of "sign" do
      l := solvelib::preImage(op(Lhs), x, Rhs*Dom::Interval(0, infinity),
                              options);
      if options[Real] then
        return(pw([Rhs = -1 or Rhs = 0 or Rhs = 1, l],
                  [Rhs <> -1 and Rhs <> 0 and Rhs <> 1, {}]));
      else
        ar := genident("alpha");
        s  := _union(Dom::ImageSet(exp(2*PI*I*ar), ar,
                                   Dom::Interval([0], [1])),
                     {0});
        return(pw([Rhs in s, l],
                  [not Rhs in s, {}]));
      end_if;
    of "signIm" do
      if options[Real] or is(op(Lhs, 1) in R_, Goal = TRUE) then
        l := {};
      elif MAXEFFORT <= 1000 then
        return(hold(solve)(Lhs = Rhs, x,
                         solvelib::nonDefaultOptions(options)))
      else
        u := genident();
        v := genident();
        l := Dom::ImageSet(u+v*I, [u, v], [R_, Dom::Interval(0, infinity)]);
      end_if;
      return(pw([Rhs in {-1, 1},
                 solvelib::preImage(op(Lhs), x,
                   _union(Rhs*l,
                          Rhs*Dom::Interval(-infinity, 0)),
                   options)],
                [Rhs = 0,
                 solvelib::isolate(op(Lhs), 0, x,
                                   options, reclevel+1, known)],
                [not Rhs in {-1, 0, 1},
                 {}]));
    of "Im" do
      if options[Real] then
        return(pw([Rhs = 0,
                   solvelib::preImage(op(Lhs), x, R_, options)],
                  [Rhs <> 0,
                   {}]));
      else
        /* Im(a) = b  <=>  a in {I*b+lambda; lambda in R_}
         */
        return(pw([Rhs in R_,
                   solvelib::preImage(op(Lhs), x, I*Rhs+R_)],
                  [not Rhs in R_,
                   {}]));
      end_if;
    of "Re" do
      if options[Real] then
        return(solvelib::isolate(op(Lhs), Rhs, x,
                                   options, reclevel+1, known))
      else
        /* Re(a) = b  <=>  a in {b+I*lambda; lambda in R_}
         */
        return(pw([Rhs in R_,
                   solvelib::preImage(op(Lhs), x, Rhs+I*R_)],
                  [not Rhs in R_,
                   {}]));
      end_if;
    of "ceil" do
    of "floor" do
    of "round" do
    of "trunc" do
      if options[Real] then
        /* real case:
         * solution is a half-open interval
         */
        if is(Rhs in R_) = FALSE then
          return({});
        else
          case type(Lhs)
            of "ceil" do
              a := Dom::Interval(Rhs-1, [Rhs]);
              break;
            of "floor" do
              a := Dom::Interval([Rhs], Rhs+1);
              break;
            of "round" do
              a := Dom::Interval([Rhs-1/2], Rhs+1/2);
              break;
            of "trunc" do
              a := pw([Rhs < 0, /* ceil case */
                       Dom::Interval(Rhs-1, [Rhs])],
                      [Rhs > 0, /* floor case */
                       Dom::Interval([Rhs], Rhs+1)],
                      [Rhs = 0,
                       Dom::Interval(-1, 1)]);
              break;
            otherwise
              /* not reached */
              assert(FALSE);
          end_case;
          return(pw([Rhs in Z_,
                     solvelib::preImage(op(Lhs), x, a, options)],
                    [not Rhs in Z_,
                     {}]));
        end_if;
      else
        /* general case:
         * solution is a 1x1-square in C_
         */
        case type(Lhs)
          of "ceil" do
            a := Dom::Interval(Re(Rhs)-1, [Re(Rhs)]);
            b := Dom::Interval(Im(Rhs)-1, [Im(Rhs)]);
            break;
          of "floor" do
            a := Dom::Interval([Re(Rhs)], Re(Rhs)+1);
            b := Dom::Interval([Im(Rhs)], Im(Rhs)+1);
            break;
          of "round" do
            a := Dom::Interval([Re(Rhs)-1/2], Re(Rhs)+1/2);
            b := Dom::Interval([Im(Rhs)-1/2], Im(Rhs)+1/2);
            break;
          of "trunc" do
            a := pw([Re(Rhs) < 0, /* ceil case */
                     Dom::Interval(Re(Rhs)-1, [Re(Rhs)])],
                    [Re(Rhs) > 0, /* floor case */
                     Dom::Interval([Re(Rhs)], Re(Rhs)+1)],
                    [Re(Rhs) = 0,
                     Dom::Interval(-1, 1)]);
            b := pw([Im(Rhs) < 0, /* ceil case */
                     Dom::Interval(Im(Rhs)-1, [Im(Rhs)])],
                    [Im(Rhs) > 0, /* floor case */
                     Dom::Interval([Im(Rhs)], Im(Rhs)+1)],
                    [Im(Rhs) = 0,
                     Dom::Interval(-1, 1)]);
            break;
          otherwise
            /* not reached */
            assert(FALSE);
        end_case;
        u := genident();
        v := genident();
        return(pw([Re(Rhs) in Z_ and Im(Rhs) in Z_,
                   solvelib::preImage(op(Lhs), x,
                     Dom::ImageSet(u+I*v, [u, v], [a,b]), options)],
                  [not Re(Rhs) in Z_ or not Im(Rhs) in Z_,
                   {}]));
      end_if;
      /* not reached */
      assert(FALSE);
    of "fact" do
      if is(Rhs in Z_ minus {0}) = FALSE then
        return({});
      elif iszero(Rhs-1) then
        /* note: 0! = 1! = 1 */
        return(solvelib::preImage(op(Lhs), x, {0, 1}, options));
      elif type(Rhs) = DOM_INT then
        n := Rhs;
        for i from 2 to Rhs do
          n := n/i;
          if type(n) <> DOM_INT then
            break;
          elif n = 1 then
            return(solvelib::isolate(op(Lhs), i, x,
                                     options, reclevel+1, known));
          end_if;
        end_for;
        return({});
      else
        return(hold(solve)(Lhs = Rhs, x,
                           solvelib::nonDefaultOptions(options)));
      end_if;
    of "gamma" do
      if is(Rhs in Z_) = TRUE then
        /* For k in Z_, we use gamma(k) = fact(k-1).
         */
        return(solvelib::isolate(fact(op(Lhs)-1), Rhs, x,
                                 options, reclevel+1, known));
      else
        /* Otherwise, we give up.
         */
        return(hold(solve)(Lhs = Rhs, x,
                           solvelib::nonDefaultOptions(options)));
      end_if;
    of "heaviside" do
      return(pw([Rhs = 0,
                 solvelib::preImage(op(Lhs), x,
                                    Dom::Interval(-infinity, 0), options)],
                [Rhs = 1,
                 solvelib::preImage(op(Lhs), x,
                                    Dom::Interval(0, infinity), options)],
                [Rhs = 1/2,
                 solvelib::isolate(op(Lhs), 0, x,
                                   options, reclevel+1, known)],
                [Rhs <> 0 and Rhs <> 1 and Rhs <> 1/2,
                 {}]));
    of "binomial" do
    of "conjugate" do
      /* handle below by rewriting */
      break;
    of piecewise do
      return(piecewise::extmap(Lhs, solvelib::isolate, args(2..args(0))));
    otherwise
      /* cannot solve f(x) = C if f is unknown */
      return(hold(solve)(Lhs = Rhs, x,
                         solvelib::nonDefaultOptions(options)));
  end_case;

 
   /*
     if Lhs contains some expressions like cos(x/n),
     then substitute y:= x/n, solve for y and substitute back
   */
   s:= select(indets(Lhs, RatExpr),
              ind -> has(ind, x) and contains({"sin", "cos", "tan"},
                                              type(ind))
              );
   s:= map(s, op, 1);
   s:= gcd(op(s));
   if type(s) = "_mult" and op(s, 1) = x and type((q:= op(s, 2))) = DOM_RAT then
     u:= genident("YINTERN");
     MAXEFFORT:= MAXEFFORT*3/4;
     p:= subsex(Lhs, x=u*denom(q));
     u:= solvelib::isolate(p, Rhs, u, options, reclevel+1, known);
     if not has(u, hold(solve)) then
       return(solvelib::preImage(x/denom(q), x, u, options))
     end_if;
   end_if;

   
     


  /* remove calls to abs, sign, signIm, heaviside, binomial, min and max
   */
  Lhs := eval(misc::maprec(Lhs,
                {"abs"} = proc(t)
                  name rewriteAbs;
                  local a;
                begin
                  a := op(t);
                  if has(a, x) then
                    if options[Real] or is(a, Type::Real) = TRUE then
                      return(piecewise([a < 0, -a], [a >= 0, a]));
                    else
                      return(sqrt(Re(a)^2+Im(a)^2));
                    end_if;
                  else
                    return(t);
                  end_if;
                end_proc,
                {"sign"} = proc(t)
                  name rewriteSign;
                  local a;
                begin
                  a := op(t);
                  if has(a, x) then
                    if options[Real] or is(a, Type::Real) = TRUE then
                      return(piecewise([a < 0, -1], [a = 0, 0], [a > 0, 1]));
                    else
                      return(piecewise([a <> 0,
                                        a/sqrt(Re(a)^2+Im(a)^2)],
                                       [a = 0,
                                        0]));
                    end_if;
                  else
                    return(t);
                  end_if;
                end_proc,
                {"signIm"} = proc(t)
                  name rewriteSignIm;
                  local a;
                begin
                  a := op(t);
                  if has(a, x) then
                    if options[Real] or is(a, Type::Real) = TRUE then
                      return(piecewise([a < 0, 1], [a = 0, 0], [a > 0, -1]));
                    else
                      return(piecewise([a < 0 or Im(a) > 0,
                                        1],
                                       [a > 0 or Im(a) < 0,
                                        -1],
                                       [a = 0,
                                        0]));
                    end_if;
                  else
                    return(t);
                  end_if;
                end_proc,
                {"heaviside"} = proc(t)
                  name rewriteHeaviside;
                  local a;
                begin
                  a := op(t);
                  if has(a, x) then
                    return(piecewise([a < 0, 0], [a = 0, 1/2], [a > 0, 1]));
                  else
                    return(t);
                  end_if;
                end_proc,
                {"binomial"} = proc(t)
                  name rewriteBinomial;
                begin
                  if has(t, x) then
                    return(rewrite(t, gamma));
                  else
                    return(t);
                  end_if;
                end_proc,
                {"min"} = proc(t)
                  name rewriteMin;
                  local argv, i: DOM_INT, j: DOM_INT;
                begin
                  argv := split([op(t)], has, x);
                  if argv[1] = [] then
                    /* do no case analysis for arguments that do
                     * not contain x
                     */
                    return(t);
                  elif argv[2] = [] then
                    argv := argv[1];
                  else
                    argv := argv[1].[min(op(argv[2]))];
                  end_if;
                  return(piecewise([_and(argv[i] <= argv[j]
                                         $ j = 1..nops(argv)),
                                    argv[i]]
                                   $ i = 1..nops(argv)));
                end_proc,
                {"max"} = proc(t)
                  name rewriteMax;
                  local argv, i: DOM_INT, j: DOM_INT;
                begin
                  argv := split([op(t)], has, x);
                  if argv[1] = [] then
                    /* do no case analysis for arguments that do
                     * not contain x
                     */
                    return(t);
                  elif argv[2] = [] then
                    argv := argv[1];
                  else
                    argv := argv[1].[max(op(argv[2]))];
                  end_if;
                  return(piecewise([_and(argv[i] >= argv[j]
                                         $ j = 1..nops(argv)),
                                    argv[i]]
                                   $ i = 1..nops(argv)));
                end_proc));

  if type(Lhs) = piecewise then
    return(piecewise::solve(piecewise::extmap(Lhs, _subtract, Rhs), x,
                            options));
  end_if;

  if Lhs = undefined then 
    return({})
  end_if;

  if hastype(Lhs, piecewise) then
    // give up
    return(hold(solve)(lhside=Rhs, x, solvelib::nonDefaultOptions(options)))
  end_if;

  if options[Real] then
    /* real case:
     * remove calls to Im, Re and conjugate
     */
    Lhs := misc::maprec(Lhs, {"Im"} = 0, {"Re", "conjugate"} = op);
  else
    /* general case:
     * expand expressions like Re(x^2) using rectform
     */
    Lhs :=  misc::maprec(Lhs,
              {"Im", "Re", "conjugate"} = proc(t)
                name applyRectform;
              begin
                if has(t, x) then
                  return(expr(rectform(t)));
                else
                  return(t);
                end_if;
              end_proc);
    userinfo(10, "Applying rectform to certain special functions gives ".
                 expr2text(Lhs)." = ".expr2text(Rhs));
  end_if;



  ////////////////////////////////////////////////////////////////////
  ///  handle equations that contain Re or Im
  ///////////////////////////////////////////////////////////////////

  if has(Lhs, hold(Re)(x)) or has(Lhs, hold(Im)(x)) then
    // substitute both Re(x) and Im(x) by new variables u and v, x=u+Iv
    [u, v]:= solvelib::getMultipleIdent(R_, 2, indets(Lhs) union indets(Rhs));
    // assume does not evaluate its arguments, we need a trick
    assume(u in R_);
    assume(v in R_);
    // immediately substitute Re(x) and Im(x) in order
    // to avoid the computation of Re(u+I*v) etc.
    Lhs:= subs(Lhs, hold(Re)(x)=u, hold(Im)(x)=v);
    Lhs:= subs(Lhs, x=u+I*v,EvalChanges);
    userinfo(10, "Substituting ".expr2text(hold(Re)(x))." by ".expr2text(u).
             " and ".expr2text(hold(Im)(x))." by ".expr2text(v)." gives ".
             expr2text(Lhs=Rhs));
    MAXEFFORT:= MAXEFFORT/2;
    opt:= options;
    opt[BackSubstitution]:= TRUE;
    opt[VectorFormat]:= TRUE;
    opt[IgnoreProperties]:= FALSE;
    // test whether the system can be split into real and imaginary part
    s:= [Re(Lhs-Rhs), Im(Lhs-Rhs)];
    // work in a local framework where all assumptions on x are removed
    // this is important in cases where properties of x contain the parameters of the equation
    // otherwise, x might appear as a free indentifier in the solution
    proc()
    begin
      unassume(x);
      if not hastype(s, {"Re", "Im"}) then
        l:=solve(s, [u,v], opt)
      else
        l:=solve(Lhs-Rhs, [u,v], opt);
      end_if;
    end_proc();  
    userinfo(10, "Solving the system gives ".expr2text(l));
    if has(l, hold(solve)) then
      return(hold(solve)(lhside=Rhs, x, solvelib::nonDefaultOptions(options)))
    end_if;

    // l contains solutions (u_0,v_0) for u and v; for each such solution,
    // u_0 + I*v_0 is a solution of the orig. equation for x

    assert(type(l) <> "_in");

    // local method R2toC
    // maps [u, v] -> u+I*v to a given set of vectors
    // throws away vectors whose components are not real

    R2toC:=
    proc(set)
      local f, S, x;
    begin
      case type(set)
        of piecewise do
          return(piecewise::extmap(set, R2toC))
        of DOM_SET do
          map(set, proc(t)
                     begin
                       piecewise([t[1] in R_ and t[2] in R_, {t[1] + I*t[2]}],
                                 [not t[1] in R_ or not t[2] in R_, {}])
                   end_proc
              );
          return(_union(op(%)))
        of Dom::ImageSet do
        of solvelib::VectorImageSet do
          f:= expr(set);
          if nops(set::dom::variables(set))=1 and type(set::dom::sets(set)[1]) <> "_intersect" then
            x:= set::dom::variables(set)[1];
            S:= (solve(f[1] in R_, x) intersect solve(f[2] in R_, x))
                assuming (x in set::dom::sets(set)[1], _and);
            return(Dom::ImageSet(f[1] + I*f[2], x, S))
          end_if;
          // hope for the best ..
          return(Dom::ImageSet(f[1] + I*f[2],
                               set::dom::variables(set),
                               set::dom::sets(set)
                               )
                 )
        of "Union" do
          return(solvelib::Union(R2toC(op(set, 1)), op(set, 2..3)))
        of "_union" do
        of "_intersect" do
        of "_minus" do
          return(map(set, R2toC))
        of solvelib::cartesianPower do
          assert(op(set, 1) = R_ and op(set, 2) = 2);
          return(C_)
      end_case;
      // NOT REACHED
      assert(FALSE);
      return(FAIL)
    end_proc;

    l:= R2toC(l);
    // should not happen:
    if l = FAIL then
      return(hold(solve)(lhside=Rhs, x))
    end_if;


    userinfo(10, "Back substituting gives ".expr2text(l));
    // indeterminates in the solution must be understood as
    // ranging over all reals
    if (a:=indets(l) intersect {u,v})={} then
      return(l)
    elif nops(a)=2 then
      // should not happen !
      return(solvelib::Union(solvelib::Union(l, u, R_), v, R_))
    else
      return(solvelib::Union(l, op(a), R_))
    end_if;
  end_if;


 
  
  
  //////////////////////////////////////////////////////////////////////
  ///  Rewrite by system
  //
  // try to see the expression Lhs as an element of a field
  // extension of Q(x)
  //////////////////////////////////////////////////////////////////////

  if options[DontRewriteBySystem] = FALSE and MAXEFFORT > 5000 then
    l:= solvelib::makeSystem({Lhs - Rhs}, [x]);
    if nops(op(l,3))>1 then
    // there was at least one variable created by a substitution
      vars:= op(l, 3);
      newvars:= select(vars, _unequal, x);
      userinfo(10, "Newly created variables are :".expr2text(newvars));
//      a:=op(l,4);
//      userinfo(10, "Conditions are ".expr2text(a));
      u:=subs(op(l,2), `#I` = I);
      userinfo(10, "Nonalgebraic part is ".expr2text(u));
      l:=op(l,1), op(l,3);
      userinfo(10, "Algebraic system is  ".expr2text(l));

      // did we have to rewrite radicals as extra equations ?
      roots:= bool(nops(op(l, 1)) > 1);


      // choose variable to solve for
      case nops(u)
        of 0 do
          v:=x;
          newlhs:= Lhs;
          break
        of 1 do
          // only one non-alg. condition
          v:=lhs(op(u,1));
          newlhs:= subs(Lhs, rhs(op(u,1)) = v)
      end_case;

      if nops(u)<= 1 then
        userinfo(10, "Solving the system for ".expr2text(v))
      else
        userinfo(10, "Cannot solve system with more than one non-algebraic".
                 " equation")
      end_if;

      if nops(u)=0
        // purely algebraic case
        or nops(u)=1 and not has(op(l,1),x) then
      // not really an eq. in x, but in f(x) (with f(x)=rhs(op(u,1)) )
        options[VectorFormat]:= TRUE;
        options[BackSubstitution]:= TRUE;
        MAXEFFORT:= MAXEFFORT/4;
        l:=solve(l, options);
        userinfo(10, "Solving gives ".expr2text(l));
        
      // now l is the set of all tuples [v, var1, var2, ...]
      // that solve the system created by makeSystem
      // (but possibly, there are too many solutions)
      // Here, var1,  var2, etc. are variables newly created by makeSystem
      // We apply a projection to the variable v:

        if type(l) <> "solve" then
          l:= solvelib::selectIndices(l, [contains(vars, v)])
        else
          l:= FAIL
        end_if;
        
        if l <> FAIL then
          l:= solvelib::vectorSetToNumberSet(l)
        end_if;

        if l=FAIL then
          return(hold(solve)(lhside=Rhs, x))
        end_if;

        // may be useful to remove radicals
        if type(l) = DOM_SET then
          l:= simplify(l)
        end_if;
        
       // now we have to throw away the additional solutions by attaching
       // an extra condition "valid if satisfying Lhs=Rhs" to every solution
       // provided that there were roots in the equations

        checkSols:=
        proc(l)
          local l2, conds: DOM_LIST, cond,
          checkSolution: DOM_PROC;
          
        begin
          checkSolution:=
          proc(sol)
            local cond;
          begin
            if traperror(
                         (cond:= subs(newlhs, v = sol) = Rhs )) <> 0 then
              FALSE
            else
              is(cond) 
            end_if
          end_proc;

          case type(l)
             of "_union" do 
                return(map(l, checkSols))
             of DOM_SET do
                break
             of solvelib::BasicSet do
                if l=C_ then
                  cond:= piecewise::evalAssuming(newlhs, _and(op(showprop(newlhs))));
                  cond:= cond=0;
                  if not has(cond, x) then
                    return(piecewise([cond, C_],
                                     [not cond, {}]
                                    )
                          )
                  end_if
                end_if;
                // fall through
             otherwise
                // gives additional solutions, improve !!
                roots:= TRUE;
                return(l)
           end_case;
          
          
          l:= split(l, checkSolution);
            
          if options[IgnoreSpecialCases] or nops(l[3]) > 3 or 2^nops(l[3])*100 > MAXEFFORT then
            // we do not construct a piecewise, but just check
            // whether something *can* be a solution
            roots:= TRUE;
            l[1] union l[3]
          else
            l2:= [op(l[3])];
            conds:= map(l2,
                        proc(sol)
                          local result;
                        begin
                          if traperror(
                            (result:= subs(newlhs, v = sol) = Rhs )) = 0 then
                            result
                          else
                            FALSE
                          end_if
                        end_proc
                        );
            l[1] 
            union 
            _union(piecewise([conds[i], {l2[i]}],
                           [not conds[i], {}])
                 $i=1..nops(l2))
          end_if
        end_proc;

        if roots then
          // checking solutions
          roots:= FALSE;
          if type(l) = piecewise then
            l:= piecewise::extmap(l, checkSols)
          else
            l:= checkSols(l)
          end_if;
          if roots and not options[IgnoreSpecialCases] and not options[NoWarning] then 
            warning("Possibly spurious solutions")
          end_if;
        end_if;

        if nops(u)=0 then
          return(l)
        else
          // we have solved for v= f(x); return the set of all x
          // such that f(x) is in that set
          if domtype(l) = DOM_SET and nops(l) = 1 then
            return(solvelib::isolate(rhs(op(u, 1)), op(l, 1), x, options,
                                     reclevel+1, known  ))
          else
            return(solvelib::preImage(rhs(op(u,1)), x, l, options))
          end_if
        end_if;
      end_if;
    end_if;
  end_if; // option[DontRewriteBySystem] = FALSE

  // check whether we want to give up here
  s:= select(indets(Lhs, RatExpr), has, x);
  u:= map(s, type);
  if u intersect {"Ci", "Si", "ellipticF"} <> {} // we cannot handle these types
  then 
    // give up
    return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
  end_if;  
  
  // check for incompatible operands in trigonometric functions
  l:= [op(select(s, X -> contains({"sin", "cos", "tan"}, type(X))))];
  if nops(l) >= 2 then 
    l:= map(l, op, 1);
    l:= map(l, _divide, l[1]);
    for i from 2 to nops(l) do
      if not has(l[i], x) and not contains({DOM_INT, DOM_RAT}, type(l[i])) then
        // give up
        return(hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options)))
      end_if
    end_for
  end_if;  
  
  
  /*****************************************************************************
  Below here, , we use various methods to rewrite Lhs into an equivalent form,
  and re-start solving with the equivalent form
  *****************************************************************************/
  
   /* try to expand the left hand side if it contains tan, cot, ... */
  if MAXEFFORT > 500 and
    traperror((l:=expand(Lhs, MaxExponent = 10)), MaxSteps = 2) = 0 and
    (s:= indets(l,RatExpr)) <> indets(Lhs,RatExpr) then
    userinfo(5, "Expanding the left hand side gives ".
               expr2text(l). " = ".expr2text(Rhs));
    MAXEFFORT:= MAXEFFORT - 500;
    s:= map(s, type);
    if contains(s, "ln") and s intersect {"arcsin", "arccos", "arctan"} <> {} then
       l:= rewrite(l, ln)
    end_if;
    return(solvelib::isolate(l, Rhs, x, options, reclevel+1, known))
  end_if;

  if testtype(Lhs, Type::RatExpr(x)) then
    MAXEFFORT:= MAXEFFORT/2;
    if traperror((l:= poly(numer(Lhs-Rhs), [x]))) = 0 and l <> FAIL then
      if options[IgnoreAnalyticConstraints] or not options[hold(DiscontCheck)] then
       return(solvelib::solve_poly(l, x, options))
      else
       return(solvelib::solve_minus
             (solvelib::solve_poly(l, x, options),
              discont(Lhs-Rhs, x, Undefined, options),
              options
              )
             )
      end_if
     end_if
  end_if;


  // try to rewrite by exp 

  if options[Real] then
    // do not use rewrite(..., exp) as this would create complex
    // subexpressions
    l := subs(Lhs, hold(_power) =
                   proc(a, b)
                   begin
                     if is(a>0)=TRUE and has(b, x) then
                       exp(b*ln(a))
                     else
                       a^b
                     end_if
                   end_proc,
                   EvalChanges)
  else




    l:= subs(Lhs, 
    hold(_power) = proc(a, b)
                   begin
                   if not iszero(a) and indets(b) <> {} then
                     if options[IgnoreAnalyticConstraints] or has(a, x) then
                       exp(b*ln(a))
                     else
                       piecewise([a <>0, exp(b*ln(a))], [a=0, 0^b])
                     end_if
                   else
                     a^b
                   end_if
                 end_proc,
     EvalChanges);
     l:= misc::maprec(l,
     {"sin", "cos", "tan", "cosh", "tanh", "sinh"} = (X -> rewrite(X, exp))
     ):
  end_if;


  if MAXEFFORT > 500 then
// unify expressions like exp(a) and exp(-a)
    l:= misc::maprec(l, {"exp"} =
                     proc(subexpr)
                       local a;

                     begin
                       assert(type(subexpr) = "exp");
                       a:= op(subexpr);
                       if stdlib::hasmsign(a) then
                         1/exp(-a)
                       else
                         subexpr
                       end_if
                     end_proc
                     );
// the following does some damage in a few examples:
//    l:= combine(l, exp);

    l:= misc::maprec(l, {"ln"} = ln::simplify);
    if l <> Lhs then
      MAXEFFORT:= MAXEFFORT - 500;
      q:= solvelib::isolate(l, Rhs, x, options, reclevel + 1, known);
      if not has(q, hold(solve)) then
        return(q)
      end_if
    end_if;
  end_if;
  
  
  


/* try to decompose */
  if options[Real] then
    l := rationalize(Lhs, FindRelations = ["_power", "sin"]);
  else
    l := rationalize(Lhs, FindRelations =  ["_power", "exp", "sin"]);
  end_if;
  userinfo(3, "after rationalize, equation is ".expr2text(l[1]=Rhs)." with".
           expr2text(l[2]));
// special case: only one substitution and equations remains univariate
  [q, p, a]:= split(l[2], has, x);
  // currently, we get errors when we handle the more general case
  // nops(q) = 1 here, too
  if nops(l[2]) = 1 and not has(l[1], x) then
    l[1] := l[1] - Rhs;
    q:= op(q, 1);
    u:=op(q, 1);
    a:=solvelib::solve_eq(l[1], u, options);
    userinfo(3, "Set of solutions of ".expr2text(l[1])." where ".
             expr2text(u)." is ".expr2text(a));
    if not has(a, hold(solve))
      and a <> {u-l[1]} then
      /* to avoid inf. loops, i.e. in solve(exp(x)^(1/4) = C, x) */
      newlhs:= op(q, 2);
      // special case: rationalize(..., "sin") may produce spurious
      // singularities by rewriting sin by tan
      // if newlhs does not occur inside Lhs and +/- infinity is a solution
      // of l[1], then we have to add all singularities of tan
      if type(newlhs) = "tan" and not has(Lhs, newlhs) and
        limit(l[1], u = infinity) = 0 then
        l:= evalAt
               (solvelib::preImage(newlhs, x, a, options)
               union
               solvelib::preImage(op(newlhs, 1), x, tan::undefined, options),
                p
               );
        return(l)
               
      else
        return(evalAt(solvelib::preImage(newlhs, x, a, options), p))
      end_if
    end_if
  else
    // search for wrightOmega and lambertW, and try to eliminate them
    for i from 1 to nops(q) do
      a:= op(q, [i, 2]);
      if type(a) = "wrightOmega" then
        u:= op(q, [i, 1]);
        assert(type(u) = DOM_IDENT);
        // we solve for u, arriving at wrightOmega(g(x))=f(x)
        // this becomes g(x) = f(x) + ln(f(x))
        newlhs:= subs(l[1], {op(q, 1..i-1), op(q, i+1..nops(q))} );
        s:= solvelib::isolate(newlhs, Rhs, u, options, reclevel + 1, known);
        if type(s) = DOM_SET then
          if options[IgnoreAnalyticConstraints] or not options[Real] and
            is(Im(op(a, 1)) <> PI and Im(op(a, 1)) <> -PI,
                                   Goal = TRUE)
            then
            return
            (
             solvelib::solve_union(solve(op(a, 1) = op(s, j) + ln(op(s, j)), x,
                                         options)
                                   $j=1..nops(s)
                                   )
             )
          elif options[Real] then
            // f(x) must be > 0 for the above conclusion to hold
            return(
                   solvelib::solve_union
                   (
                   solvelib::checkAngle
                    (solve(op(a, 1) = op(s, j) + ln(op(s, j)), x, options),
                     op(s, j), x, 1/2, options, maxreclevel-reclevel
                     )
                    $i=j..nops(s)
                    )
                   )
          else
            // the closed form of the conditions is a bit difficult
            // if wrightOmega(g(x)) = f(x), it may be that
            // g(x) = f(x) + ln(f(x)) or g(x) + 2*PI*I = f(x) + ln(f(x))
            solutions:= _union(solve(op(a, 1) + 2*k*PI*I = op(s, j) +
                                     ln(op(s, j)), x, options
                                     )
                               $j=1..nops(s) $k=0..1
                               );
            if not has(solutions, hold(solve)) then
              return(solvelib::checkSolutions(solutions, Lhs = Rhs, x, options))
            end_if
          end_if
        end_if;
      end_if
    end_for;
  end_if;
    // end of special case


  l:= select(indets(Lhs, RatExpr), has, x);
  if map(l, type) intersect {"sin", "cos"} <> {} and 
    not contains(l, x) then 
    MAXEFFORT:= MAXEFFORT * 3/4;
    if traperror((l:= rewrite(Lhs, tan)), MaxSteps = 2) = 0 then
      if traperror((q:= normal(l)), MaxSteps = 2) = 0 then
        l:= q
      end_if;
      if l <> Lhs then
        u:= solvelib::isolate(l, Rhs, x, options, reclevel + 1, known);
        if not has(u, hold(solve)) then
          return(u)
        end_if;
      end_if
    end_if;
  end_if;


  // in case of option IgnoreAnalyticConstraints, try simplification in IgnoreAnalyticConstraints-mode

  if options[IgnoreAnalyticConstraints] and reclevel <= 19 then
    MAXEFFORT:= MAXEFFORT/4;
    if traperror((l:= simplify(Lhs, IgnoreAnalyticConstraints)),
                 MaxSteps = ceil(MAXEFFORT/2000)) = 0 and
      l <>Lhs then
      MAXEFFORT:= MAXEFFORT*3;
      // solve the simplified version
      return(solvelib::isolate(l, Rhs, x, options, reclevel + 1, known))
    else
      MAXEFFORT:= MAXEFFORT*3
    end_if;
  end_if;

  // try simplification: but only once, and increase the recursion level
  // such that few further calls are possible

  if reclevel <= 19 then
    // a quarter of effort for simplifying, three quarters for the recursive call
    MAXEFFORT:= MAXEFFORT/4;
    l:= Simplify(Lhs, Steps = ceil(MAXEFFORT/1000), KernelSteps = 1);
    MAXEFFORT:= 3*MAXEFFORT;
    if l <>Lhs then
      // solve the simplified version
      // note that the singularities may be different now!
      return(solvelib::isolate(l, Rhs, x, options, max(reclevel+1, 20), known))
    end_if;
  end_if;


  // give up
  hold(solve)(Lhs=Rhs, x, solvelib::nonDefaultOptions(options))

end_proc:




// checkAngle
// return those elements el of S such that
// the argument of subs(formula, var=el) is > -angle*PI and <= angle*PI
// negative angle: 1/formula must be in that interval

solvelib::checkAngle:=
proc(S, formula, var: DOM_IDENT, angle, options, maxreclevel = 10)
  local yes, no, dontknow, correct, testel, selectFromSET, i;
begin
  // correct - formula that is equivalent to "x has an arg. > -angle and
                                                // <= angle "
                                                
  if abs(angle) >= 1 then
    // the conditions under which this is true are very difficult to check
    warning("Possibly spurious solutions");
    return(S)
  end_if;  
    
  if options[Real] then
    correct:=
    proc(x)
    begin
      x >= 0
    end_proc
  elif angle = 1/2 then
    // special case for square roots
    correct:= 
    proc(x)
    begin
      signIm(I*x) = 1 or x=0
    end_proc    
  elif angle= -1/2 then
    correct:= 
    proc(x)
    begin
      if is(I*x < 0, Goal = TRUE) then 
        FALSE
      elif is(I*x >= 0, Goal = TRUE) then
        TRUE
      else 
        signIm(I*x) = 1 or x=0
        // Re(x) > 0 or (Re(x) = 0 and Im(x) <=0)
        // if x=0 then TRUE else x=0 or signIm(I/x) = 1 end_if
      end_if  
    end_proc

  else  
    correct:=
    proc(x)
      local argx, cond, xpr, absangle;
    begin
      argx:= arg(x);
      if angle >= 0 then
        if type(argx)=piecewise then
          // flatten out
          cond:= [extop(argx)];
          [cond, xpr]:= [map(cond, op, 1), map(cond, op, 2)];
          _or((cond[i] and xpr[i] in Dom::Interval(-angle*PI, [angle*PI]))
          $i=1..extnops(argx))
        else
          argx in Dom::Interval(-angle*PI, [angle*PI])
        end_if
      else
        absangle:= abs(angle);
        if type(argx)=piecewise then
          // flatten out
          cond:= [extop(argx)];
          [cond, xpr]:= [map(cond, op, 1), map(cond, op, 2)];
          _or((cond[i] and xpr[i] in Dom::Interval([-absangle*PI], absangle*PI))
          $i=1..extnops(argx))
        else
          argx in Dom::Interval([-absangle*PI], absangle*PI)
        end_if
      end_if  
    end_proc
  end_if;


  // testel -> tests whether el has to be an element of the set
  // to be returned
  testel:=
  proc(el)
    local result;
  begin
    if traperror((result:=correct(subs(formula, var=el)))) <> 0 then
      return(FALSE)
    else
      return(result)
    end_if
  end_proc;

  selectFromSET :=
  proc(S)
    save MAXEFFORT;
    local sol;
  begin
    case type(S)
      of DOM_SET do
        // yes: elements that must be returned
        // no : elements that must not be returned
        // dontknow: (parametrized) elements where it depends on the parameters
        //           whether they are solutions or not
        [yes, no, dontknow] := split(S, is@testel);
        if options[IgnoreSpecialCases] = FALSE then
          if nops(dontknow) > 1 then MAXEFFORT:= MAXEFFORT/nops(dontknow) end_if;
          yes union _union(op(map(dontknow,
                                  el -> piecewise([testel(el), {el}],
                                                  [not testel(el), {}]))))
        else
          yes union dontknow
        end_if;
        break
      of piecewise do
        piecewise::extmap(S, selectFromSET); break
      of "_union" do
      of "_intersect" do
        MAXEFFORT:= MAXEFFORT/nops(S);
        map(S, selectFromSET); break
      of "_minus" do
        selectFromSET(op(S, 1)) minus op(S, 2); break
      otherwise
        if options[IgnoreSpecialCases] = TRUE then
          S
        else
          // too complicated ?
          options[MaxRecLevel]:= maxreclevel;
          MAXEFFORT:= min(5000.0, MAXEFFORT/2);
          sol:= solve(testel(var), var, options);
          solvelib::solve_intersect(S, sol)
        end_if
    end_case;
  end_proc;


  selectFromSET(S)

end_proc:


/* end of file */
