/*--
	solvelib::solve_poly3
	Reference: "Handbook of mathematical functions"
		edited by M. Abramowitz and I. Stegun, Dover, New York,
		page 17.

   rewritten by walter (was causing problems for complex coeffs)


--*/

solvelib::solve_poly3 :=
proc(p:DOM_POLY, x:DOM_IDENT, options: DOM_TABLE) 
  local numeric_mode,a,b,c,d,q,r,s1,s2,result, s,
  CasusReducibilis: DOM_PROC,
  CasusIrreducibilis: DOM_PROC;
begin
  userinfo(2,"entering degree 3 poly solver");
  assert(degree(p) = 3);

  if options[Multiple]=TRUE then
    s:= S ->
        if type(S) = piecewise then
          piecewise::fillCases(
          piecewise::extmap(S, Dom::Multiset), {})
        else
          Dom::Multiset(args())
        end_if
  else
    s:= S ->
        if type(S) = piecewise then
          piecewise::fillCases(
          piecewise::extmap(S, DOM_SET), {})
        else
          {args()}
        end_if
  end_if;
  if stdlib::hasfloat(expr(p)) then 
    p:= mapcoeffs(p, float);
    numeric_mode:= TRUE;
    if {op(map(poly2list(p), domtype@op, 1))}
        minus {DOM_FLOAT, DOM_COMPLEX} = {}
          then // no symbolic coefficients remain
               // Use numeric solver, because the symbolic
               // formulas may be numerically unstable.
               // E.g., try p:= (x-1)*(x-1/10^50)*(x-10^50)+10^(-100) !!
      return(s(op(numeric::polyroots(p))));
    end_if:
  else // there are no float entries
    numeric_mode:= FALSE;
  end_if:


  /*---------------------------------------*/
  /* input form p(x) = a*x^3+b*x^2+c*x+d   */
  /*---------------------------------------*/
  a:=coeff(p,3); b:=coeff(p,2); c:=coeff(p,1); d:=coeff(p,0);
  /*---------------------------------------*/
  /* convert to p(x) = x^3+b*x^2+c*x+d     */
  /*---------------------------------------*/
  b:=b/a; c:=c/a; d:=d/a;
  /*---------------------------------------*/
  /* replace b,c,d by parameters b,q,r     */
  /*---------------------------------------*/
  q:=/*expand*/(c/3-b^2/9);
  r:=/*expand*/(1/6*(c*b-3*d)-1/27*b^3);
  /*---------------------------------------*/
  /* now p(x-b/3)= x^3 + 3*q*x - 2*r       */
  /*---------------------------------------*/

  /*---------------------------------------*/
  /* Solve  Q(x) = x^3 + 3*q*x - 2*r       */
  /* If z is a root of Q(x) = p(x-b/3),    */
  /* then z-b/3 is a root of p!            */
  /*---------------------------------------*/
  //  Ansatz: roots z1,z2,z3 of Q(x):
  //  [z1, z2, z3]:= [     (s1+s2),
  //                  -1/2*(s1+s2)+I*3^(1/2)/2*(s1-s2),
  //                  -1/2*(s1+s2)-I*3^(1/2)/2*(s1-s2)]:
  //  ->  (x-z1)*(x-z2)*(x-z3) = x^3 -3*x*s1*s2 -s1^3 -s2^3
  //  So s1, s2 are determined by the two equations:
  //          s1*s2       = -q ,   (1) 
  //          s1^3 + s2^3 = 2*r    (2)
  // Case 1:  q=0  -> s2=0, s1 = (2*r)^(1/3)
  // Case 2:  q<>0 -> s1<>0, s2=-q/s1<>0 
  //          s2 must satisfy:
  //              s1^3 - q^3/s1^3 = 2*r           (2')
  //          --> (s1^3)^2 - 2*r*(s1)^3 -q^3 = 0  (2'')
  //
  //          Solve this quadratic equation in s1^3 !
  //          s1:= (r +- sqrt(r^2+q^3))^(1/3)  (2'')
  //          s2:= -q/s1                       (1)


  if options[Real] then
    d:= q^3+r^2;

    CasusReducibilis:=
    proc()
      local mysurd3;
    begin
      userinfo(2, "Discriminant is positive, one real solution");
      // no good simplification with surd
      mysurd3:= x -> piecewise([x>=0, x^(1/3)], [x<0, -(-x)^(1/3)]);
      s1:= mysurd3(r + d^(1/2));
      s2:= piecewise([q=0, mysurd3(2*r)], [q<>0, -q/s1]);
      // result
      s((s1+s2)-b/3)
    end_proc;


    // casus irreducibilis: avoid using I
    // with s1 = (r + (abs(d))^(1/2)*I)^(1/3), the polar form is
    // s1 = sqrt(r^2 + abs(d))^(1/3) * exp(I/3*arctan(abs(d)^(1/2)/r))
    // if r >= 0, and
    // s1 = sqrt (r^2 + abs(d))^(1/3) *
    // exp(I/3*(PI+arctan(abs(d)^(1/2)/r)))
    // correct but involving I: [d<0, (r + abs(d)^(1/2)*I)^(1/3)]
    CasusIrreducibilis:=
    proc()
      local Res1, Ims1, Resum, Imdiff,  Abss1, Abss1Squared,
      absd, sqrtabsd;
    begin
      absd:= -d;
      sqrtabsd:= sqrt(absd);
      userinfo(2, "Discriminant is negative (casus irreducibilis)");
      Abss1Squared:= (r^2 + absd)^(1/3);
      Abss1:=  sqrt(Abss1Squared);
      Res1:= piecewise([q=0, surd(2*r, 3)],
                       [q<>0 and r>=0, Abss1*
                        cos(1/3*arctan(sqrtabsd/r))],
                       [q<>0 and r<0,Abss1 *
                        cos(1/3*(PI +arctan(sqrtabsd/r) ))]
                       );
      Ims1:= piecewise([q=0, 0],
                     [q<>0 and r>=0, Abss1*
                      sin(1/3*arctan(sqrtabsd/r))],
                     [q<>0 and r<0, Abss1 *
                      sin(1/3*(PI +arctan(sqrtabsd/r) ))]
                     );
      
      // s2 = -q/s1 where q is real
      // thus s2 = -q*Re(s1)/|s1|^2 + I*q*Im(s1)/|s1|^2
    
      // Re(s2) = Res1*(-q/`|s1|^2`);
      // Im(s2) = Ims1*(q/`|s1|^2`);
      // thus Re(s1) + Re(s2) and Im(s2) - Im(s1) satisfy
      Resum:= Res1 * (1 - q/Abss1Squared);
      Imdiff:=Ims1 * (q/Abss1Squared- 1);

      _union(
             s(Resum-b/3),
             s(-1/2*Resum-b/3+
               3^(1/2)/2*Imdiff),
             s(-1/2*Resum-b/3-
               3^(1/2)/2*Imdiff
               )
             )
    end_proc;




    result:= piecewise([d>=0, CasusReducibilis()],
                       [d<0,  CasusIrreducibilis()]);

    if numeric_mode // there are both symbolic entries as well as floats
      then
      return(subs(map(result, float), float(0)=0))
    else
      return(result);
    end_if:
  end_if;

  assert(options[Real] = FALSE);
  
  if iszero(q) then
    s1:= (2*r)^(1/3); 
    s2:= 0;
  else
    d:= q^3+r^2;
    // d is the discriminant
    // If the coefficients of p are real,
    // two cases d>=0 or d<0 (casus irreducibilis) are possible.
    // Let us take care that we are using a *real* third roots if
    // the discriminant is positive
    if numeric::isless(r + d^(1/2), 0) = TRUE then
      s1:= -(-r - d^(1/2))^(1/3)     
    else
      s1:= (r + d^(1/2))^(1/3)
    end_if;
    // Do not use  s2:=(r-d^(1/2))^(1/3) to make sure that
    // the symbolic result is correct after substituting
    // floats for the symbols. We do need s1*s2 = - q !!
    s2:= -q/s1;
  end_if;
 
   
  result:= s(     (s1+s2)-b/3,
             -1/2*(s1+s2)-b/3+I*3^(1/2)/2*(s1-s2),
             -1/2*(s1+s2)-b/3-I*3^(1/2)/2*(s1-s2) );
 
  if numeric_mode // there are both symbolic entries as well as floats
    then
    return(subs(map(result, float), float(0)=0))
  else
    return(result);
  end_if:
end_proc:


solvelib::realRootCubic:=
proc(p, x)
  local a, b, c, d, q, r, CasusReducibilis, CasusIrreducibilis;
begin

  CasusReducibilis:=
  proc()
    local s1, s2; // mysurd;
  begin
    //mysurd:= x -> piecewise([x>=0, x^(1/3)], [x<0, -(-x)^(1/3)]);
    //s1:= mysurd(r + d^(1/2), 3);
    // r + d^(1/2) >= 0 iff d >= r^2 or r >= 0 iff q >= 0 or r >= 0
    // r + d^(1/2) < 0 iff d < r^2 and r < 0 iff q < 0 and r < 0
    s1:= piecewise([q>=0 or r>=0, (r + sqrt(d))^(1/3)],
                   [q<0 and r<0, -(-r - sqrt(d))^(1/3)]
                   );
    s2:= piecewise([q=0, 0], [q<>0, -q/s1]);
    // result
    (s1+s2)-b/3
  end_proc;


  CasusIrreducibilis:=
  proc()
    local  Abss1Squared, Res1, absd, Abss1; 
  begin
    absd:= -d;
   
    Abss1Squared:= (r^2 + absd)^(1/3);
    Abss1:= sqrt(Abss1Squared);
    Res1:= piecewise([q=0, surd(2*r, 3)],
                     [q<>0 and r>=0, Abss1*
                      cos(1/3*arctan(absd^(1/2)/r))],
                     [q<>0 and r<0, Abss1 *
                      cos(1/3*(PI +arctan(absd^(1/2)/r) ))]
                     );
    Res1 * (1 - q/Abss1Squared) - b/3
  end_proc;

  
  a:=coeff(p,3);
  b:=coeff(p,2);
  c:=coeff(p,1);
  d:=coeff(p,0);
  b:=b/a; c:=c/a; d:=d/a;
  /*---------------------------------------*/
  /* replace b,c,d by parameters b,q,r     */
  /*---------------------------------------*/
  q:=/*expand*/(c/3-b^2/9);
  r:=/*expand*/(1/6*(c*b-3*d)-1/27*b^3);
  
  d:= /*expand*/(q^3+r^2);

  piecewise([d>=0, CasusReducibilis()],
            [d<0, CasusIrreducibilis()])
  

  
end_proc:

/* end of file */

