
/*

solvelib::purePolynomial(l, r, x)

solves a polynomial l + r for x where l = f(x)^n for some integer n,
and r is independent of x

*/


solvelib::purePolynomial:=
proc(l: "_power", r, x: DOM_IDENT, options: DOM_TABLE)
  local j: DOM_INT, useRectform, nthroot: DOM_PROC, solveRoots;
begin
  assert(type(op(l, 2)) = DOM_INT);
  assert(not has(r, x));

  nthroot:=
  proc(x, n)
  begin
    // for exponent 1/2, we call sqrt because it tends to find better
    // simplifications
    case n
      of 2 do
        sqrt(x);
        break
      of -2 do
        1/sqrt(x);
        break
      otherwise
        x^(1/n)
    end_case
  end_proc;
  
  // local method solveRoots
  // uses l, r, x, options from outer scope
  // returns {r*s; s in S} where S is the set of all solutions to f(x)^n = 1
  solveRoots:=
  proc()
    local rf, j;
  begin
    if useRectform then
      rf:= expr@rectform
    else
      rf:= id
    end_if;  
    
    if op(l, 2) < Pref::autoExpansionLimit() then 
      _union
      (solvelib::solve_eq
      (op(l,1)- rf(
      r *
      exp(2*PI*I*j/op(l,2))),
      x, options)
      $ j=0..op(l,2)-1)
    else
      j:= genident("j");
      solvelib::Union(
      solvelib::solve_eq
      (op(l,1)- rf(
      r *
      exp(2*PI*I*j/op(l,2))),
      x, options),
      j, 
      Z_ intersect Dom::Interval([0, op(l, 2) - 1])
      )
    end_if   
  end_proc;
  
  
  if options[Real] then
    if op(l, 2) mod 2 = 0 then
      return(piecewise([r <= 0,
                        solvelib::solve_eq(op(l, 1) - nthroot((-r), op(l, 2)),
                                           x,
                                           options)
                        union
                        solvelib::solve_eq(op(l, 1) + nthroot((-r), op(l, 2)),
                                           x,
                                           options) ],
                       [r > 0, {}]
                       ))
    else
      // odd power: solution is unique
      return(solvelib::solve_eq(op(l, 1) - surd(-r, op(l, 2)), x,
                                           options))
    end_if
  end_if;

  
  if testtype(r, Type::Real) = TRUE then
    // start with the real solution
    r:= -r;
    // now we have to solve x^op(l,2)=r
    // we have to use simplify here and in the following,
    // since we want e.g. that 8^(1/3) becomes 2
    if r>=0 then
      r:= simplify(r^(1/op(l,2)))
    else
      assert(r<0);
      if op(l,2) mod 2 = 1 then
        r:= - simplify(abs(r)^(1/op(l,2)))
      else
        // no particular solution is real
        assert(op(l,2) mod 2 = 0);
        r:= exp(PI*I/op(l,2))*simplify(abs(r)^(1/(op(l,2))))
      end_if
    end_if;
    useRectform:= FALSE
    
  else
       
    // try to factor r
    r:= [op(factor(-r))];
    // try to make r[1] nonnegative
    if stdlib::hasmsign(r[1]) then
      for j from 3 to nops(r) step 2 do
        if r[j] mod 2 = 1 then
          // odd exponent -> we may put the negative sign of r[1]
          // into r[j-1]
          r[j-1]:= _negate(r[j-1]);
          r[1]:= _negate(r[1]);
          break
        end_if
      end_for
    end_if;
    // now if r=u*f1^a1*...*fk^ak, our solution is obtained
    // by dividing each exponent a_i
    r[1]:= nthroot(r[1], op(l, 2));
    for j from 3 to nops(r) step 2 do
      r[j]:= r[j]/op(l, 2)
    end_for;
    // obtain special solution
    r:= r[1] * _mult(r[2*j]^r[2*j+1] $j=1..nops(r) div 2);
    useRectform:=  testtype(r, Type::AlgebraicConstant);
      
  end_if;
  
  solveRoots()
  
end_proc:


