

/***************************************************

solvelib::solveSzalay(f)

solves the superelliptic diophantine equation
y^2 = f(x)

f must be a monic univariate integer polynomial of even degree

Source:

Laszlo Szalay, Fast algorithm for solving superelliptic
equations of certain types, Acta Acad. Paed. Agriensis,
Sectio Mathematicae 27 (2000), p.19-24

***************************************************/



solvelib::solveSzalay:=
proc(f: Type::PolyOf(Type::Integer))
  local k, x, B, C, b, i, j, alpha, alpha1, alpha2,
  P1, P2, H, result;

begin

  assert(nops(op(f, 2)) = 1);
  assert(degree(f) mod 2 = 0);
  assert(lcoeff(f) = 1);
 
  k:= degree(f)/2;
  x:= op(f, [2, 1]);
  
// Step 1. We want to write f(x) = B(x)^2 + C(x) where
// B, C are rational polynomials with deg(C) < deg(f)/2

  b:= table();
  b[k] := 1;

  // use formula (13) of the paper:
  // if f = f0 + .. + f_{2k}*X^{2k} and
  //    B = b0 + ...+ bk X^k
  //
  // then f_{2*k-i} = \sum_{j=0}^i b_{k-j} b_{k+j-i}

  for i from 1 to k do
    // determine b[k-i]
    b[k-i]:= (coeff(f, x, 2*k-i) - _plus(b[k-j] * b[k+j-i] $j=1..i-1))/2     
  end_for;

  B:= poly(_plus(b[i]*x^i $i=0..k));
  C:= f - B^2;

  assert(degree(C) < k);

  // Step 2

  // If C=0, we know that f = B^2
  // hence y^2 = B(x)^2 has the solution y = +/- B(x), x in Z_

  // note that B must be an integer polynomial in this case

  if iszero(C) then
    return(Dom::ImageSet([x, expr(B)], x, Z_) union
           Dom::ImageSet([x, -expr(B)], x, Z_))
  end_if;

  // Step 3

  // find the least positive integer alpha such that
  // 2 alpha B and alpha^2 C are integer polynomials

  alpha1:= denom(icontent(B));
  if alpha1 mod 2 = 0 then
    alpha1:= alpha1 div 2
  end_if;

  alpha2:= denom(icontent(C));
  alpha2:= stdlib::ifactor(alpha2, UsePrimeTab);
  for i from 3 to nops(alpha2) step 2 do
    alpha2[i]:= (alpha2[i]+1) div 2
  end_for;
  alpha2:= _mult(alpha2[2*i]^alpha2[2*i+1] $i=1..nops(alpha2) div 2);

  alpha:= ilcm(alpha1, alpha2);

  // Step 4

  P1:= multcoeffs(B, 2*alpha) - poly(1, [x]) + multcoeffs(C, alpha^2);
  P2:= multcoeffs(B, 2*alpha) + poly(1, [x]) - multcoeffs(C, alpha^2);

  // Step 5

  H:= polylib::realroots(P1). polylib::realroots(P2);

  // Step 6

  if nops(H) = 0 then
    return({})
  end_if;
  
  H:= map(H, op);
  H:= [$ceil(min(op(H))).. floor(max(op(H)))];

  result:= map(H, proc(u)
                    local y;
                  begin
                    y:= sqrt(evalp(f, x=u));
                    if type(y) = DOM_INT then
                      [u, y], [u, -y]
                    else
                      null()
                    end_if
               end_proc
                    );

  // Step 7

  {op(result)} union map(solvelib::iroots(C),
                         proc(u)
                         begin
                           evalp(f, x=u);
                           [u, %], [u, -%]
                         end_proc)
  
  
end_proc:

