/*++
sign.mu

	sign -- sign function for real and complex expressions

	sign(x)

	The sign function returns the sign of a real or complex
	number, defined by x/abs(x) for x<>0.

	Mathematically, 'sign(0)' is undefined.
        We use: sign(0) -> 0.
	
	For user defined functions, 'sign' tests for the
	function attribute "sign" and call the procedure
	if it exists.

++*/

sign := funcenv(proc(x)
   name sign;
   local t, s, ss, f;
begin
   if (t:= x::dom::sign) <> FAIL then
     return( t(args()) )
   elif args(0) <> 1 then
     error("expecting only one argument")
   end_if;

   // handle sets

   case type(x) 
   of DOM_SET do
   of "_union" do
      return(map(x, sign));
   of "_intersect" do
   of "_minus" do
      return(procname(args()))
   end_case;
   
   case domtype(x)
   of DOM_INT do
   of DOM_RAT do
   of DOM_FLOAT do
     return( specfunc::sign(x) )
   of DOM_IDENT do
       case x
         of PI do
         of EULER do
         of CATALAN do
           return( 1 )
         otherwise
           if property::hasprop(x) then
              case is(x > 0)
                of TRUE do
                  return( 1 )
                of FALSE do
                  if is(x < 0) = TRUE then return( -1 ) end_if;
                  if is(x = 0) = TRUE then return(sign(0)) end_if;
              end_case;
           end_if;
           return( procname(x) )
        end_case
   of DOM_COMPLEX do
     if iszero(x) then
       return(0)
     else
       return( x/abs(x) )
     end_if
   of DOM_EXPR do
     if testargs() and not testtype( x,Type::Arithmetical ) then
       error("expecting an argument of Type::Arithmetical")
     end_if;

     t:= eval(op(x,0));
     if domtype(t) = DOM_FUNC_ENV then
       // this includes _plus, _mult, _power, ..
       if (s:= slot(t,"sign")) <> FAIL then
         return( s(x) )
       elif not has(x, infinity) and testtype(x, Type::Constant) then
         // First, check whether a float test is appropriate:
         if numeric::isnonzero(x) <> TRUE then
           if hastype(x, unit) then
             // units have the property '> 0':
             if is(x < 0) = TRUE then return(-1);
             elif is(x > 0) = TRUE then return(1)
             elif is(x = 0) = TRUE then return(0)
             end_if;
           end_if;
           // no chance to determine sign via float
           // because of severe cancellation
           return(procname(args()));
         end_if;
         // float test is reliable with very high probability!
         t:= float(x);
         // Is t real or complex ?
         if domtype(t)=DOM_FLOAT then
           return(specfunc::sign(t))
         end_if;
         // If t is a complex constant, then further
         // simplifications via sign(a*b*..)=sign(a)*sign(b)*..
         // may be possible. So, do not return here.
       end_if         
     elif domtype(t) = DOM_IDENT then
       return( procname(args()) )
     end_if;

     t:=op(x,0);
     if t = hold(_mult) then
       // simplifications via sign(a*b*..)=sign(a)*sign(b)*..
       s := 1; ss := 1;
       for f in x do
         t := sign(f);
         if has(t,hold(sign)) then ss := ss*f else s := s*t end_if
       end_for;
       if ss <> x then
         return( s*sign(ss) )
       end_if
     elif t = hold(_plus) then
       s := {};
       for f in x do
         t := sign(f);
         if not iszero(t) then 
           s := (s union {t});	
           if nops(s) > 1 then break end_if
         end_if
       end_for;
       if nops(s) = 1 then return( op(s,1) ) end_if;
       if nops(s) = 0 then return( sign(0) ) end_if;
     elif t = hold(_power) then
       if testtype(op(x,2),Type::Real)
       then
         return( sign(op(x,1))^op(x,2) )
       end_if
     end_if;

     // finally, try is(..)!
     if property::hasprop(x) then
       case is(x > 0) 
       of TRUE do
           return( 1 )
       of FALSE do
           if is(x < 0) = TRUE then return( -1 ) end_if;
           if is(x = 0) = TRUE then return(sign(0)) end_if;
       end_case;
     end_if;
     // normalize sign(a - b) and sign(b - a):
     [s, x]:= stdlib::normalizesign(x);
     return(s*procname(x));
   of DOM_LIST do
   of DOM_POLY do
   of DOM_TABLE do
   of DOM_ARRAY do
   of DOM_FAIL do
   of DOM_BOOL do
   of DOM_VAR do
   of DOM_NULL do
   of DOM_STRING do
     error("expecting a real or complex arithmetical expression")
   otherwise
     return(procname(x))
   end_case
end_proc,
NIL,
table("type"="sign", 
      "print"="sign",
      "info"="sign(x) -- sign function for real and complex expressions [try ?sign for details]" )
):
sign := prog::remember(sign,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("sign")]):

sign::float :=
proc(x)
begin
   // Warning: the PARI function specfunc::sign only accepts
   // real values. Further, it does not float its argument.
   x:= float(x):
   if iszero(x) then
      // make sure that we get the same value
      // as with the symbolic sign(0)
      return(float(sign(0)));
   end_if:
   case domtype(x)
   of DOM_FLOAT do
        return(specfunc::sign(x));   // return integers 0,1,-1 for real floats
   of DOM_COMPLEX do
        return(x/specfunc::abs(x));  // return floats for complex floats
   otherwise 
        return(sign(x)); // do not return x/abs(x), because abs calls
                         // sign --> infinite loop
   end_case;
end_proc:

sign::sign := id:

sign::hull := () -> hull(sign(hull(args()))):

sign::series := loadproc(sign::series, pathname("SERIES"), "sign"):

sign::rectform :=
    loadproc(sign::rectform, pathname("STDLIB","RECTFORM"), "sign"):

sign::diff :=
    loadproc(sign::diff, pathname("STDLIB","DIFF"), "sign"):

/*-- expand attribut for sign

expand(sign(a*b)) -> sign(a)*sign(b) for any a and b
expand(sign(a^b)) -> sign(a)^b  if b is Type::Real

--*/
sign::expand :=
proc(a: "sign")
    local n, s, ss, t, xx, i, x;
begin
    x:= op(a, 1);
    case type(x)
    of "_mult" do
        return(expand(_mult(sign(op(x, i)) $ i=1..nops(x))))
    of "_power" do
        n:= op(x, 2):
        if testtype(n, Type::Real) then
           return(expand(sign(op(x, 1))^n))
        end_if;
    end_case;
    s := Re(x);
    if not has(s,{Re,Im}) and s <> x then
        ss := Im(x);
        if not has(ss,{Re,Im}) then
            t := sqrt( s^2*sign(s)^2+ss^2*sign(ss)^2 );
            if not iszero(t) then return( (s+I*ss)/t ) end_if
        end_if
    end_if;
    xx := expand(x):
    if xx <> x then
         return(sign(xx))
    else return(hold(sign)(xx))
    end_if;
end_proc:

sign::getprop := proc(xpr)
  local sgn;
begin
  sgn := simplify::simplifyCondition::simplifySyntactically::reduceModRPlus(xpr,
    table("decide" = property::_decide, "typereal" = property::_typereal, "isReal" = (X->bool(property::_typereal(X)=TRUE))));
  if not contains(sgn[2], I) then return(sgn[2]); end_if;
  return(Dom::Interval([-1,1])+Dom::Interval([-1,1])*I);
//   return(exp(Dom::Interval(0,2*PI)*I) union {0});
end_proc:
