/*++

Description:

        Shi -- The Hyperbolic Sine Integral

	Shi(x) = int( sinh(t)/t, t=0..x )

	       = -I * Si(x*I) 

	       = ( Ei(1, x) - Ei(1, -x) - PI*I*csign(x*I) ) / 2


	           /  1    if Re(x) > 0 or Re(x) = 0 and Im(x) > 0
        csign(x) = |
                   \ -1    if Re(x) < 0 or Re(x) = 0 and Im(x) < 0

Properties:

  	Shi(x) = -Shi(-x) if x real, x < 0

Parameters:

  	x - an arithmetical expression

Examples:

	Shi(0.0)	->   0.0
	Shi(1.0)	->   1.057250875 
	Shi(-1.0)	-> - 1.057250875
	Shi(1.0*I)	->   0.9460830704*I
	Shi(-1.0*I)	-> - 0.9460830704*I
	Shi(1.0*I+2.0)	->   2.039684555 + 1.678240488*I
	Shi(-1.0*I+2.0)	->   2.039684555 - 1.678240488*I
	Shi(1.0*I-2.0)	-> - 2.039684555 + 1.678240488*I
	Shi(-1.0*I-2.0)	-> - 2.039684555 - 1.678240488*I

++*/

Shi:=
proc(x)
  option noDebug;
  local f;
begin

  if args(0) < 1 then 
     error("expecting one argument")
  elif x::dom::Shi <> FAIL then 
     return(x::dom::Shi(args()))
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, Shi))
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-Shi(-x)) end_if; break
    of DOM_FLOAT do return(Shi::float(x))
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT
      then return(Shi::float(x))
      end_if;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, Shi));
      else
        return(Dom::ImageSet(Shi(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

// in analogy to sin, cos:
  if type(x)= "_mult" then
      f:= op(x, nops(x));
      if testtype(f, Type::Real) and f < 0
         then return(-Shi(-x))
      end_if;
  end_if;

  procname(x)
end_proc:

Shi := prog::remember(Shi,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("Shi")]):

Shi(0):=0:
Shi(infinity):=infinity:
Shi(-infinity):=-infinity:

Shi:= funcenv(Shi):
Shi::print:= "Shi":
Shi::type:= "Shi":
Shi::info:= "Shi(x) -- the hyperbolic sine integral: int(sinh(t)/t, t=0..x)":

Shi::float := proc(y)
  local x;
begin
  x := float(y); 
  if testtype(x, Type::Complex) then 
     Si::float(-I*x)*I
  else
     Shi(x)
  end
end:

Shi::diff:= proc(x, y) // diff(Shi(op1), y)
             local op1;
           begin  // x=Shi(op1)
             op1 := op(x,1);
             sinh(op1)/op1*diff(op1, y)
           end_proc:

Shi::complexDiscont:= {}:
Shi::realDiscont:= {}:
Shi::undefined:= {}:

//-----------------------------------------------------------------------
// Shi(x) = real for real x.
// conjugate(Shi(x)) = Shi(conjugate(x)) for all x in C_.
//-----------------------------------------------------------------------
Shi::conjugate:=
  proc(x)
  begin
    x:= conjugate(x):
    if type(x) = "conjugate" then
       return(hold(Shi)(x))
    else
      return(Shi(x));
    end_if;
  end_proc:

Shi::series:= loadproc(Shi::series, pathname("SERIES"), "Shi"):

// end of file 
