/*++
        Ssi -- the shifted sine integral

	Ssi(x) := Si(x) - PI/2:= int(sin(t)/t,t=0..x) - PI/2

	Reference: Abramowitz & Stegun
++*/

Ssi:=
proc(x)
  option noDebug;
  local f;
begin
  if args(0) <> 1 then
     error("expecting one argument");
  elif x::dom::Ssi <> FAIL then
     return(x::dom::Ssi(args()))
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, Ssi))
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-Ssi(-x) - PI) end_if; break
    of DOM_FLOAT do return(Ssi::float(x))
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT then 
         return(Ssi::float(x))
      end_if;
    //also normalize complex numbers ?
    //if op(x,1)<0 or (op(x,1)=0 and op(x,2)<0) then
    //   return(-Ssi(-x) - PI)
    //end_if;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, Ssi));
      else
        return(Dom::ImageSet(Ssi(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

// in analogy to sin, cos:
  if type(x)= "_mult" then
      f:= op(x, nops(x));
      if testtype(f, Type::Real) and f < 0 then 
         if domtype(f) = DOM_FLOAT or
           (domtype(f) = DOM_COMPLEX and
            domtype(op(f, 0)) = DOM_FLOAT) then
           return(-Ssi(-x) - float(PI))
         else
           return(-Ssi(-x) - PI)
         end_if:
      end_if;

    //also normalize complex factors ?
    //if testtype(f, Type::Complex) and 
    //   (Re(f) < 0 or (Re(f)=0 and Im(f)<0))
    //   then return(-Ssi(-x) - PI)
    //end_if;
  end_if;
  procname(x)
end_proc:

Ssi := prog::remember(Ssi,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("Ssi")]):

Ssi(0):= -PI/2:
Ssi(infinity):=0:
Ssi(-infinity):=-PI:

Ssi:= funcenv(Ssi, op(specfunc::Ssi, 2)):
Ssi::print:= "Ssi":
Ssi::type:= "Ssi":
Ssi::info:= "Ssi(x) -- the shifted sine integral int(sin(t)/t,t=0..x)-PI/2":

Ssi::float:= proc(x)
local fx, tmp, 
      p0, q0, p1, q1, a, b, i, extraDigits;
save DIGITS;
begin
  fx:= float(x);
  if domtype(fx) = DOM_FLOAT or
     domtype(fx) = DOM_COMPLEX then
     if fx = RD_INF then
        return(float(0))
     elif fx = RD_NINF then
        return(float(-PI))
     elif fx = RD_NAN then
        return(RD_NAN);
     end_if:
     if domtype(fx) = DOM_FLOAT and fx > 10^100 then
       // We would need to boost DIGITS by more than
       // log(10, fx) to compute the value via 
       // Si(fx) - PI/2. Hence, do the computation
       // directly without using Si.
       // We evaluate the contfrac representation 
       //  Ssi(x) = Im(exp(-I*x)/(1+I*x-1^2/(3+I*x-2^2/(5+I*x-3^2/(7+I*x-..)..))
       // on the library level:
       // The Wallis method for evaluating the contfrac
       // f = b0 + a1/ (b1 + a2/(b2 + a3/(b3 + ..)..) 
       // is the following recursion:
       // p0:= 1: q0:= 0:
       // p1:= b0: q1:= 1:
       // p.n = b.n * p.(n-1) + a.n * p.(n-2);
       // q.n = b.n * q.(n-1) + a.n * q.(n-2);
       // Then f.n := p.n/q.n is the approximation of f
       // given by the contrac [b0, a1, b1, .., a.n, b.n].
       // Here is code for Ssi: 
         
          p0:= 1: q0:= 0:
          p1:= 1 + I*fx: q1:= 1:
          b:= p1:
          for i from 1 to RD_INF do
            a:= -i^2;
            b:= b + 2;
            [p0, p1]:= [p1, a*p0 + b*p1]; 
            [q0, q1]:= [q1, a*q0 + b*q1]; 
            if abs(p1*q0 - p0*q1) <= 10^(-DIGITS)*abs(p1*q0) then
               break;
            end_if;
          end_for:
          // we need to boost DIGITS for a stable evaluation of exp(-I*fx)
          extraDigits:=  max(5, min(1000, ceil(ln(fx)/2.302585093)));  // 2.30.. = ln(10)
          DIGITS:= DIGITS  + extraDigits;
          tmp:= exp(-I*float(x));
          DIGITS:= DIGITS  - extraDigits;
          return(Im(tmp*q1/p1));
     elif domtype(fx) = DOM_FLOAT and fx > 10 then
       // We loose at least one DIGITS because of cancellation
       // in Ssi(x) = Si(x) - PI/2. Above, we avoided Si to avoid
       // the need to boost DIGITS. However, if the argument
       // is not too large, it's much cheaper to 
       // boost DIGITS and let the kernel do the job via Si. However, 
       // with boosted DIGITS, insignificant changes in float
       // arguments might change the result significantly!
       // Do use numeric::rationalize on floats!
       if domtype(x) = DOM_FLOAT or
         (domtype(x) = DOM_COMPLEX and 
          domtype(op(x, 1)) = DOM_FLOAT) then
         x:= numeric::rationalize(x);
       end_if:
       DIGITS:= DIGITS + min(5, ceil(ln(fx)/2.302585093));  // 2.30.. = ln(10)
       return(specfunc::Si(x) - float(PI/2)):
     else
       extraDigits:= 0:
       repeat
         tmp:= specfunc::Si(x) - float(PI/2):
         if specfunc::abs(tmp) < 10^(-1 - extraDigits) then
           if domtype(x) = DOM_FLOAT or
             (domtype(x) = DOM_COMPLEX and 
              domtype(op(x, 1)) = DOM_FLOAT) then
             x:= numeric::rationalize(x);
           end_if:
           extraDigits:= extraDigits+ min(5, ceil(-ln(specfunc::abs(tmp))/2.302585093));
           DIGITS:= DIGITS + extraDigits;
         else
           return(tmp);
         end_if:
       until FALSE end_repeat;
     end_if:
  else
     return(hold(Ssi)(fx));
  end_if:
end_proc:

Ssi::diff:= proc(x, y)
              local op1;
            begin  // e=Ssi(f)
              op1 := op(x,1);
              sin(op1)/op1*diff(op1, y)
            end_proc:

Ssi::complexDiscont:= {}:
Ssi::realDiscont:= {}:
Ssi::undefined:= {}:

Ssi::expand:= proc(x)
begin   // x = Ssi(x)
   x:= op(x, 1);
   return(Si(x) - PI/2);
end_proc:

Ssi::"transform::laplace":= loadproc(Ssi::"transform::laplace",
                                    pathname("TRANS", "LAPLACE"), "L_Ssi"):

Ssi::rectform:= loadproc(Ssi::rectform, pathname("STDLIB", "RECTFORM"), "Ssi"):

Ssi::series:= loadproc(Ssi::series, pathname("SERIES"), "Ssi"):

// end of file 
