
// Wb = Wurzelbehandlung is a utility for computing sqrt(z) in
// those cases, where the branch of the resulting root does
// not matter (e.g., solve(x^2 = y^4, x) can be expressed as
// {Wb(y^4), -Wb(y^2)}. It automatically simplifies Wb(y^e)
// to y^(e/2), Wb(exp(y)) to exp(y/2) and maps itself into
// products such as (2*PI^4*y^6). Apart from these special
// cases, it just returns Wb(z) = sqrt(z).
// BTW: The name is perfect. It was thought of by Kai in some
// dark moment when facing root extraction in the ode library.

specfunc::Wurzelbehandlung:= proc(z)
local a, b, c, i;
begin
   if iszero(z) = TRUE then 
     return(z)
   end_if;  
   if type(z) = "_plus" or type(1/z) = "_plus" then
     // factor the expression. If there is at least one
     // factor with a non-trivial exponent, then use this
     // factorization, e.g. Wurzel((x + y)*(x - y)^2)
     // --> (x+y)^(1/2)*(x - y).
     // Do not use the factorization, if no simplification
     // by Wurzel is achieved, e.g.: do not rewrite
     //    Wurzel(1 - x^2)  to  (1 -x)^(1/2)*(1 + x)^(1/2).
     z:= [op(factor(z))];
     // split z into linear factors (b), inverse linear factors (c)
     // and other stuff (a).
     // Wurzel( (x^4 - x^2) = x * sqrt(x^2 - 1)
     a:= 1; 
     b:= z[1];  // the coefficient of the factored object
     c:= 1;
     for i from 2 to nops(z)-1 step 2 do
       if z[i + 1] = 1 then // z[i] is a linear factor
          b:= b*z[i]
       elif z[i + 1] = -1 then // z[i] is an inverse linear factor
          c:= c*z[i]
       else // z[i] is a nonlinear factor
          a:= a*z[i]^z[i+1];
       end_if;
     end_for:
     return(specfunc::Wurzelbehandlung(a) * sqrt(expand(b)) / sqrt(expand(c))):
   end_if;
   if type(z) = "_power" then
      return(op(z, 1)^(op(z,2)/2))
   elif type(z) = "exp" then
      return(exp(op(z, 1)/2))
   elif type(z) = "_mult" then
      return(map(z, specfunc::Wurzelbehandlung))
   else
      return(sqrt(z))
   end_if;
end_proc:
