
/*
airyAi(x, n) : AiryAi function and its derivatives


x - an expression
n - a non negative integer
*/

airyAi:=
proc(x, n)
  local tx, m, coeffn, fngn;
begin
  if args(0)=1 then n := 0: end_if:

  if x::dom::airyAi <> FAIL then
    return(x::dom::airyAi(args()))
  elif args(0) <> 1 and args(0) <> 2 then
    error("wrong number of arguments")
  end_if:

  if testtype(x, Type::Set) and not testtype(x, Type::Arithmetical) then
    if testtype(n, Type::Set) and not testtype(n, Type::Arithmetical) then
      return(Dom::ImageSet(airyAi(#x, #n), [#x, #n], [x, n]));
    else
      return(Dom::ImageSet(airyAi(#x, n), [#x], [x]));
    end_if;
  elif testtype(n, Type::Set) and not testtype(n, Type::Arithmetical) then
    return(Dom::ImageSet(airyAi(x, #n), [#n], [n]));
  end_if;

  if not testtype(x, Type::Arithmetical) then
    error("the first argument must be of 'Type::Arithmetical'")
  end_if:
  if domtype(float(n)) = DOM_FLOAT then
    if iszero(frac(n)) then 
       n:= round(n);
    end_if;
    if not (testtype(n, Type::PosInt) or n=0) then
      error("the second argument must be a nonnegative integer")
    end_if;
  end_if:

  tx:=domtype(x);
  if domtype(float(n)) = DOM_FLOAT and (tx=DOM_FLOAT or 
    (tx=DOM_COMPLEX and domtype(op(x, 1))=DOM_FLOAT) or
    (tx=DOM_COMPLEX and domtype(op(x, 2))=DOM_FLOAT)) then
       return(airyAi::float(x,n))
  end_if:

  if iszero(x) then
    if n=0 then return(3^(-2/3)/gamma(2/3))
    elif n=1 then return(-3^(1/6)*gamma(2/3)/2/PI)
    end_if:
  end_if:

  if x = infinity then
     return(0);
  end_if;
  if x = -infinity then
    if  n = 0 then
     return(0);
    end_if;
    if domtype(n) = DOM_INT and n > 0 then
       return(infinity);
    end_if;
  end_if;
  
  if n=2 then
    return(x*airyAi(x,0))
  else
    m:= n - 2;
    if domtype(m) = DOM_INT and m > 0 then
      MAXDEPTH:=max(MAXDEPTH, n);
      
      coeffn:=proc(n,x)
        local f0,f1,f2, g0,g1,g2, c1, c2;
        option remember;
      begin
        [f0, f1, f2, g0, g1, g2]:=[1, 0, x, 0, 1, 0];
        if n=0 then
          [f0, g0];
        elif n=1 then
          [f1, g1]
        elif n=2 then
          [f2, g2]
        else
          c1:=coeffn(n-2,x); c2:=coeffn(n-3,x) ;
          map([x*c1[1]+(n-2)*c2[1], x*c1[2]+(n-2)*c2[2]], expand)
        end_if;
      end_proc:

      fngn:=coeffn(n,x);
      return(fngn[1]*airyAi(x,0)+fngn[2]*airyAi(x,1))

    end_if;
  end_if:

  procname(x, n)
end_proc:

airyAi := prog::remember(airyAi,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("airyAi")]):

airyAi := funcenv(airyAi):
airyAi::type := "airyAi":
airyAi::print := "airyAi":
airyAi::info := "airyAi(x,n) -- solution of the Airy differential equation and
its derivatives":

  
airyAi::float :=
proc()
  local n, x, fx, c1, c2, z, coeffn, fngn;
begin
  x:=args(1); n:=args(2); fx:=float(x);
  if not( type(fx)=DOM_FLOAT or type(fx)=DOM_COMPLEX ) then
       return(airyAi(fx,n))
  end_if:

  c1:=float(3^(-2/3)/gamma(2/3));
  c2:=float(3^(-1/3)/gamma(1/3));

  x:= fx:
  // here, x = fx is a real or complex float
  if n=0 then
    if iszero(x) then
      return(c1)
    elif Re(fx)<0 then
      x:=-x; z:=2/3*x^(3/2);
      return(1/3*sqrt(x)*(besselJ(-1/3,z)+besselJ(1/3,z)))
    else
      return(float(1/PI*sqrt(x/3)*besselK(1/3,2/3*x^(3/2))))
    end_if:
    
  elif n=1 then
    if iszero(x) then
      return(-c2)
    elif Re(fx)>0 then
      return(float(-1/PI*x/sqrt(3)*besselK(2/3,2/3*x^(3/2))))
    else
      x:=-x; z:=2/3*x^(3/2);
      return(-x/3*(besselJ(-2/3,z)-besselJ(2/3,z)))
    end_if:
    
  elif n=2 then
    return(x*airyAi(x,0))
    
  else
    MAXDEPTH:=max(MAXDEPTH,n);
    
    coeffn:=proc(n,x)
      local f0,f1,f2, g0,g1,g2, c1, c2;
      option remember;
    begin
      [f0, f1, f2, g0, g1, g2]:=[1.0, 0.0, x, 0.0, 1.0, 0.0];
      if n=0 then
        [f0, g0];
      elif n=1 then
        [f1, g1]
      elif n=2 then
        [f2, g2]
      else
        c1:=coeffn(n-2,x); c2:=coeffn(n-3,x);
        [x*c1[1]+(n-2)*c2[1], x*c1[2]+(n-2)*c2[2]]
      end_if;
    end_proc:

    fngn:=coeffn(n,x);
    return(fngn[1]*airyAi(x,0)+fngn[2]*airyAi(x,0))
  end_if:

end_proc:


airyAi::diff :=
proc(f, y)
  local n, x;
begin
  x:=op(f,1); n:=op(f,2);
  if has(y,n) then hold(diff)(args())
  else 
    return(airyAi(x,n+1)*diff(x,y)):
  end_if:
end_proc:



airyAi::series :=
proc(f, n, x, o, dir, opt)
  local t, l, P, Q1, Q2, s;
begin
  t:=Series::series(f, x, o, dir, opt);

  if dir <> Undirected then
    l:=limit(f,x,dir);
  elif domtype(t) = Series::Puiseux then
    if ldegree(t) = FAIL then
      l:=FAIL;
    elif ldegree(t) >= 0 then
      l:=coeff(t,x,0);
    else
      return(FAIL)
    end_if;
  else
    l:=FAIL;
  end_if;
  
  if n=0 then // for Airy(x)
    if l=infinity or l=-infinity then // c.f. A&S 10.4.59, A&S 10.4.60
      if sign(l) > 0 then
        P:=series(1/2/sqrt(PI)*hypergeom([1/6,5/6],[],-3/4*x^(3/2)),
                  x = 0, o, dir, opt);
        if f <> 1/x then
          s:=Series::series(1/f,x,o,dir, opt);
          P:=P @ s;
        end_if:
        P:=P * Series::series((1/f)^(1/4),x,o,dir,opt);
        return(P * Series::series(exp(-2/3*f^(3/2)),x,o,dir,opt));
      else
        Q1:=series(1/sqrt(PI)*hypergeom([1/12,5/12,7/12,11/12],[1/2],-9/4*x^3),
                   x = 0, o, dir);
        Q2:=series(-5/48/sqrt(PI)*hypergeom([7/12,11/12,13/12,17/12],[3/2],
                   -9/4*x^3),x = 0, o, dir);
        if f <> -1/x then
          s:=Series::series(-1/f,x,o,dir, opt);
          Q1:=Q1 @ s; Q2:=Q2 @ s;
        end_if:
        Q1:=Q1 * Series::series((-1/f)^(1/4),x,o,dir, opt);
        Q1:=Q1 * sin::series(PI/4+2/3*(-f)^(3/2),x,o,dir, opt);
        Q2:=Q2 * Series::series((-1/f)^(7/4),x,o,dir, opt);
        Q2:=Q2 * cos::series(PI/4+2/3*(-f)^(3/2),x,o,dir, opt);
        return(Q1+Q2);
      end_if:
    end_if:

  
  elif n=1 then // for Airy'(x)
    if l=infinity or l=-infinity then // A&S 10.4.61, A&S 10.4.62
      if sign(l) > 0 then
        P:=series(-1/2/sqrt(PI)*hypergeom([-1/6,7/6],[],-3/4*x^(3/2)),
                  x = 0, o, dir);
        if f <> 1/x then
          s:=Series::series(1/f,x,o,dir, opt);
          P:=P @ s;
        end_if:
        P:=P * Series::series(f^(1/4),x,o,dir, opt);
        return(P * Series::series(exp(-2/3*f^(3/2)),x,o,dir, opt));
      else
        Q1:=series(-1/sqrt(PI)*hypergeom([-1/12,5/12,7/12,13/12],[1/2],
                   -9/4*x^3), x = 0, o, dir);
        Q2:=series(7/48/sqrt(PI)*hypergeom([5/12,11/12,13/12,19/12],[3/2],
                   -9/4*x^3),x = 0, o, dir);
        if f <> -1/x then
          s:=Series::series(-1/f,x,o,dir, opt);
          Q1:=Q1 @ s; Q2:=Q2 @ s;
        end_if:
        Q1:=Q1 * Series::series((-f)^(1/4),x,o,dir, opt);
        Q1:=Q1 * cos::series(PI/4+2/3*(-f)^(3/2),x,o,dir, opt);
        Q2:=Q2 * Series::series((-1/f)^(5/4),x,o,dir, opt);
        Q2:=Q2 * sin::series(PI/4+2/3*(-f)^(3/2),x,o,dir, opt);
        return(Q1+Q2);
      end_if:
    end_if:
  end_if:

  Series::unknown(airyAi(f, n), x, o ,dir)
end_proc:
