/*++
arg -- the polar angle ("the argument") of a complex number


arg(z)

z - arithmetical expression

Let z = x + I*y, x,y, real.


For compatibility reasons, also the following call is allowed:

arg(x, y)

x, y - real quantifiers, representing the complex number x+I*y

arg(x,y) computes the principal value of the argument of the complex
number x+I*y with -PI < arg(x,y) <= PI.

Mathematical identities:
  1) arg(x,y) = ln(sign(x+I*y))/I
  2) arg(x,y) = arctan(y/x) + PI/2*sign(y)*(1-sign(x))
  
1) holds for any finite real x, y
2) holds for any x <> 0, y <> 0  (unfortunately, sign(0) = 0 making
                                this formula wrong for y = 0, x < 0)

The value for x = y = 0 is undefined. 

++*/

arg:= prog::remember(
proc(x, y)
  local floatconvert, fx, fy, sx, sxknown, sy, syknown, xx, yy, dummy;
  name arg;
begin
  // ---------- argument check -----------------------
  if args(0) < 1 or args(0) > 2 then
       error("wrong no of args") 
  elif x::dom::arg <> FAIL then
       return(x::dom::arg(args()))
  // elif y::dom::arg <> FAIL then
       // return(y::dom::arg(args()))
  end_if;


  if args(0) = 1 then

    // handle sets
    
    case type(x)
      of DOM_SET do
      of "_union" do
        return(map(x, arg))
    end_case;

    if not testtype(x, Type::Arithmetical) then
      /* generic handling of sets */
      if testtype(x, Type::Set) then
        if type(x)=Dom::ImageSet then
          return(map(x, arg));
        else
          return(Dom::ImageSet(arg(#x), #x, x));
        end_if;
      end_if;

      error("Argument has wrong type")
    end_if;



    case type(x)
      of DOM_FLOAT do
        if x>=0 then
          return(float(0))
        else
          return(float(PI))
        end_if

      of DOM_RAT do
      of DOM_INT do
        if x>=0 then
          return(0)
        else
          return(PI)
        end_if;
      of DOM_COMPLEX do
        return(arg(op(x,1), op(x,2)))
    end_case;

// handle constants

    if testtype(x, Type::Constant) then
      // rectform should not be too expensive
      sx:= rectform(x);
      if not iszero(op(sx, 3)) then
	 return(procname(x))
      else
         return(arg(op(sx, 1), op(sx, 2)))
      end_if
    end_if;


  // handle symbolic arguments
  
    if is(x, Type::Real) = TRUE then
      return(piecewise([x>=0, 0], [x<0, PI]))
    end_if;

    
  // handle infinities

    if has(x, infinity) then
      if x = infinity then
        return(0)
      elif x = -infinity then
        return(PI)
      end_if;
      
      return(procname(x))

    end_if;

  // last simplification: remove positive factors from a product
  // replace negative factors by -1

    if type(x) = "_mult" then
      xx:= map(x, a -> if is(a > 0) = TRUE then
                        1
                      elif is(a < 0) = TRUE then
                        -1
                      elif type(a) = DOM_COMPLEX and iszero(op(a, 1)) then
                        // r*I may be replaced by sign(r)*I
                        specfunc::sign(op(a, 2))*I
                      else
                        a
                      end_if
                        );
      if xx <> x then
        return(arg(xx))
      end_if;
    end_if;

    if type(x) = "sign" then
       // arg(sign(x)) = arg(x) for all x in C_
       return(procname(op(x, 1)));
    end_if;
  
  // default case
  
    return(procname(x))
    
  end_if; // args(0) = 1



  // from here, we know that args(0) = 2

  assert(args(0) = 2);

  

  // ----------  map to sets -------------------------
  // This is quite complicated, because both arguments have
  // to be handled
  case [type(x),type(y)]
    of [DOM_FLOAT, DOM_FLOAT] do
        return(specfunc::arg(x+I*y));
    of [DOM_SET, DOM_SET] do
        return({arg(sx + I*sy) $ sx in x $ sy in y});
    of [DOM_SET, "_union"] do
    of ["_union", DOM_SET] do
        return(_union((arg(sx + I*sy) $ sx in x) $ sy in y));
    of ["_union", "_union"] do

        // arg({a} union A, C union D)
        //  -> arg({a},C) union arg({a},D) union arg(A,C) union arg(A,D)
        // Make sure that A, C, D are interpreted as sets, i.e.,
        // arg({a},C) --> arg(a, C), not {arg(a,C)} !

        [x, xx, dummy]:= split(x, testtype, DOM_SET):
        if type(xx) <> "_union" then xx:= [xx] end_if:
        [y, yy, dummy]:= split(y, testtype, DOM_SET):
        if type(yy) <> "_union" then yy:= [yy] end_if:
 
        return(_union({(arg(sx + I*sy) $ sx in x ) $ sy in y},
                       (arg(sx + I*sy) $ sx in xx) $ sy in y,
                       (arg(sx + I*sy) $ sx in x ) $ sy in yy,
                       (arg(sx + I*sy) $ sx in xx) $ sy in yy ));
  end_case;

  case type(x)
    of DOM_SET do
    of "_union" do
       // y cannot be a set, if x is a set
       return(map(x, arg, y));
  end_case;

  case type(y)
    of DOM_SET do
    of "_union" do
       // x cannot be a set, if y is a set
       return(map(y, (y,x) -> arg(x,y), x))
  end_case;
  // --------------------------------------------------

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if testtype(y, Type::Set) and not testtype(y,Type::Arithmetical) then
        return(Dom::ImageSet(arg(#x, #y), [#x, #y], [x, y]));
      else
        return(Dom::ImageSet(arg(#x, y), [#x], [x]));
      end_if;
    end_if;

    error("first argument must be of 'Type::Arithmetical'")
  end_if;
  if not testtype(y,Type::Arithmetical) then
    if testtype(y, Type::Set) then
      return(Dom::ImageSet(arg(x, #y), [#y], [y]));
    end_if;

     error("second argument must be of 'Type::Arithmetical'")
  end_if;

  if traperror((fx := float(x))) = 0 and
    domtype(fx) = DOM_COMPLEX then
    error("first argument must be real")
  end_if;
  if traperror((fy := float(y))) = 0 and
    domtype(fy) = DOM_COMPLEX then
    error("second argument must be real")
  end_if;
  // --------------------------------------------------

  if domtype(x) = DOM_FLOAT or domtype(y) = DOM_FLOAT 
     then floatconvert:= float;
     else floatconvert:= id;
  end_if:

  sx:= sign(x); sxknown:= bool(domtype(sx)=DOM_INT);
  sy:= sign(y); syknown:= bool(domtype(sy)=DOM_INT);

  if iszero(sx) and iszero(sy) then
     // each value between -PI < phi <= +PI would make sense here:
     return(floatconvert(0));
  end_if:
  if iszero(sx) /*and syknown*/ then // sy <> 0 here !!
        return(floatconvert(sy*PI/2));
  end_if:
  if iszero(sy) /*and sxknown*/ then // sx <> 0 here !!
        return(floatconvert((1-sx)*PI/2));
  end_if:

  // now sx<>0 and sy<>0 in the following !!

  // ----- handle infinity ----------
  if has(x,infinity) then
     if has(y,infinity)
        then return(procname(x + I*y))
        else if sxknown and syknown then
                return(floatconvert(sy*(1-sx)*PI/2));
             end_if;
     end_if;
  end_if;
  if has(y,infinity) then
     if syknown then
        return(floatconvert(sy*PI/2));
     end_if;
  end_if;

  // ------------------------------------------
  // explicit result in terms of arctan, if
  // sign(x) and sign(y) are known. Note that
  // sign has already taken care of properties
  // of x, y set via assume.

  //Do not use the following piece of code to avoid
  //return values involving a symbolic sign(x). This
  //is not differentiable at x = 0 (and arctan(y/x)
  //is not smooth at x=0), so differentiation and 
  // series expansion would give odd results.
  //if syknown and not iszero(sy) then 
  //   // the following formula holds for all (x,y) with y<>0
  //   return(floatconvert(arctan(y/x) + PI/2*sy*(1-sx)));
  //end_if:

  //Instead, use this simplication only if sign(x) is known,
  //or is x is known to be non-zero via assume(x<>0)

  if (sxknown or is(x, Type::NonZero)=TRUE) and syknown then
     // note that sx<>0 and sy<>0 here
     return(floatconvert(arctan(y/x) + PI/2*sy*(1-sx)));
  end_if:
  
  // In the right half plane we do not need to know the
  // sign of y, because arg(x,y) = arctan(y/x) for any y.

  if sx = 1 then
     return(floatconvert(arctan(y/x)))
  end_if:

  // Last step: sign(x) or sign(y) are not known, so return
  // an unevaluated call arg(x + I*y). However, the following
  // "projective" simplification such as arg(2*x,2*y) = arg(x,y)
  // can still be done for any symbolic x, y.
  // The idea is: normalize the first argument

  if type(x) = "_mult" then
     sx:= op(x, nops(x));
     if is(sx>0) = TRUE then 
        x:= x/sx; y:= y/sx;
     end_if:
     if is(sx<0) = TRUE then 
        x:= -x/sx; y:= -y/sx;
     end_if:
  end_if;
  return(procname(x + I*y));
end_proc, () -> [property::depends(args()), DIGITS, slotAssignCounter("arg")]):

//---------------------------------------------------------------------
arg:= funcenv(arg):
arg::print:= "arg":
arg::Content:= stdlib::genOutFunc("Carg", 1, 2):
arg::info := "arg -- the polar angle of a complex number [try ?arg for help]":
arg::type:= "arg":
//---------------------------------------------------------------------

// arg(0, 0):= 0:

//---------------------------------------------------------------------

arg::realDiscont:= {0}:
arg::complexDiscont :=
    loadproc(arg::complexDiscont, pathname("STDLIB","DISCONT"), "arg"):
arg::undefined := {}:

//---------------------------------------------------------------------
arg::float:=
          proc(x, y)
          begin
            if args(0)=2
              then
              arg(float(x +I*y))
            elif args(0)=1 then
              arg(float(x))
            else
              error("wrong no of args")
            end_if
          end_proc:


arg::hull := DOM_INTERVAL::arg@hull:

//---------------------------------------------------------------------
arg::diff :=
           proc()
             local x, y;
           begin
             if nops(args(1)) = 2 then
                // a call to arg with two args should not happen
                // but we handle it anyway
               x:= op(args(1),1);
               y:= op(args(1),2);
             else
               x:= Re(op(args(1), 1));
               y:= Im(op(args(1), 1));
             end_if;
             if not iszero(x) then
                diff(y/x, args(2..args(0))) * x^2/(x^2+y^2)
             else
                // The general formula (for 1st derivatives) is:
                //(diff(y, args(2..args(0)))*x 
                //-diff(x, args(2..args(0)))*y))/(x^2+y^2)
                // For x = 0 we get 0:
                return(0);
             end_if;
           end_proc:
//---------------------------------------------------------------------

arg::series := loadproc(arg::series, pathname("SERIES"), "arg"):

//---------------------------------------------------------------------

arg::rectform :=
           proc(xx, yy)
             local x, y;
           begin
             x:= rectform::new( xx );
             if args(0) = 2 then
               y:= rectform::new( yy );
               if extop(x,3) <> 0 or extop(y,3) <> 0 then
                 new(rectform, 0, 0, hold(arg)(rectform::expr(x + I*y)))
               else
      // assume that arg has been correctly called with two real arguments
                 new(rectform, hold(arg)(xx+ I*yy), 0, 0 )
               end_if;
             else
               if extop(x, 3) <> 0 then
                 new(rectform, 0, 0, hold(arg)(rectform::expr(x)))
               else
                 new(rectform, hold(arg)(xx), 0, 0)
               end_if  
             end_if
           end_proc:
//---------------------------------------------------------------------
arg::Re :=
           proc(x, y)
           begin
             if args(0) = 2 then
               hold(arg)(x + I*y)
             else
               hold(arg)(x)
             end_if
           end_proc:
//---------------------------------------------------------------------
arg::Im :=
           proc(x, y)
           begin
             if args(0)= 1 then
               if type(x) = DOM_FLOAT or
                 type(x) = DOM_COMPLEX and type(op(x,1))=DOM_FLOAT then
                 return(float(0))
               else
                 return(0)
               end_if
             else // two arguments
             
           
               if domtype(x)=DOM_FLOAT or
                 domtype(y)=DOM_FLOAT
                 then
                 return(float(0))
               else
                 return(0)
               end_if;
             end_if
           end_proc:
//---------------------------------------------------------------------
// arg::expand:= proc() begin ... end_proc:

arg::sign := proc(e)
  local x, y, z;
begin
  if nops(e) = 2 then
     [x, y] := [op(e)]:
     piecewise([y = 0 and x >= 0, 0],
               [y = 0 and x < 0, 1],
               [y > 0, 1],
               [y < 0, -1]):
  else
     z := op(e):
     y := Im(z):
     piecewise([z >= 0, 0],
               [z < 0 or y > 0, 1],
               [y < 0, -1]):
  end_if:
end_proc:

// for speed:
DOM_COMPLEX::arg :=
  proc(z)
    local x, y;
  begin
    x := Re(z);
    y := Im(z);
    if domtype(x) <> DOM_FLOAT and domtype(y) <> DOM_FLOAT then
      return(arg(x, y));
    end_if;
    if iszero(y) then // this can't happen, can it?
      if x < 0 then
	float(PI);
      else
	float(0);
      end_if;
    else
      if iszero(x) then
	if y < 0 then
	  float(-PI/2);
	else
	  float(PI/2);
	end_if;
      else
	arctan(y/x) + float(PI/2)*sign(y)*(1-sign(x));
      end_if;
    end_if;
  end_proc:

arg::minprop:= TRUE:

// end of file 
