/*++
arcsin -- the inverse sine

arcsin(x)

x - expression

NB:  The branch cuts are the real intervals (-infinity, -1) and
     (1, infinity).
     Note: for x >= 1:  arcsin(x) =  limit(arcsin(x-eps*I),eps=0, eps>0)
           for x <=-1:  arcsin(x) =  limit(arcsin(x+eps*I),eps=0, eps>0)

     I.e., on the right real line the values coincide with the
     limits from below, on the left real line with the limit from
     above. OK.
++*/

arcsin:=
proc(x)
  option noDebug;
  local f, handleSin;
  name arcsin;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif x::dom::arcsin <> FAIL then
    return(x::dom::arcsin(args()))
  elif args(0) <> 1 then
    error("wrong no of args")
  elif domtype(x) = DOM_SET or type(x)="_union" then
    return(map(x,arcsin))
  end_if;

  case domtype(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then
        return(-arcsin(-x))
      end_if;
      break;

    of DOM_FLOAT do
      return(arcsin::float(x));

    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
        return(arcsin::float(x))
      end_if;
      if op(x,1) = 0 then
        return(arcsinh(-x*I)*I)
      end_if;
      break;
    of DOM_SET do
    of "_union" do
       return(map(x,arcsin))
    of "_intersect" do
    of "_minus" do
        return(Dom::ImageSet(arcsin(`#t`), `#t`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arcsin(#t), #t, x));
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  //=============================
  handleSin:= proc(x)
  local f, s;
  begin
     if (is(x >=-PI/2) and 
         is(x <= PI/2))=TRUE then
        return(x);
     end_if;
     s:= -signIm(x);
     if domtype(s) <> DOM_INT then
        return(FAIL);
     end_if;
     f:= s*ceil(s*Re(x)/PI - 1/2);
     if type(f) = DOM_INT then
       return((-1)^f*(x-f*PI))
     end_if:
     FAIL;
  end_proc:
  //=============================

  case type(x) 
    of "_mult" do // normalize using arcsin(-x) = -arcsin(x)
       f:= op(x, nops(x));
       if (testtype(f, Type::Real) and f < 0) or
          (testtype(f/I, Type::Real) and f/I < 0) then
          return(-arcsin(-x));
       end_if;
       if type(x/I) = "sinh" then //I*sinh(y) = sin(y*I)
          f:= handleSin(I*op(x/I));
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       if type(-x/I) = "sinh" then //-I*sinh(y) = sin(-y*I)
          f:= handleSin(-I*op(-x/I));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(-x) = "cosh" then //-cosh(y) = -cos(y*I) = sin(-PI/2 + y*I)
          f:= handleSin(I*op(x) - PI/2);
          if f <> FAIL then
            return(f);
          end_if;
       end_if:
       break;
    of "cos" do  // cos(y) = sin(PI/2 - y);
       f:= handleSin(-op(x) + PI/2);
       if f <> FAIL then
          return(f);
       end_if;
       break;
    of "sin" do
       f:= handleSin(op(x));
       if f <> FAIL then
          return(f);
       end_if;
       break;
    of "cosh" do //cosh(y) = cos(y*I) = sin(PI/2 + y*I)
       f:= handleSin(I*op(x) + PI/2);
       if f <> FAIL then
          return(f);
       end_if;
       break;
  end_case;

  // it is useful to have a normal form such that expressions like
  // arcsin(x-y) + arcsin(y-x) are simplified automatically. Cf. bug #2172
  if stdlib::hasmsign(x) then
    -procname(-x)
  else
    procname(x)
  end_if
end_proc:

arcsin := prog::remember(arcsin,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arcsin")]):

arcsin(0):= 0:
arcsin( 1):= PI/2:
arcsin(-1):=-PI/2:
arcsin( (3^(1/2))/2):= PI/3:
arcsin(-(3^(1/2))/2):=-PI/3:
arcsin( (2^(1/2))/2):= PI/4:
arcsin(-(2^(1/2))/2):=-PI/4:
arcsin( 1/4*2^(1/2)*(5-5^(1/2))^(1/2)):= PI/5:
arcsin(-1/4*2^(1/2)*(5-5^(1/2))^(1/2)):=-PI/5:
arcsin( 1/4*2^(1/2)*(5+5^(1/2))^(1/2)):= 2*PI/5:
arcsin(-1/4*2^(1/2)*(5+5^(1/2))^(1/2)):=-2*PI/5:
arcsin( 1/2):= PI/6:
arcsin(-1/2):=-PI/6:
arcsin( ((2-2^(1/2))^(1/2))/2):= PI/8:
arcsin(-((2-2^(1/2))^(1/2))/2):=-PI/8:
arcsin( ((2+2^(1/2))^(1/2))/2):= 3*PI/8:
arcsin(-((2+2^(1/2))^(1/2))/2):=-3*PI/8:
arcsin( 1/4*(5^(1/2)-1)):= PI/10:
arcsin(-1/4*(5^(1/2)-1)):=-PI/10:
arcsin( 1/4*(5^(1/2)+1)):= 3*PI/10:
arcsin(-1/4*(5^(1/2)+1)):=-3*PI/10:
arcsin( (6^(1/2)-2^(1/2))/4):= PI/12:
arcsin(-(6^(1/2)-2^(1/2))/4):=-PI/12:
arcsin( (6^(1/2)+2^(1/2))/4):= 5*PI/12:
arcsin(-(6^(1/2)+2^(1/2))/4):=-5*PI/12:

arcsin( infinity):=  PI/2 - I*infinity:
arcsin(-infinity):= -PI/2 + I*infinity:
arcsin(I*infinity):=   I*infinity:
arcsin(-I*infinity):= -I*infinity:


arcsin:= funcenv(arcsin, op(specfunc::arcsin, 2)):
arcsin::print := "arcsin":
arcsin::info := "arcsin -- the inverse sine":
arcsin::type := "arcsin":
arcsin::float := specfunc::arcsin:

arcsin::hull := DOM_INTERVAL::arcsin@hull:

arcsin::inverse := "sin":

arcsin::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(1-op1^2) then
       0
    else
       diff(op1, args(2..args(0))) / sqrt(1-op1^2)
    end_if;
  end_proc:

arcsin::expand := x -> arcsin(expand(op(x))):

arcsin::simplify:= loadproc(arcsin::simplify, pathname("STDLIB", "SIMPLIFY"),
                            "arcsin"):


// the arcsin has a branch cut at the negative real axis
arcsin::complexDiscont :=
    loadproc(arcsin::complexDiscont, pathname("STDLIB","DISCONT"), "arcsin"):
arcsin::realDiscont := {}:
arcsin::undefined := {}:

arcsin::rectform :=
    loadproc(arcsin::rectform, pathname("STDLIB","RECTFORM"), "asin"):
arcsin::getprop :=  proc(x:"arcsin") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arcsin(v)); else return(C_); end_if; end_proc:

arcsin::series :=
    loadproc(arcsin::series, pathname("SERIES"), "arcsin"):

arcsin::Content := stdlib::genOutFunc("Carcsin", 1):

// end of file
