/*++
arcsinh -- the inverse hyperbolic sine

arcsinh(x)

x - expression
++*/

arcsinh:=
proc(x)
  option noDebug;
  local f, handleSinh;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::arcsinh <> FAIL then return(x::dom::arcsinh(args()))
  elif args(0) <> 1 then error("wrong number of arguments")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-arcsinh(-x)) end_if;
      break;
    of DOM_FLOAT do
      return(arcsinh::float(x));
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT then
        return(arcsinh::float(x))
      end_if;
      break;
    of DOM_SET do
    of "_union" do
      return(map(x,arcsinh))
    of "_intersect" do
    of "_minus" do
      return(Dom::ImageSet(arcsinh(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arcsinh(#t), #t, x));
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  //===============================================
  handleSinh:= proc(x)  // compute arcsinh(sinh(x))
  local f, s;
  begin
    if testtype(x, Type::Real) or
      is(x in R_) = TRUE then
      return(x);
    end_if;
    s:= signIm(I*x):
    if domtype(s) = DOM_INT then
       f:= s*round(-s*Im(x)/PI);
       if domtype(f) = DOM_INT then
         return((-1)^f*(x+f*I*PI));
       end_if;
    end_if;
    return(FAIL);
  end_proc:
  //===============================================

  case type(x)
    of "_mult" do
      f:= op(x, nops(x));
      if (testtype(f, Type::Real) and f < 0) or
         (testtype(f/I, Type::Real) and f/I < 0) then 
          return(-arcsinh(-x));
      end_if;
      if type(x/I) = "sin" then // I*sin(y) = sinh(I*y)
        f:= handleSinh(I*op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(-x/I) = "sin" then // -I*sin(y) = sinh(-I*y)
        f:= handleSinh(-I*op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(-x) = "sinh" then // -sinh(x) = sinh(-x) 
        f:= handleSinh(-op(-x));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(x/I) = "cosh" then // I*cosh(x) = sinh(PI/2*I + x)
        f:= handleSinh(PI*I/2 + op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(-x/I) = "cosh" then // -I*cosh(x) = sinh(-PI/2*I + x)
        f:= handleSinh(-PI*I/2 + op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(x/I) = "cos" then // I*cos(x) = sinh(PI/2*I + I*x)
        f:= handleSinh(PI*I/2 + I*op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      if type(-x/I) = "cos" then // -I*cos(x) = sinh(-PI/2*I + I*x)
        f:= handleSinh(-PI*I/2 + I*op(x/I));
        if f <> FAIL then
          return(f);
        end_if;
      end_if:
      break;
    of "sinh" do // arcsinh(sinh(x))=x when -PI/2 < Im x <= PI/2
      f:= handleSinh(op(x));
      if f <> FAIL then
        return(f);
      end_if;
      break;
  end_case;

  procname(x)
end_proc:

arcsinh := prog::remember(arcsinh,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arcsinh")]):

arcsinh(0):= 0:
arcsinh( I):= I*PI/2:
arcsinh(-I):=-I*PI/2:
arcsinh( I*(3^(1/2))/2):= I*PI/3:
arcsinh(-I*(3^(1/2))/2):=-I*PI/3:
arcsinh( I*(2^(1/2))/2):= I*PI/4:
arcsinh(-I*(2^(1/2))/2):=-I*PI/4:
arcsinh( I/2):= I*PI/6:
arcsinh(-I/2):=-I*PI/6:
arcsinh( infinity):=  infinity:
arcsinh(-infinity):= -infinity: 

arcsinh:= funcenv(arcsinh, op(specfunc::arcsinh, 2)):
arcsinh::print := "arcsinh":
arcsinh::info := "arcsinh -- the inverse hyperbolic sine":
arcsinh::type := "arcsinh":
arcsinh::float := specfunc::arcsinh:

arcsinh::hull := DOM_INTERVAL::arcsinh@hull:

arcsinh::inverse := "sinh":

arcsinh::undefined := {}:
arcsinh::realDiscont := {}:
arcsinh::complexDiscont :=
  loadproc(arcsinh::complexDiscont, pathname("STDLIB","DISCONT"), "arcsinh"):

arcsinh::expand := x -> arcsinh(expand(op(x))):

arcsinh::simplify:= loadproc(arcsinh::simplify, pathname("STDLIB", "SIMPLIFY"),
                             "arcsinh"):

arcsinh::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    if iszero(op1^2+1) then
       0
    else
      diff(op1, args(2..args(0))) / sqrt(op1^2+1)
    end_if;
  end_proc:

arcsinh::series := loadproc(arcsinh::series, pathname("SERIES"), "arcsinh"):

arcsinh::rectform :=
    loadproc(arcsinh::rectform, pathname("STDLIB","RECTFORM"), "arcsinh"):

arcsinh::getprop :=  proc(x:"arcsinh") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arcsinh(v)); else return(C_); end_if; end_proc:

arcsinh::Content := stdlib::genOutFunc("Carcsinh", 1):

// end of file
