/*++
arctan -- the inverse tangens

arctan(x)
// arctan(u,v), replaced by arg(u,v)

x   - expression
u,v - real quantifiers, representing the complex number u+I*v

arctan(x) computes the inverse tangens of an expression x.

++*/

arctan:=
proc(x)
  option noDebug;
  local f, handleTan; 
begin
  if args(0) = 0 then 
     error("no arguments given")
  elif x::dom::arctan <> FAIL then 
     return(x::dom::arctan(args()))
  elif args(0) > 1 then 
     error("wrong no of args")
  end_if;

  case type(x)
    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(-arctan(-x)) end_if;
      break;

    of DOM_FLOAT do
      return(arctan::float(x));

    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
        return(arctan::float(x))
      end_if;
      if x = I then error("singularity") end_if;
      if op(x,1) = 0 then return(I*arctanh(-I*x)) end_if;
      break;

    of DOM_SET do
    of "_union" do
      return(map(x, arctan));
    of "_intersect" do
    of "_minus" do
       return(Dom::ImageSet(arctan(`#t`), `#t`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(arctan(#t), #t, x));
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;


  // ==========================================
  handleTan:= proc(x) // compute arctan(tan(x))
  local f, s;
  begin
    if (is(x>=-PI/2 and x<=PI/2))=TRUE then
        return(x)
    end:
    s:= -signIm(x):
    if domtype(s) <> DOM_INT then
       return(FAIL);
    end_if:
    f:= s*floor(s*Re(x)/PI + 1/2);
    if type(f) = DOM_INT then
       return(x - f*PI)
    end_if;
    return(FAIL);
  end_proc;
  // ==========================================

  case type(x)
    of "_mult" do  // normalize using arctan(-x) = -arctan(x)
       f:= op(x, nops(x));
       if (testtype(f, Type::Real) and f < 0) or
          (testtype(f/I, Type::Real) and f/I < 0) then
          return(-arctan(-x));
       end_if;
       if type(-x) = "tan" then //-tan(y) = tan(-y)
          f:= handleTan(-op(-x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(-1/x) = "cot" then // -1/cot(y) = tan(-y)
          f:= handleTan(-op(-1/x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(x/I) = "tanh" then //I*tanh(y) = tan(y*I)
          f:= handleTan(I*op(x/I));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(-x/I) = "tanh" then //-I*tanh(y) = tan(-y*I)
          f:= handleTan(-I*op(-x/I));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(I/x) = "coth" then // I/coth(y) = tan(I*y)
          f:= handleTan(I*op(I/x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       if type(-I/x) = "coth" then // -I/coth(y) = tan(-I*y)
          f:= handleTan(-I*op(-I/x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       break;
    of "_power" do
       if type(1/x) = "cot" then // 1/cot(y) = tan(y)
          f:= handleTan(op(1/x));
          if f <> FAIL then
             return(f);
          end_if;
       end_if:
       break;
    of "tan" do // [[ the real part of arctan is in ]-PI/2,PI/2]
       f:= handleTan(op(x));
       if f <> FAIL then 
         return(f);
       end_if:
       break;
  end_case;

  // it is useful to have a normal form such that expressions like
  // arctan(x-y) + arctan(y-x) are simplified automatically. 
  if stdlib::hasmsign(x) then
    -procname(-x)
  else
    procname(x)
  end_if
end_proc:

arctan := prog::remember(arctan,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("arctan")]):

arctan:= funcenv(arctan, op(specfunc::arctan, 2)):
arctan::print := "arctan":
arctan::info :=
  "arctan -- the inverse tangens (including the two-argument function arctan) [try ?arctan for help]":
arctan::type := "arctan":

arctan(0):= 0:
arctan( infinity):=  PI/2:
arctan(-infinity):= -PI/2:
arctan( 3^(1/2)):=  PI/3:
arctan(-3^(1/2)):= -PI/3:
arctan( 1):= PI/4:
arctan(-1):=-PI/4:

// Note that sin(PI/5)/cos(PI/5) yields a more complicated
// expression than tan(PI/5). Add this value to the remember
// table as well:
arctan( (5-2*5^(1/2))^(1/2))         :=   PI/5:
arctan( 5^(1/2)/(5+2*5^(1/2))^(1/2)) :=   PI/5:
arctan( sin(  PI/5)/cos(  PI/5))     :=   PI/5:
arctan(-sin(4*PI/5)/cos(4*PI/5))     :=   PI/5:
arctan(-(5-2*5^(1/2))^(1/2))         := - PI/5:
arctan(-5^(1/2)/(5+2*5^(1/2))^(1/2)) := - PI/5:
arctan(-sin(  PI/5)/cos(  PI/5))     := - PI/5:
arctan( sin(4*PI/5)/cos(4*PI/5))     := - PI/5:
arctan( (1+5^(1/2))*(2*5^(1/2)+10)^(1/2)/4):= 2*PI/5:
arctan( 5^(1/2)/(5-2*5^(1/2))^(1/2))       := 2*PI/5:
arctan( sin(2*PI/5)/cos(2*PI/5))           := 2*PI/5:
arctan(-sin(3*PI/5)/cos(3*PI/5))           := 2*PI/5:
arctan(-(1+5^(1/2))*(2*5^(1/2)+10)^(1/2)/4):=-2*PI/5:
arctan(-5^(1/2)/(5-2*5^(1/2))^(1/2))       :=-2*PI/5:
arctan(-sin(2*PI/5)/cos(2*PI/5))           :=-2*PI/5:
arctan( sin(3*PI/5)/cos(3*PI/5))           :=-2*PI/5:

arctan( (3^(1/2))/3):= PI/6:
arctan(-(3^(1/2))/3):=-PI/6:

arctan( 2^(1/2)-1)              :=   PI/8:
arctan(1/(2^(1/2)+1))           :=   PI/8:
arctan( sin(  PI/8)/cos(  PI/8)):=   PI/8:
arctan(-sin(7*PI/8)/cos(7*PI/8)):=   PI/8:
arctan(-2^(1/2)+1)              := - PI/8:
arctan(-1/(2^(1/2)+1))          := - PI/8:
arctan(1/(-2^(1/2)-1))          := - PI/8:
arctan(-sin(  PI/8)/cos(  PI/8)):=  -PI/8:
arctan( sin(7*PI/8)/cos(7*PI/8)):=  -PI/8:
arctan( 2^(1/2)+1)              := 3*PI/8:
arctan(1/(2^(1/2)-1))           := 3*PI/8:
arctan(-1/(1-2^(1/2)))          := 3*PI/8:
arctan( sin(3*PI/8)/cos(3*PI/8)):= 3*PI/8:
arctan(-sin(5*PI/8)/cos(5*PI/8)):= 3*PI/8:
arctan(-2^(1/2)-1)              :=-3*PI/8:
arctan(-1/(2^(1/2)-1))          :=-3*PI/8:
arctan( 1/(1-2^(1/2)))          :=-3*PI/8:
arctan(-sin(3*PI/8)/cos(3*PI/8)):=-3*PI/8:
arctan( sin(5*PI/8)/cos(5*PI/8)):=-3*PI/8:

// Note that -sin(PI/10) is not exactly the same
// as sin(-PI/10) (no normal form)
arctan( 1/5*5^(1/2)*(5-2*5^(1/2))^(1/2)):=   PI/10:
arctan( 1/(5+2*5^(1/2))^(1/2))          :=   PI/10:
arctan( sin(  PI/10)/cos(  PI/10))      :=   PI/10:
arctan(-1/5*5^(1/2)*(5-2*5^(1/2))^(1/2)):= - PI/10:
arctan(-1/(5+2*5^(1/2))^(1/2))          := - PI/10:
arctan(-sin(  PI/10)/cos(  PI/10))      := - PI/10:
arctan( sin( -PI/10)/cos(  PI/10))      := - PI/10:
arctan( 1/5*5^(1/2)*(5+2*5^(1/2))^(1/2)):= 3*PI/10:
arctan( 1/(5-2*5^(1/2))^(1/2))          := 3*PI/10:
arctan( sin(3*PI/10)/cos(3*PI/10))      := 3*PI/10:
arctan(-sin(7*PI/10)/cos(7*PI/10))      := 3*PI/10:
arctan(sin(-7*PI/10)/cos(7*PI/10))      := 3*PI/10:
arctan(-1/5*5^(1/2)*(5+2*5^(1/2))^(1/2)):=-3*PI/10:
arctan(-1/(5-2*5^(1/2))^(1/2))          :=-3*PI/10:
arctan(-sin(3*PI/10)/cos(3*PI/10))      :=-3*PI/10:
arctan(sin(-3*PI/10)/cos(3*PI/10))      :=-3*PI/10:
arctan( sin(7*PI/10)/cos(7*PI/10))      :=-3*PI/10:

arctan( 2-3^(1/2))                  :=   PI/12:
arctan( 1/( 2+3^(1/2)))             :=   PI/12:
arctan(-1/(-2-3^(1/2)))             :=   PI/12:
arctan( sin(  PI/12)/cos(  PI/12))  :=   PI/12:
arctan(-sin(11*PI/12)/cos(11*PI/12)):=   PI/12:
arctan(sin(-11*PI/12)/cos(11*PI/12)):=   PI/12:
arctan(-2+3^(1/2))                  := - PI/12:
arctan( 1/(-2-3^(1/2)))             := - PI/12:
arctan(-1/( 2+3^(1/2)))             := - PI/12:
arctan(-sin(  PI/12)/cos(  PI/12))  := - PI/12:
arctan( sin( -PI/12)/cos(  PI/12))  := - PI/12:
arctan( sin(11*PI/12)/cos(11*PI/12)):= - PI/12:
arctan( 2+3^(1/2))                  := 5*PI/12:
arctan( 1/(2-3^(1/2)))              := 5*PI/12:
arctan(-1/(3^(1/2)-2))              := 5*PI/12:
arctan(-sin(7*PI/12)/cos(7*PI/12))  := 5*PI/12:
arctan(sin(-7*PI/12)/cos(7*PI/12))  := 5*PI/12:
arctan( sin(5*PI/12)/cos(5*PI/12))  := 5*PI/12:
arctan(-2-3^(1/2))                  :=-5*PI/12:
arctan(-1/(2-3^(1/2)))              :=-5*PI/12:
arctan( 1/(3^(1/2)-2))              :=-5*PI/12:
arctan(-sin(5*PI/12)/cos(5*PI/12))  :=-5*PI/12:
arctan(sin(-5*PI/12)/cos(5*PI/12))  :=-5*PI/12:
arctan( sin(7*PI/12)/cos(7*PI/12))  :=-5*PI/12:

arctan( I*infinity):=  PI/2:
arctan(-I*infinity):= -PI/2:

//-------------------------------------------------------------------
arctan::inverse := "tan":

//-------------------------------------------------------------------
arctan::undefined := {}:
arctan::realDiscont := {}:
arctan::complexDiscont :=
  loadproc(arctan::complexDiscont, pathname("STDLIB","DISCONT"), "arctan"):

//-------------------------------------------------------------------
arctan::float := specfunc::arctan@float:

// old version includion 2-argument version of arctan:
//arctan::float:= proc(x,y)
//begin
//   if args(0)=1 then
//        specfunc::arctan(float(x))
//   elif args(0)=2 then
//        arctan(float(x),float(y))
//   else error("Wrong number of arguments")
//   end_if
//end_proc:

arctan::hull := DOM_INTERVAL::arctan@hull:

//-------------------------------------------------------------------

arctan::expand := x -> arctan(expand(op(x))):

arctan::simplify:= loadproc(arctan::simplify, pathname("STDLIB", "SIMPLIFY"),
                            "arctan"):

//-------------------------------------------------------------------
arctan::diff :=
  proc() 
  local op1;
  begin
    if nops(args(1)) = 1 then 
       op1:= op(args(1),1)
    else 
       op1:= op(args(1),2)/op(args(1),1)
    end_if;
    if iszero(1+op1^2) then
       0
    else
       diff(op1, args(2..args(0))) / (1+op1^2)
    end_if;
  end_proc:

//-------------------------------------------------------------------
arctan::series :=
    loadproc(arctan::series, pathname("SERIES"), "arctan"):

//-------------------------------------------------------------------
arctan::rectform :=
    loadproc(arctan::rectform, pathname("STDLIB","RECTFORM"), "atan"):

arctan::"transform::invlaplace":=
    loadproc( arctan::"transform::invlaplace",
        pathname("TRANS","LAPLACE"), "IL_atan"):
arctan::getprop :=  proc(x:"arctan") local v; begin v := getprop(op(x)); if contains({Dom::Interval, solvelib::BasicSet}, type(v)) then return(arctan(v)); else return(C_); end_if; end_proc:

arctan::Content := stdlib::genOutFunc("Carctan", 1):

// end of file
