/*++
Bernoulli numbers and polynomials

bernoulli(n)
bernoulli(n, x)

n - a nonnegative integer
x - an expression
++*/

bernoulli :=
proc(n, x)
  local t, k, s, k0, y, x2;
begin
  case args(0)
    of 1 do
       if domtype(n) = DOM_FLOAT and
          iszero(frac(n)) then
          n:= round(n);
       end_if:
       if n = 0 or n = 1 then
          return(specfunc::bernoulliNumber(n));
       end_if:
       if domtype(n) = DOM_INT and n >= 2 then
          if n mod 2 = 1 then
             return(0)
          elif n <= Pref::autoExpansionLimit() then
             return(specfunc::bernoulliNumber(n));
          else break;
          end_if:
       end_if;
       if testtype(float(n), Type::Numeric) then
          error("the first argument must be symbolic or a nonnegative integer")
       end_if;
       if not testtype(n,Type::Arithmetical) then
          if testtype(n, Type::Set) then
            return(Dom::ImageSet(bernoulli(#n), #n, n intersect Z_ intersect Dom::Interval([0], infinity)));
          end_if;
          error("the first argument must be of type 'Type::Arithmetical'")
       end_if;
       return(procname(n));
    of 2 do
        if not testtype(n,Type::Arithmetical) then
          if testtype(n, Type::Set) then
            if testtype(x,Type::Arithmetical) then
              return(Dom::ImageSet(bernoulli(#n, x), #n, n intersect Z_ intersect Dom::Interval([0], infinity)));
            elif testtype(x,Type::Set) then
              return(Dom::ImageSet(bernoulli(#n, #x), [#n, #x], [n intersect Z_ intersect Dom::Interval([0], infinity), x]));
            end_if;
          end_if;
          error("the first argument must be of type 'Type::Arithmetical'")
        elif not testtype(x,Type::Arithmetical) then
          if testtype(x,Type::Set) then
            return(Dom::ImageSet(bernoulli(n, #x), #x, x));
          end_if;
          error("the second argument must be of type 'Type::Arithmetical'")
        end_if;

        // most probable case first: n is a nonnegative integer
        if domtype(n) = DOM_INT and n>=0 then
           // special cases, no need to evaluate the polynomial:
           if n = 0 then return(1)
           elif n = 1 then return(x - 1/2)
           elif n = 2 then return(x^2 - x + 1/6)
           elif n = 3 then return(x^3 - 3/2*x^2 + x/2)
           elif n = 4 then return(x^4 - 2*x^3 + x^2 -1/30)
           elif x = 0  then return(specfunc::bernoulliNumber(n))
           elif x = float(0)  then return(float(specfunc::bernoulliNumber(n)))
           elif x = 1 then return((-1)^n*specfunc::bernoulliNumber(n))
           elif x = float(1) then return(float((-1)^n*specfunc::bernoulliNumber(n)))
           elif x = 1/2 then return((2^(1-n)-1)*specfunc::bernoulliNumber(n)) 
           elif x = float(1/2) then return(float((2^(1-n)-1)*specfunc::bernoulliNumber(n))) 
           elif x = 1/3 and domtype(n/2)=DOM_INT then
                return((3^(1-n)-1)/2*specfunc::bernoulliNumber(n)) 
           elif x = float(1/3) and domtype(n/2)=DOM_INT then
                return(float((3^(1-n)-1)/2*specfunc::bernoulliNumber(n))) 
           elif x = 2/3 and domtype(n/2)=DOM_INT then
                return((3^(1-n)-1)/2*specfunc::bernoulliNumber(n)) 
           elif x = float(2/3) and domtype(n/2)=DOM_INT then
                return(float((3^(1-n)-1)/2*specfunc::bernoulliNumber(n))) 
           elif x = 1/4 and domtype(n/2)=DOM_INT then
                return((2^(1-n)-1)/2^n*specfunc::bernoulliNumber(n)) 
           elif x = float(1/4) and domtype(n/2)=DOM_INT then
                return(float((2^(1-n)-1)/2^n*specfunc::bernoulliNumber(n)))
           elif x = 3/4 and domtype(n/2)=DOM_INT then
                return((2^(1-n)-1)/2^n*specfunc::bernoulliNumber(n)) 
           elif x = float(3/4) and domtype(n/2)=DOM_INT then
                return(float((2^(1-n)-1)/2^n*specfunc::bernoulliNumber(n))) 
           elif x = 1/6 and domtype(n/2)=DOM_INT then
                return((2^(1-n)-1)*(3^(1-n)-1)/2*specfunc::bernoulliNumber(n)) 
           elif x = float(1/6) and domtype(n/2)=DOM_INT then
                return(float((2^(1-n)-1)*(3^(1-n)-1)/2*specfunc::bernoulliNumber(n)))
           elif x = 5/6 and domtype(n/2)=DOM_INT then
                return((2^(1-n)-1)*(3^(1-n)-1)/2*specfunc::bernoulliNumber(n)) 
           elif x = float(5/6) and domtype(n/2)=DOM_INT then
                return(float((2^(1-n)-1)*(3^(1-n)-1)/2*specfunc::bernoulliNumber(n)))
           end_if:

           //====================================================
           // bernoulli::float only works for real x between 0 and 1.
           // There, however, it is much faster than the evaluation
           // of the polynomial further down below.
           //====================================================
           if domtype(x) = DOM_FLOAT and
              x > 0 and
              x < 1 then
              return(bernoulli::float(n, x));
           end_if:

           //====================================================
           // evaluate the polynomial, use
           // B(n,x) = sum(binomial(n,k)*x^k*B(n-k), k=0..n)
           //====================================================
           if n mod 2 = 0
           then  y:= 1:
                 t:= 1;
                 s:= specfunc::bernoulliNumber(n);
                k0:= 2;
           else  y:= x:
                 t:= n;
                 s:= t*specfunc::bernoulliNumber(n-1)*y:
                k0:= 3;
           end_if;
           //compute  sum(binomial(n,k)*x^k*B(n-k), k=0..n-2)
           x2:= x^2:
           for k from k0 to n-2 step 2 do
             y:= y*x2;
             t:= t*(n-k+2)*(n-k+1)/k/(k-1);
             s:= s + t*specfunc::bernoulliNumber(n - k)*y;
           end_for:
           // add the last 2 terms:
           y:= y*x;
           t:= t/(n-1);
           s:= s - t*y;
           y:= y * x;
           t:= 2*t/n;
           return(s+t*y);
        end_if;

        //============================================================
        // Various simplifications for symbolic values of n.
        // The call bernoulli(n) checks, if n is valid:
        //============================================================

        if   x = 0 then return(bernoulli(n))
        elif x = float(0) then return(float(bernoulli(n)))
        elif x = 1 then return((-1)^n*bernoulli(n))
        elif x = float(1) then return(float((-1)^n*bernoulli(n)))
        elif x = -1 then return(bernoulli(n) + n*(-1)^n)
        elif x = float(-1) then return(float(bernoulli(n) + n*(-1)^n))
        elif x = 1/2 then return((2^(1-n)-1)*bernoulli(n)) 
        elif x = float(1/2) then return(float((2^(1-n)-1)*bernoulli(n)))
        elif x = 1/3 and domtype(n/2)=DOM_INT then
             return((3^(1-n)-1)/2*bernoulli(n)) 
        elif x = float(1/3) and domtype(n/2)=DOM_INT then
             return(float((3^(1-n)-1)/2*bernoulli(n)))
        elif x = 2/3 and domtype(n/2)=DOM_INT then
             return((3^(1-n)-1)/2*bernoulli(n)) 
        elif x = float(2/3) and domtype(n/2)=DOM_INT then
             return(float((3^(1-n)-1)/2*bernoulli(n))) 
        elif x = 1/4 and domtype(n/2)=DOM_INT then
             return((2^(1-n)-1)/2^n*bernoulli(n)) 
        elif x = float(1/4) and domtype(n/2)=DOM_INT then
             return(float((2^(1-n)-1)/2^n*bernoulli(n)))
        elif x = 3/4 and domtype(n/2)=DOM_INT then
             return((2^(1-n)-1)/2^n*bernoulli(n)) 
        elif x = float(3/4) and domtype(n/2)=DOM_INT then
             return(float((2^(1-n)-1)/2^n*bernoulli(n)))
        elif x = 1/6 and domtype(n/2)=DOM_INT then
             return((2^(1-n)-1)*(3^(1-n)-1)/2*bernoulli(n)) 
        elif x = float(1/6) and domtype(n/2)=DOM_INT then
             return(float((2^(1-n)-1)*(3^(1-n)-1)/2*bernoulli(n)))
        elif x = 5/6 and domtype(n/2)=DOM_INT then
             return((2^(1-n)-1)*(3^(1-n)-1)/2*bernoulli(n)) 
        elif x = float(5/6) and domtype(n/2)=DOM_INT then
             return(float((2^(1-n)-1)*(3^(1-n)-1)/2*bernoulli(n)))
        elif testtype(x, Type::Real) and x > 1/2 and x < 1 then
             return((-1)^n*bernoulli(n,1-x))
        elif testtype(x, Type::Real) and x < 0 then
             return((-1)^n*bernoulli(n,-x) + n*(-1)^n*(-x)^(n-1))
        // errors: 
        elif testtype(float(n), Type::Numeric) then
             error("the first argument must be symbolic or a nonnegative integer")
        // symbolic return
        else return(procname(n, x))
        end_if;
        break;
    otherwise
        error("1 or 2 arguments expected.")
  end_case;
  procname(args());
end_proc:

bernoulli := prog::remember(bernoulli, 
    () -> [property::depends(args()), DIGITS, Pref::autoExpansionLimit(),
           slotAssignCounter("bernoulli")]):

bernoulli:= funcenv(bernoulli):
bernoulli::type:= "bernoulli":
bernoulli::print:= "bernoulli":

bernoulli::conjugate:=
  proc(n, x)
  begin
    if args(0) = 1 then 
         bernoulli(n)
    else bernoulli(n, conjugate(x))
    end_if;
  end_proc:

//----------------------------------------------------------------
// bernoulli::float
// This is just a utility called by bernoulli(n, x) for numerical n
// and x = real floating point value between 0 and 1.
// It uses the series expansion
//  bernoulli(2*n-1, x) = (-1)^n*2*(2*n-1)!/(2*PI)^(2*n-1) *
//                  sum( sin(2*k*PI*x)/k^(2*n-1), k = 1 .. infinity)
//  bernoulli(2*n, x) = (-1)^(n-1)*2*(2*n)!/(2*PI)^(2*n) *
//                  sum( cos(2*k*PI*x)/k^(2*n), k = 1 .. infinity)
// This computation is much faster than the direct computation
// of bernoulli(n, x) as a finite sum.
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// !!!! Only use for 0 < x < 1.!!!!!
// !!!! It produces completely !!!!!
// !!!! wrong values for other !!!!!
// !!!! values of x!           !!!!!
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//----------------------------------------------------------------
bernoulli::float:= proc(n, x)
local tol, fPI2, fPIx2, s, k, n2, a, b, ds;
begin
   if args(0) = 1 then
      return(hold(bernoulli)(n));
   end_if:
   x:= float(x):
   if domtype(n) = DOM_INT and
      domtype(x) = DOM_FLOAT and
      x > 0 and
      x < 1 then
     tol:= 10^(-DIGITS);
     fPI2:= 2*float(PI);
     fPIx2:= fPI2*x;
     s:= 0:
     k:= 1:
     if n mod 2 = 0 then // n is even
        [n2, n]:= [n, n/2];
        while TRUE do
          a:= cos::float(k*fPIx2):
          b:= 1/k^n2;
          ds:= a*b;
          if max(b, specfunc::abs(ds)) <= tol*specfunc::abs(s) then
             return((-1)^(n-1)*2*fact(n2)/fPI2^n2*(s + ds))
          end_if:
          s:= s + ds;
          k:= k + 1;
        end_while;
      else // n is odd
        [n2, n]:= [n, (n+1)/2];
        while TRUE do
          a:= sin::float(k*fPIx2):
          b:= 1/k^n2;
          ds:= a*b;
          if max(b, specfunc::abs(ds)) <= tol*specfunc::abs(s) then
             return((-1)^n*2*fact(n2)/fPI2^n2*(s + ds))
          end_if:
          s:= s + ds;
          k:= k + 1;
        end_while;
      end_if:
   end_if:
   return(hold(bernoulli)(n, x));
end_proc:

//----------------------------------------------------------------
// expand implements 
// (1) bernoulli(n, x + 1) = n*x^(n-1) + bernoulli(n, x)
// (2) bernoulli(n, m*x) = m^(n-1)*sum(bernoulli(n, x + k/m), k = 0 .. m-1)
// (3) bernoulli(n, -x) = (-1)^n*bernoulli(n, x) + n*(-1)^n*x^(n-1)
//----------------------------------------------------------------
bernoulli::expand:= proc(e)
local n, x, m, k, s;
begin
  if nops(e) = 1 then
     x:= op(e, 1);
     if domtype(x) = DOM_INT and x > 0 then
        return(specfunc::bernoulliNumber(x))
     else
        return(e)
     end_if:
  end_if:
  [n, x]:= map([op(e)], expand);

  case type(x)
     of "_plus" do
       //----------------------------------------------------------
       // implement rule (1): bernoulli(n, x + 1) = n*x^(n-1) + bernoulli(n, x)
       //----------------------------------------------------------
       m:= select(x,  testtype, Type::Numeric);
       m:= floor(Re(m));
       if m >= 1 then
          //bernoulli(n, x) = bernoulli(n, x-1) + n*(x-1)^(n-1)
          //                = bernoulli(n, x-2) + n*(x-2)^(n-1) + n*(x-1)^(n-1)
          //                = bernoulli(n, x-3) + n*(x-3)^(n-1) + n*(x-2)^(n-1) + n*(x-1)^(n-1)
          //                = ...
          return(expand(bernoulli(n, x-m) + _plus(n*(x-k)^(n-1) $ k = 1..m)))
       elif m = 0 then
          // Do nothing. Fall through to the code for expand(bernoulli(n, -x));
          break;
       elif m < 0 then
          //bernoulli(n, x) = bernoulli(n, x+1) - n*(x + 0)^(n-1)
          //                = bernoulli(n, x+2) - n*(x + 1)^(n-1) - n*(x+0)^(n-1)
          //                = bernoulli(n, x+3) - n*(x + 2)^(n-1) - n*(x+1)^(n-1) - n*(x+0)^(n-1)
          //                = ...
          return(expand(bernoulli(n, x-m) + _plus(-n*(x+k)^(n-1) $ k = 0..-m-1)))
       end_if:
     of "_mult" do
       //-------------------------------------------------------
       // implement rule (2):
       //   bernoulli(n, m*x) = m^(n-1)*sum(bernoulli(n, x + k/m), k = 0 .. m-1)
       //-------------------------------------------------------
       m := op(x, nops(x));
       // Do rule (2), if m is integer and x contains at least one symbol,
       // because we do not want to expand bernoulli(n, 5*sqrt(2)) etc.
       if indets(x) minus Type::ConstantIdents = {} then
         break;
       end_if:
       if type(m) = DOM_INT then
          if m < 0 then
             return(expand(-n*x^(n-1) + (-1)^n*bernoulli(n, -x)))
          else // m > 0
             x := x / m;
             return(expand(m^(n-1)*_plus(bernoulli(n, x + k/m) $ k = 0..m-1)));
          end_if;
       end_if;
       break;
  end_case:

  //-------------------------------------------------------
  // implement rule (3): 
  //    bernoulli(n, -x) = (-1)^n*bernoulli(n, x) + n*(-1)^n*x^(n-1)
  //-------------------------------------------------------
  [s, x]:= stdlib::normalizesign(x);
  if s = -1 then
     return(expand(n*(-1)^n*x^(n-1) + (-1)^(n)*bernoulli(n, x)))
  else
     return(bernoulli(n, x));
  end_if:
end_proc:

//----------------------------------------------------------------
// bernoulli::diff implements bernoulli'(n, x) = n*bernoulli(n-1, x)
//----------------------------------------------------------------
bernoulli::diff:= proc(e, x)
local n, f;
begin
  if has(op(e, 1), x) then
     return(hold(diff)(args()))
  end_if;
  if not has(e, x) then
     return(0);
  end_if:
  [n, f]:= [op(e)];
  return(n*bernoulli(n-1, f)*diff(f, x));
end_proc:


//----------------------------------------------------------------
// auxiliary routine for the Bernoulli numbers uses ideas from
// Hans Riesel: An exact formula for the 2n-th Bernoulli number
// Acta arithmetica, volume 26 (1975), pages 273-277
specfunc::bernoulliNumber:=
proc(n: Type::PosInt)
  local Q, prefact, bound, zetaval, k,
  bernoullDenom: DOM_PROC;
  save DIGITS;
  option remember;
begin
  if n mod 2 <> 0 then
    return(0)
  else
    n:= n div 2
  end_if;
  
  // local method
  // returns the denominator of the  2n - th Bernoulli number
  // it is known that this one equals the product of all primes p
  // for which p-1 divides 2n
  bernoullDenom:=
  proc(n: DOM_INT): DOM_INT
  begin
    _mult(op(select(map(numlib::divisors(2*n), _plus, 1), isprime)))
  end_proc:

  
  DIGITS:= 8;
  Q:= bernoullDenom(n);

  // looking for the absolute value of the numerator
  // Q*bernoulli(2*n) =( Q* (2*n)! / 2^(2*n-1)/PI^(2*n) ) * zeta(2*n)
  // first, we compute the first factor with low precision
  prefact:= gamma(float(2*n+1)) * Q/2.0^(2*n-1)/float(PI)^2*n;
  // we know that our numerator must be an integer; hence it suffices
  // to sum the zeta series up to a bound such that
  // sum(k^(-2*n), k=bound..infinity) * prefact  < 1
  // the left hand side is < 2*bound^(-2*n) * prefact
  // hence we need bound^(2*n) > 2*prefact
  bound:= ceil((2*prefact)^(1/2/n));
  userinfo(20, "Bound:".expr2text(bound));
  // since zeta equals 1 (rather precisely), we need as many digits as
  // prefact has in front of the decimal point
  // add some safety digits
  DIGITS:= max(8, ceil(ln(prefact) / ln(10.0)) + 5);
  userinfo(20, "Setting DIGITS to ".expr2text(DIGITS));
  prefact:= expand(fact(2*n)) * Q/2.0^(2*n-1)/float(PI)^(2*n);
  zetaval:= _plus(float(k^(-2*n)) $k=1..bound);
  (-1)^(n-1)*ceil(prefact*zetaval) / Q
end_proc:

specfunc::bernoulliNumber(0):= 1:
specfunc::bernoulliNumber(1):= -1/2:
specfunc::bernoulliNumber(2):=  1/6:
specfunc::bernoulliNumber(4):= -1/30:
specfunc::bernoulliNumber(6):=  1/42:
specfunc::bernoulliNumber(8):= -1/30:
specfunc::bernoulliNumber(10):=  5/66:

/* end of file */
