/*++
 besselJ(v,z) is the bessel function of the first kind:
 besselJ(v,z) = 1/gamma(v+1)*(z/2)^v*0F1(v+1; -z^2/4)
    = (z/2)^v * sum( (-z^2/4)^k/k!/gamma(v+1+k), k=0..infinity)
++*/

besselJ:=
proc(v,z)
  local cv,sv,fv,fz,r,r2,t1,t2,z2,sz,Vv,zz,dummy,f;
  option noDebug;
begin
  if args(0) <> 2 then
    error("expecting two arguments")
  elif z::dom::besselJ <> FAIL then
    return( z::dom::besselJ(args()) )
  elif {DOM_BOOL,DOM_FAIL,DOM_NULL,DOM_NIL} intersect
       {domtype(v), domtype(z)} <> {} then
    error("invalid operand")
  end_if;

  // ----------  map to sets -------------------------
  // This is quite complicated, because both arguments have
  // to be handled
  case [type(v),type(z)]
    of [DOM_SET, DOM_SET] do
        return({besselJ(sv, sz) $ sv in v $ sz in z});
    of [DOM_SET, "_union"] do
    of ["_union", DOM_SET] do
        return(_union((besselJ(sv, sz) $ sv in v) $ sz in z));
    of ["_union", "_union"] do

        // besselJ({a} union A, C union D)
        //  -> besselJ({a},C) union besselJ({a},D) union besselJ(A,C) union besselJ(A,D)
        // Make sure that A, C, D are interpreted as sets, i.e.,
        // besselJ({a},C) --> besselJ(a, C), not {besselJ(a,C)} !

        [v, Vv, dummy]:= split(v, testtype, DOM_SET):
        if type(Vv) <> "_union" then Vv:= [Vv] end_if:
        [z, zz, dummy]:= split(z, testtype, DOM_SET):
        if type(zz) <> "_union" then zz:= [zz] end_if:

        return(_union({(besselJ(sv, sz) $ sv in v ) $ sz in z},
                       (besselJ(sv, sz) $ sv in Vv) $ sz in z,
                       (besselJ(sv, sz) $ sv in v ) $ sz in zz,
                       (besselJ(sv, sz) $ sv in Vv) $ sz in zz ));
  end_case;

  case type(v)
    of DOM_SET do
    of "_union" do
       // z cannot be a set, if v is a set
       return(map(v, besselJ, z));
  end_case;

  case type(z)
    of DOM_SET do
    of "_union" do
       // v cannot be a set, if z is a set
       return(map(z, (z,v) -> besselJ(v,z), v))
  end_case;
  // --------------------------------------------------
  if testtype(z, Type::Set) and not testtype(z, Type::Arithmetical) then
    if testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselJ(#v, #z), [#z, #v], [z, v]));
    elif testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselJ(v, #z), [#z], [z]));
    end_if;
  elif testtype(v, Type::Set) and not testtype(v, Type::Arithmetical) then
      return(Dom::ImageSet(besselJ(#v, z), [#v], [v]));
  end_if;
  if map([v, z], testtype, Type::Arithmetical) <> [TRUE, TRUE] then
    error("arguments must be of type 'Type::Arithmetical'")
  end_if:
  // --------------------------------------------------

  if z=infinity then
    return(0);
  end_if:
  fv:= domtype(float(v)):
  fz:= domtype(float(z)):
  if (fv=DOM_FLOAT or fv=DOM_COMPLEX) and 
     (fz=DOM_FLOAT or fz=DOM_COMPLEX) and 
     has([map((op(v),op(z)),type)],DOM_FLOAT) then
      return(besselJ::float(v,z))
  end_if:

  if iszero(z) then
     if iszero(v) then
        return(1)
     elif (fv=DOM_FLOAT or fv=DOM_COMPLEX) and 
          (Re(float(v)) > 0 or (domtype(v)=DOM_INT or (domtype(v)=DOM_FLOAT and iszero(v-round(v))))) 
    then return(0):
 elif (fv=DOM_FLOAT or fv=DOM_COMPLEX) and 
       not(Re(float(v)) > 0 or 
       (domtype(v)=DOM_INT or (domtype(v)=DOM_FLOAT and iszero(v-round(v))))) then
        error("singularity"):
      else
        return(procname(v,z));
      end_if:
  end_if:

  // implement the reflection formulas for integer index v:
  if domtype(v)=DOM_INT or 
    (domtype(v) = DOM_FLOAT and iszero(v - round(v))) then
     case type(z)
     of DOM_INT do
     of DOM_RAT do 
        if z < 0 and v < 0 then return(       procname(-v, -z)); end_if;
        if z < 0 and v >=0 then return((-1)^v*procname( v, -z)); end_if;
        if z >=0 and v < 0 then return((-1)^v*procname(-v,  z)); end_if;
      //if z >=0 and v >=0 then return(       procname( v,  z)); end_if;
     of "_mult" do
      f:= op(z, nops(z)); // f = the numerical factor in a product
      if testtype(f, Type::Real) then
        if f < 0 and v < 0 then return(       procname(-v, -z)); end_if;
        if f < 0 and v >=0 then return((-1)^v*procname( v, -z)); end_if;
        if f >=0 and v < 0 then return((-1)^v*procname(-v,  z)); end_if;
      //if f >=0 and v >=0 then return(       procname( v,  z)); end_if;
      end_if;
     of DOM_IDENT do
      if v < 0 then return((-1)^v*procname(-v,z)); end_if;
     end_case:
  end_if:

  if v=1/2 then
     return(2^(1/2)/PI^(1/2)/z^(1/2)*sin(z));
  elif v=-1/2 then
     return(2^(1/2)/PI^(1/2)/z^(1/2)*cos(z)); 
  elif type(v) <> DOM_INT and type(2*v)=DOM_INT then
      // might want to limit this expansion to say abs(v) <= 10
      // and let "expand" take care of cases > 10.
      //cv:= cos((v+1/2)*PI/2);
      //sv:= sin((v+1/2)*PI/2);
      case (v+1/2) mod 4
        of 0 do cv:= 1: sv:= 0: break;
        of 1 do cv:= 0: sv:= 1: break;
        of 2 do cv:=-1: sv:= 0: break;
        of 3 do cv:= 0: sv:=-1: break;
      end_case;

      if v<1/2
          then v:= -v-1/2:
          else v := v-1/2;
      end_if:
      z2:= z^2:
      t1:= 1:
      t2:= 1:
      return(
         2^(1/2)/PI^(1/2)/sqrt(z)*((cos(z)*cv+sin(z)*sv)*
      _plus(1,(r2:=2*r;t1:=-t1/8*(v+r2-1)*(v+r2)*(-v+r2-1)*(-v+r2-2)/z2/r/(r2-1))
                    $ r=1..trunc(v/2))
        +(cos(z)*sv-sin(z)*cv)* v*(v+1)/2/z*
      _plus(1,(r2:=2*r;t2:=-t2/8*(v+r2)*(v+r2+1)*(-v+r2)*(-v+r2-1)/z2/r/(r2+1))
                    $ r=1..trunc((v-1)/2))));
  else 
      procname(v,z) 
  end_if
end_proc:

besselJ := prog::remember(besselJ,
  () -> [property::depends(args()), DIGITS, slotAssignCounter("besselJ")]):

besselJ := funcenv(besselJ):
besselJ::type := "besselJ":
besselJ::print := "besselJ":
besselJ::info :=
  "besselJ(v,z) -- bessel function of first kind, order v and argument z":

besselJ::float :=
    loadproc(besselJ::float, pathname("STDLIB","FLOAT"), "besselJ"):

besselJ::series :=
    loadproc(besselJ::series, pathname("SERIES"), "besselJ"):

besselJ::expand :=
    loadproc(besselJ::expand, pathname("STDLIB","EXPAND"), "besselJ"):

besselJ::"transform::laplace":=
    loadproc( besselJ::"transform::laplace",
        pathname("TRANS","LAPLACE"), "L_bessJ"):


besselJ::Content := stdlib::genOutFunc("CbesselJ", 2):

besselJ::MMLContent :=
(Out, data) -> specfunc::Bessel::MMLContent(Out, data, "J"):


/* ----------------------------
//================================================================
// The diff2 slot serves for overloading diff calls with more than
// 2 arguments (if diff2 <> FAIL, then use it, otherwise fall back
// to the usual diff slot). For the Bessels, we just do an expand
// after each single differentiation, which reduces the expression
// swell considerably.
//================================================================
besselJ::diff2:= proc(e /* , x, x, ... */)   // higher derivatives 
local res, i;
begin
  res:= e:
  for i from 3 to args(0) do
      res:= expand(diff(res, args(i)));
  end_for:
  return(res);
end_proc:
-----------------------------*/

//================================================================
// There are 2 formulas for diff of the bessels:
// (1) diff(besselJ(v,x), x) =  besselJ(v-1,x) - v/x*besselJ(v,x)  
// (2) diff(besselJ(v,x), x) = -besselJ(v+1,x) + v/x*besselJ(v,x)
// We choose the 'downward' formula (1), if v >= 1 or
// v = something symbolic + number with number >= 1.
// Otherwise, the 'upward' formula (2) is used.
//================================================================
besselJ::diff := proc(e, x)
   local v, f, dfdx;
begin
   // We cannot differentiate w.r.t. the index (=op(e, 1)):
   if has(op(e,1),[args(2..args(0))]) then 
      return(hold(diff)(args()))
   end_if:
   if args(0)>2 then 
      return(diff(diff(e,x),args(3..args(0))));
   end_if:
   v:=op(e,1): // the index of besselJ
   f:=op(e,2): // the argument of besselJ
   dfdx:= diff(f, x):
   if iszero(dfdx) then
      return(0)
   end_if:
   // Switch between the 'upward' and the 'downward' formula to 
   // avoid drifting towards high values of v when computing 
   // high derivatives. For the decision, split v into 
   //   v = symbolic/imaginary + Type::Real
   if besselJ::split_index(v)[2] > 0 then // downward
     return( besselJ(v-1,f)*dfdx - v/f*besselJ(v, f)*dfdx);
   else // upward
     return(-besselJ(v+1,f)*dfdx + v/f*besselJ(v, f)*dfdx);
   end_if;
end_proc:

//=======================================================
// Split the index v of bessel into v = s + n, where s is
// something symbolic or imaginary and n is a real number
// of Type::Real = {integers, rationals, floats}. 
// The strategy is:
// 1) check if v is Type::Real (without using properties (used in Re)).
// 3) check if v = symbolic + n with n of Type::Real
// 2) check if Re(v) is Type::Real (slower, Re uses properties!) 
// 4) check if Re(v) = symbolic + n with n of Type::Real
// Return [s, n] with v = s + n, n of Type::Real
//=======================================================
besselJ::split_index:= proc(v)
local n, Rev;
option remember;
begin
   v:= expand(v):
   if testtype(v, Type::Real) then
      n:= v;
      return([0, n]);
   elif testtype(v, "_plus") then
      n:= select(v, testtype, Type::Real):
      if not iszero(n) then
         return([v-n, n]);
   // else need to investigate the real parts of numbers
      end_if;
   end_if;
   Rev:= Re(v);
   if testtype(Rev, Type::Real) then
      n:= Rev;
      return([v - n, n]);
   elif testtype(Rev, "_plus") then
      n:= select(Rev, testtype, Type::Real):
      return([v-n, n]);
   end_if:
   return([v, 0]);
end_proc:

//==========================================================
// The simplify attribute of besselJ(v, x) splits the index
// v=s+n, where s is symbolic/imginary and n is of Type::Real.
// The recursion besselJ(v-1,x)+besselJ(v+1,x)=2*v/x*besselJ(v,x)
// is used upwards or downwards to reduce n to a value n0 with
// -1 < n0 <= 1. 
// Thus, 
//   besselJ(s+n,x) = c0(n,x)*besselJ(s,x) + c1(n,x)*besselJ(s+1, x)
// where c0(n, x) and c1(n, x) are rational expressions in x:
//==========================================================
besselJ::simplify:= proc(f)
local v, x, vx, n, cn, maxn;
begin // f = besselJ(v, x) or besselY(v, x)
   if not type(f) in {"besselJ", "besselY"} then
      return(f);
   end_if:
   // First, simplify the arguments
   [v, x]:= map([op(f)], simplify, op(args(2..args(0)))); 
   // Check, if the simplified arguments yields an explicit result
   f:=  eval(subsop(f, 1 = v, 2 = x));
   // If there is an explicit result: run!
   if not contains({"besselJ", "besselI", "besselK", "besselY"}, type(f)) then
      return(f);
   end_if;
   vx:= [v, x]:
   [v, n]:= besselJ::split_index(v);
   f:= op(f, 0):
   cn:= ceil(n);
   v:= v - (cn - n);
   n:= cn;
   // if n is large, the reduction to besselJ(v, x) and besselJ(v+1,x)
   // will be quite costly. If arithmetic with s or x can cause 
   // expression swell, we should restrict the shifts n. If s or x
   // are of Type::Numeric, there is no expression swell, but only
   // basic arithmetic with O(n) operations, i.e., n may be larger.
   if testtype(v, Type::Numeric) and 
      testtype(x, Type::Numeric) then
      maxn:= 10^4: // hardcoded limit. Shouldn't we make if configurable?
   else
      maxn:= 20:   // hardcoded limit. Shouldn't we make if configurable?
   end_if:
   if specfunc::abs(n) > maxn then
      [v, x]:= vx:
      if f = hold(besselJ) and stdlib::hasmsign(x) then
        return(x^v/(-x)^v*f(v, -x));
      else
        return(f(v, x));
      end_if:
   else
      if f = hold(besselJ) and stdlib::hasmsign(x) then
        return( besselJ::recurrenceCoeff(v,n,x,[1,0])*x^v/(-x)^v*f( v ,-x)
               -besselJ::recurrenceCoeff(v,n,x,[0,1])*x^v/(-x)^v*f(v+1,-x));
      else
        return( besselJ::recurrenceCoeff(v,n,x,[1,0])*f( v ,x)
               +besselJ::recurrenceCoeff(v,n,x,[0,1])*f(v+1,x));
      end_if:
   end_if:
end_proc:

//==========================================================
besselJ::Simplify:= besselJ::simplify:

//==========================================================
// utility for computing the coefficients c0(v,x), c1(v,x) in the
// identity
//  besselJ(v+n,x) = c0(v,x)*besselJ(v,x) + c1(v,x)*besselJ(v+1,x),
//  besselY(v+n,x) = c0(v,x)*besselY(v,x) + c1(v,x)*besselY(v+1,x),
//  c0(v,x) = recurrenceCoeff(v,n,x,[1,0])
//  c1(v,x) = recurrenceCoeff(v,n,x,[0,1])
//==========================================================
besselJ::recurrenceCoeff:= proc(v, n : DOM_INT, x, c0c1)
local c0, c1, k;
begin
  [c0, c1]:= c0c1:
  if n > 1 then
    for k from 1 to n-1 do
      [c1, c0]:= [2*(v + k) * c1 - x*c0, x*c1];
    end_for:
    return(normal(c1/x^(n-1)));
  elif n = 1 then
    return(normal(c1));
  elif n = 0 then
    return(normal(c0));
  elif n < 0 then
    for k from 0 downto n+1 do
      [c0, c1]:= [2*(v + k) * c0 - x*c1, x*c0];
    end_for:
    return(normal(c0*x^n));
  end_if;
  assert(FALSE); // should not arrive here
end_proc:

//==========================================================
besselJ::TeX :=
proc(b, ex, prio)
  local s;
begin
  s := generate::tex(op(ex, 2), output::Priority::Fconcat);
  if length(s) < 7 or s[1..7] <> "\\left(" then
    s := "\\left(".s."\\right)";
  end_if;
  _concat("J_{", generate::TeX(op(ex,1)), "}", s);
end_proc:

besselJ::conjugate:= proc(v, z)
local b;
begin
   b:= numeric::isnonzero(Im(besselJ(v, z)));
   if b = FALSE then 
      // y = besselJ(v, z) is real
      return(hold(besselJ)(v, z));
   elif b = TRUE then
      if numeric::isnonzero(Re(besselJ(v, z))) = FALSE then 
        // y = besselJ(v, z) is on the imaginary axis
        return(-hold(besselJ)(v, z));
      else
        // y = besselJ(v, z) is somewhere in the complex plane
        return(hold(conjugate)(hold(besselJ)(v, z)));
      end_if;
   end_if;

   if is(v in R_) = TRUE and
      is(z >= 0) = TRUE then // y is real
      return(hold(besselJ)(v, z))
   end_if:
   if is(v in Z_) = TRUE and
      is(z in R_) = TRUE then
      return(hold(besselJ)(v, z))
   end_if:

   // we cannot decide whether y is real or not
   return(hold(conjugate)(hold(besselJ)(v, z)));
end_proc:

// end of file 
