/*++
coth -- the hyperbolic cotangens

coth(x)

x - expression
++*/

coth:=
proc(x)
  option noDebug;
  local f, y;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::coth <> FAIL then return(x::dom::coth(args()))
  elif args(0) <> 1 then error("wrong no of args") 
  end_if;

  if x = 0 or x = PI*I then error("singularity") end_if;

  case type(x)
  of DOM_INT do
  of DOM_RAT do
     if x < 0 then return(-coth(-x)) end_if;
     break;
  of DOM_FLOAT do
     return(coth::float(x));
  of DOM_COMPLEX do
     if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT then
         return(coth::float(x))
     end_if;
     if op(x,1) = 0 then return(-cot(-x*I)*I) end_if;
     break;
  of DOM_SET do
  of "_union" do
     return(map(x, coth))
  of "_intersect" do
  of "_minus" do
     return(Dom::ImageSet(coth(`#z`), `#z`, x))
  end_case;

  if not testtype(x,Type::Arithmetical) then
      /* generic handling of sets */
      if testtype(x, Type::Set) then
        return(Dom::ImageSet(coth(`#x`), `#x`, x));
      end_if;
     error("argument must be of 'Type::Arithmetical'")
  end_if;

  case type(x)    
  of "_mult" do
     f:= op(x, nops(x));
     if testtype(f, Type::Real) then
        if f < 0 then return(-coth(-x)) end_if;
     elif type((f:=x/I/PI))=DOM_INT then
        error("singularity");
     elif type(2*f)=DOM_INT then
        return(0);
     end_if;
     f:= x/I/PI;
     if contains({DOM_RAT, DOM_INT}, domtype(f)) then
        if f < 0 then return(-coth(-x)); end_if;
        if f = 0 then error("singularity") end_if;
        if f < 1/2 then break end_if;
        if f < 1 then return(-coth((1-f)*PI*I)) end_if;
        if f = 1 then error("singularity") end_if;
        if f < 2 then return(coth((f-1)*PI*I)) end_if;
        return(coth((f-floor(f))*PI*I));
     end_if:
     f:= x/I;
     if type(f) = "arg" then  // x = coth(I*arg(z)) = I*Re(z)/Im(z)
       y:= Im(op(f));
       if iszero(y) then
          error("singularity")
       else
          return(-I*Re(op(f))/y):
       end_if;
     end_if:
     break;
   of "_plus" do
      // due to performance: move this code to coth::simplify
      // if sysorder(-x,x)=TRUE then return(-coth(-x)) else break end_if

      // handle coth(y + integer*PI*I) -> coth(y)
      // We could also do coth(y + PI/2*I) -> -tanh(y).
      // At the moment, we decide not to do it.
      if has(x, PI) then
         for y in x do
             f:= y/PI/I;
             if testtype(f, DOM_INT) and f<>0 then
                 return(coth(x-y))
             end_if;
         end_for:
     end_if:
     break;
  of "arccoth" do return(op(x));
  of "arcsinh" do return(sqrt(op(x)^2+1)/op(x));
  of "arccsch" do return(sqrt(op(x)^2+1));
  of "arccosh" do return(op(x)/sqrt(op(x)^2-1));
  of "arcsech" do return(1/sqrt(1-op(x)^2));
  of "arctanh" do return(1/op(x));

  end_case;

  procname(x)
end_proc:

coth := prog::remember(coth, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("coth")]):

coth(  I*PI/2):= 0:
coth(  I*PI/3):=-I*(3^(1/2))/3:
coth(  I*PI/4):=-I:
coth(  I*PI/5):=-I*1/5*5^(1/2)*(5 + 2*5^(1/2))^(1/2):
coth(2*I*PI/5):=-I*1/5*5^(1/2)*(5 - 2*5^(1/2))^(1/2):
coth(  I*PI/6):=-I*(3^(1/2)):
coth(  I*PI/8):=-I*(2^(1/2)+1):
coth(3*I*PI/8):=-I*(2^(1/2)-1):
coth(  I*PI/10):=-I*(5+2*5^(1/2))^(1/2):
coth(3*I*PI/10):=-I*(5-2*5^(1/2))^(1/2):
coth(  I*PI/12):=-I*(2+3^(1/2)):
coth(5*I*PI/12):=-I*(2-3^(1/2)):
coth(    I*PI/24):= -I*( 2 + 2^(1/2) + 3^(1/2) + 2^(1/2)*3^(1/2)):
coth(  5*I*PI/24):= -I*( 2 - 2^(1/2) - 3^(1/2) + 2^(1/2)*3^(1/2)):
coth(  7*I*PI/24):= -I*(-2 - 2^(1/2) + 3^(1/2) + 2^(1/2)*3^(1/2)):
coth( 11*I*PI/24):= -I*(-2 + 2^(1/2) - 3^(1/2) + 2^(1/2)*3^(1/2)):

coth( infinity):= 1:
coth(-infinity):=-1:

coth:= funcenv(coth):
coth::type := "coth":
coth::print := "coth":
coth::info := "coth -- the hyperbolic cotangent":

coth::inverse := "arccoth":

coth::realDiscont := {0}:
coth::complexDiscont :=
    loadproc(coth::complexDiscont, pathname("STDLIB","DISCONT"), "coth"):
coth::undefined := 
    loadproc(coth::undefined, pathname("STDLIB","DISCONT"), "coth"):

coth::diff :=
  proc(f)
    local op1;
  begin
    op1 := op(f, 1);
    diff(op1, args(2..args(0))) * (1-coth(op1)^2)
  end_proc:

coth::float := specfunc::coth:

coth::hull := DOM_INTERVAL::coth@hull:

coth::expand :=
loadproc(coth::expand, pathname("STDLIB", "EXPAND"), "coth"):

coth::conjugate :=
    loadproc(coth::conjugate, pathname("STDLIB","CONJ"), "coth"):

coth::rectform :=
    loadproc(coth::rectform, pathname("STDLIB","RECTFORM"), "coth"):

coth::series :=
  proc(f,x,n,dir,opt)
  begin
    Series::series(cosh(f),x,n,dir,opt)/Series::series(sinh(f),x,n,dir,opt)
  end_proc:

coth::simplify :=
    loadproc(coth::simplify, pathname("STDLIB","SIMPLIFY"), "coth"):

coth::Re := loadproc(coth::Re, pathname("STDLIB","RE"), "coth"):

coth::Im := loadproc(coth::Im, pathname("STDLIB","IM"), "coth"):

coth::Content := stdlib::genOutFunc("Ccoth", 1):

// end of file 
