ellipticF:=
proc(phi, m)
local s;
begin
  if args(0)<>2 then
    error("expecting 2 arguments");
  end_if;
  
  if iszero(phi) then
    return(0);
  elif iszero(m) then
    return(phi);
  elif iszero(m-1) then
    if is(Re(phi)>=PI/2)=TRUE then
      return(infinity);
    elif is(Re(phi)<=-PI/2)=TRUE then
      return(-infinity);
    elif is(Re(phi)>-PI/2 and Re(phi)<PI/2)=TRUE then
      return(ln(tan(PI/4 + phi/2)));
    else
      return(procname(phi, m));
    end_if;
  end_if;

  s:=phi*2/PI;
  if testtype(s, Type::Integer)=TRUE then
    return(s*ellipticK(m));
  end_if;

  if type(phi)=DOM_FLOAT or (type(phi)=DOM_COMPLEX and type(op(phi,1))=DOM_FLOAT and type(op(phi,2))=DOM_FLOAT) or
     type(m)=DOM_FLOAT or (type(m)=DOM_COMPLEX and type(op(m,1))=DOM_FLOAT and type(op(m,2))=DOM_FLOAT) then
    return(ellipticF::float(phi, m));
  end_if;

  if not testtype(phi,Type::Arithmetical) then
    if testtype(phi, Type::Set) then
      if testtype(m, Type::Set) and not testtype(m,Type::Arithmetical) then
        return(Dom::ImageSet(ellipticF(#phi, #m), [#phi, #m], [phi, m]));
      else
        return(Dom::ImageSet(ellipticF(#phi, m), [#phi], [phi]));
      end_if;
    end_if;

    error("first argument must be of 'Type::Arithmetical'")
  end_if;
  if not testtype(m,Type::Arithmetical) then
    if testtype(m, Type::Set) then
      return(Dom::ImageSet(ellipticF(phi, #m), [#m], [m]));
    end_if;

     error("second argument must be of 'Type::Arithmetical'")
  end_if;

  return(procname(phi, m));
end_proc:

ellipticF := prog::remember(ellipticF, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("ellipticF")]):


ellipticF:=funcenv(ellipticF):

ellipticF::type:="ellipticF":


ellipticF::Content := stdlib::genOutFunc("CellipticF", 2):

ellipticF::discont:=
proc(f)
  local phi,m;
begin
  phi:=op(f,1);
  m:=op(f,2);
  return(discont(1/sqrt(1-m*sin(phi)^2), args(2..args(0))))
end_proc:


ellipticF::diff:=proc(f)
local phi, m, tmp, result;
begin
  phi:=op(f, 1);
  m:=op(f, 2);

  result:=diff(phi, args(2..args(0))) / sqrt(1-m*sin(phi)^2);

  tmp:=diff(m, args(2..args(0)));
  // avoid division by zero if m=0 or m=1
  if not iszero(tmp) then
    result:=result + tmp * (ellipticE(phi,m)/(2*m*(1-m)) - ellipticF(phi,m)/(2*m) - sin(2*phi)/(4*(1-m)*sqrt(1-m*sin(phi)^2)));
  end_if;

  return(result);
end_proc:


/*    Carlson's elliptic integral of the 1st kind
 * This is the function RF as described in "B.C.Carlson, Numerical Computation of Real or Complex
 * Elliptic Integreals", but it has been slightly modified so that the arguments are the square
 * roots of the arguments as described in the paper, i.e. it is defined by
 *
 *   RF(x,y,z):=int(1/sqrt(t+x^2)/sqrt(t+y^2)/sqrt(t+z^2), t=0..infinity)
 *
 * Why do this? If we want to evaluate ellipticF for a phi with Re(cos(phi))=0, we would lose
 * the sign of Im(cos(phi)). As a result, we have cos(phi)^2<0 but cannot decide on which side
 * of the branch cut we are. */
ellipticF::RF:=proc(x0, y0, z0)
local x,y,z,xx,yy,zz,A,A0,Q,P,lambda,E2,E3;
begin
  if Re(x0)<0 then x0:=-x0; end;
  if Re(y0)<0 then y0:=-y0; end;
  if Re(z0)<0 then z0:=-z0; end;

  if iszero(x0) and iszero(y0) then
    error("singularity");
  end;

  A0:=(x0^2+y0^2+z0^2)/3;

  x:=x0;
  y:=y0;
  z:=z0;
  A:=A0;

  Q:=float(10^(DIGITS/6)) * max(specfunc::abs(A-x^2), specfunc::abs(A-y^2), specfunc::abs(A-z^2));

  P:=1;
  while P*specfunc::abs(A)<Q do
    P:=P*4;

    lambda:=x*y + x*z + y*z;

    A:=(A+lambda)/4;
    x:=sqrt(x^2+lambda)/2;
    y:=sqrt(y^2+lambda)/2;
    z:=sqrt(z^2+lambda)/2;
  end;

  xx:=(A0-x0^2)/P/A;
  yy:=(A0-y0^2)/P/A;
  zz:=(A0-z0^2)/P/A;

  E2:=xx*yy-zz^2;
  E3:=xx*yy*zz;

  return((E2*(-1/10 + E2/24 - 3/44*E3) + E3/14 + 1) / sqrt(A));
end_proc:


ellipticF::float:=proc(phi, m)
local cs, sn, k, F;
begin
  phi:=float(phi);
  m:=float(m);

  if type(phi)<>DOM_FLOAT and type(phi)<>DOM_COMPLEX then
    return(hold(ellipticF)(phi, m));
  end;
   
  if type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX then
    return(hold(ellipticF)(phi, m));
  end;

  cs:=cos(phi);
  sn:=sin(phi);

  F:=sn * ellipticF::RF(cs, sqrt(1-m*sn^2), 1);
  
  // This helps the decision at the transition points for k:
  // If Re(cs)>=0, then k has to be even, otherwise k has to be odd.
  if Re(cs)>=0 then
    k:=2*round(Re(phi)/PI/2);
    if k=0 then
      return(F);
    else
      return(2*k*ellipticK(m) + F);
    end;
  else
    k:=2*floor(Re(phi)/PI/2) + 1;
    return(2*k*ellipticK(m) - F);
  end;
end_proc:


