/*
  ellipticPi(n,z,m):=int(1/(1-n*sin(t)^2)/sqrt(1-m*sin(t)^2), t=0..z)
  ellipticPi(n,m):=ellipticPi(n,PI/2,m)
*/
ellipticPi:=
proc()
  local i, mapSets;
begin
  case args(0)
  of 2 do
    if iszero(args(1)) then
      return(ellipticK(args(2)));
    end_if;
    if iszero(args(2)) then
      return(PI/2/sqrt(1-args(1)));
    end_if;
    if iszero(args(1)-1) or iszero(args(2)-1) then
      error("singularity");
    end_if;
    break;
    
  of 3 do
    if iszero(args(1)) then
      return(ellipticF(args(2..3)));
    end_if;
    if iszero(args(1)-1) then
      return(ellipticF(args(2..3)) + (sqrt(1-args(3)*sin(args(2))^2)*tan(args(2))-ellipticE(args(2..3)))/(1-args(3)));
    end_if;
    if iszero(args(2)) then
      return(0);
    end_if;
    if iszero(args(2)-PI/2) then
      return(ellipticPi(args(1),args(3)));
    end_if;
    if iszero(args(1)-args(3)) then
      return((ellipticE(args(2),args(3))-args(3)*sin(2*args(2))/2/sqrt(1-args(3)*sin(args(2))^2))/(1-args(3)));
    end_if;
    break;
    
  otherwise
    error ("expecting 2 or 3 arguments");
  end_case;

  for i in args() do
    if type(i)=DOM_FLOAT or (type(i)=DOM_COMPLEX and (type(op(i,1))=DOM_FLOAT or type(op(i,2))=DOM_FLOAT)) then
      return(ellipticPi::float(args()));
    end_if;
  end_for;

  mapSets := proc()
    local i, argList, ids, sets, j;
  begin
    argList := [];
    ids := [];
    sets := [];
    for i from 1 to args(0) do
      if testtype(args(i), Type::Set) and not testtype(args(i),Type::Arithmetical) then
        j := genident();
        argList := argList.[j];
        ids := ids.[j];
        sets := sets.[args(i)];
      else
        argList := argList.[args(i)];
      end_if;
    end_for;
    return(Dom::ImageSet(ellipticPi(op(argList)), ids, sets));
  end_proc;

  if not testtype(args(1),Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(args(1), Type::Set) then
      return(mapSets(args()));
    end_if;
    error("first argument must be of 'Type::Arithmetical'")
  end_if;

  if not testtype(args(2),Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(args(2), Type::Set) then
      return(mapSets(args()));
    end_if;
    error("second argument must be of 'Type::Arithmetical'")
  end_if;

  if args(0)=3 and not testtype(args(3),Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(args(3), Type::Set) then
      return(mapSets(args()));
    end_if;
    error("third argument must be of 'Type::Arithmetical'")
  end_if;

  return(procname(args(1..args(0))));
end_proc:

ellipticPi := prog::remember(ellipticPi, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("ellipticPi")]):

ellipticPi:=funcenv(ellipticPi):

ellipticPi::Content:=stdlib::genOutFunc("CellipticPi", 2, 3):

ellipticPi::diff:=proc(f)
local n, z, m, dn, dz, dm, result;
begin
  case nops(f)
    of 2 do
      n:=op(f, 1);
      m:=op(f, 2);
      dn:=diff(n, args(2..args(0)));
      dm:=diff(m, args(2..args(0)));
      return(
        dn * (ellipticE(m)/2/(m-n)/(n-1) + ellipticK(m)/2/n/(n-1) + ellipticPi(n,m)/2*(n^2-m)/n/(m-n)/(n-1)) +
        dm * (ellipticE(m)/2/(m-1)/(n-m) + ellipticPi(n,m)/2/(n-m)));

    of 3 do
      n:=op(f, 1);
      z:=op(f, 2);
      m:=op(f, 3);
      dn:=diff(n, args(2..args(0)));
      dz:=diff(z, args(2..args(0)));
      dm:=diff(m, args(2..args(0)));

      result:=0;
      if not iszero(dn) then
        result:=dn * (ellipticE(z,m)/2/(m-n)/(n-1) + ellipticF(z,m)/2/n/(n-1) + ellipticPi(n,z,m)*(n^2-m)/2/(m-n)/n/(n-1) - sqrt(1-m*sin(z)^2)*sin(2*z)/(1-n*sin(z)^2)*n/4/(m-n)/(n-1));
      end_if;
      if not iszero(dz) then
        result:=result + dz / sqrt(1-m*sin(z)^2) / (1-n*sin(z)^2);
      end_if;
      if not iszero(dm) then
        result:=result + dm * (ellipticE(z,m)/2/(m-1)/(n-m) + ellipticPi(n,z,m)/2/(n-m) - m*sin(2*z)/4/(n-m)/(m-1)/sqrt(1-m*sin(z)^2));
      end_if;
      return(result);
        
    otherwise
      return(FAIL);
  end_case;
end_proc:


/* This is Carlson's elliptic integral of the third kind, defined as
     RJ(x,y,z,p):=int(1/sqrt(t-x)/sqrt(t-y)/sqrt(t-z)/(t-p), t=0..infinity)  */
ellipticPi::RJ:=proc(x,y,z,p)
local xi,yi,zi,pi,Ai,di,ei,A0,Q,delta,lambda,pow4,rj,rc,
      sqrtx,sqrty,sqrtz,sqrtp,X,Y,Z,P,E2,E3,E4,E5;
begin
  x:=float(x);
  y:=float(y);
  z:=float(z);
  p:=float(p);

  if iszero(x) and iszero(y) then
    error("singularity");
  end;

  xi:=x; yi:=y; zi:=z; pi:=p;

  A0:=(x+y+z+2*p)/5;
  delta:=(p-x)*(p-y)*(p-z);
  Q:=1.5 * float(10^(DIGITS/6)) * max(specfunc::abs(A0-x), specfunc::abs(A0-y), specfunc::abs(A0-z), specfunc::abs(A0-p));
    
  pow4:=1.0;
  Ai:=A0;
  rj:=0.0;
  
  while Q>specfunc::abs(Ai)*pow4 do
    sqrtx:=sqrt(xi);
    sqrty:=sqrt(yi);
    sqrtz:=sqrt(zi);
    sqrtp:=sqrt(pi);
    
    lambda:=sqrtx*sqrty + sqrtx*sqrtz + sqrty*sqrtz;

    di:=(sqrtp+sqrtx) * (sqrtp+sqrty) * (sqrtp+sqrtz);
    ei:=delta/di^2/pow4^3;

    Ai:=(Ai+lambda)/4;
    xi:=(xi+lambda)/4;
    yi:=(yi+lambda)/4;
    zi:=(zi+lambda)/4;
    pi:=(pi+lambda)/4;

    // since arctan(x)=x*RC(1, 1+x^2) we can simplify things to
    // RC(1, 1+ei)=arctan(sqrt(ei))/sqrt(ei)
    if specfunc::abs(ei)<float(0.5 * 10^(-DIGITS/3)) then
      rc:=1 + ei*(ei/5 - 1/3);
    else
      rc:=arctan(sqrt(ei)) / sqrt(ei);
    end_if;

    rj:=rj + 6/pow4/di*rc;

    pow4:=4*pow4;
  end_while;

  X:=(A0-x)/pow4/Ai;
  Y:=(A0-y)/pow4/Ai;
  Z:=(A0-z)/pow4/Ai;
  P:=-(X+Y+Z)/2;

  E2:=X*Y + X*Z + Y*Z - 3*P^2;
  E3:=X*Y*Z + 2*E2*P + 4*P^3;
  E4:=(2*X*Y*Z + E2*P + 3*P^3)*P;
  E5:=X*Y*Z*P^2;

  return(rj + Ai^(-3/2)/pow4 * (1 - 3/14*E2 + 1/6*E3 + 9/88*E2^2 - 3/22*E4 - 9/52*E2*E3 + 3/26*E5));
end:


ellipticPi::float:=proc()
local phi,n,m,k,sn,cs,pi;
begin
  case args(0)
  of 2 do
    n:=float(args(1));
    m:=float(args(2));
    break;
  of 3 do
    n:=float(args(1));
    phi:=float(args(2));
    m:=float(args(3));
    break;
  otherwise
    error("expecting 2 or 3 arguments");
  end_case;
  
  if (type(n)<>DOM_FLOAT and type(n)<>DOM_COMPLEX) or (type(m)<>DOM_FLOAT and type(m)<>DOM_COMPLEX) then
    return(hold(ellipticPi)(args()));
  end_if;

  if args(0)=3 and type(phi)<>DOM_FLOAT and type(phi)<>DOM_COMPLEX then
    return(hold(ellipticPi)(args()));
  end_if;

  if args(0)=2 then
    return(ellipticK(m) + n/3*ellipticPi::RJ(0, 1-m, 1, 1-n));
  else
    sn:=sin(phi);
    cs:=cos(phi);
    k:=round(Re(phi)/PI);
  
    pi:=sn*ellipticF::RF(cs, sqrt(1-m*sn^2), 1) + n/3*sn^3*ellipticPi::RJ(cs^2, 1-m*sn^2, 1, 1-n*sn^2);
    
    if k=0 then
      return(pi);
    elif modp(k,2)=0 then
      return(2*k*ellipticPi::float(n,m) + pi);
    else
      return(2*k*ellipticPi::float(n,m) - pi);
    end_if;
  end_if;
end_proc:


