/*++
The complementary error function

erfc(x) = 1 - erf(x)
erfc(x, n) = int(erfc(t, n-1), t=x..infinity)

x - an expression
n - an integer >= -1

The 2 argument version represents the 'repeated integrals'

   erfc(x, -1) = 2/sqrt(Pi) * exp(-x^2)
   erfc(x, 0 ) = erfc(x) 
   erfc(x, n ) = int(erfc(t, n-1), t=x..infinity)

Author of the 2 argument version: W. Oevel
++*/

erfc :=
proc(x, n) 
  local f, fn;
begin
  if args(0) = 0 then error("no arguments given")
  elif args(0) = 1 then
     n:= 0:
  elif args(0) > 2 then error("no more than 2 arguments expected");
  end_if;

  if x::dom::erfc <> FAIL then 
     return(x::dom::erfc(args())) 
  end_if:

  if iszero(n + 1) then
     if domtype(n) = DOM_FLOAT or
        contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
       return(float(2/sqrt(PI)*exp(-x^2)));
     else
       return(2/sqrt(PI)*exp(-x^2));
     end_if:
  end_if:

  fn:= float(n):
  if (domtype(n) = DOM_INT and n < -1) or
     (domtype(fn) = DOM_FLOAT and iszero(frac(n)) and fn < -1) or
     (domtype(fn) = DOM_FLOAT and not iszero(frac(n))) or
     (domtype(fn) = DOM_COMPLEX) then
        error("2nd argument: expecting an integer >= -1. Got: ".expr2text(n)):
  end_if:

  // some special values:
  if iszero(x) then
     if domtype(n) = DOM_FLOAT or 
        contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
       return(1/2^n/gamma::float(n/2 + 1));
     else
       return(1/2^n/gamma(n/2 + 1));
     end_if:
  end_if;

  if domtype(n) = DOM_FLOAT then
     return(erfc::float(x, n));
  end_if:

  case type(x)
    of DOM_FLOAT do
        return(erfc::float(x, n));
    of DOM_COMPLEX do
        if type(op(x,1)) = DOM_FLOAT or type(op(x,2)) = DOM_FLOAT then
           return(erfc::float(x, n));
        end_if;
        break;
    of DOM_INT do
    of DOM_RAT do return((if iszero(n) and x<0 then 
                              return(2 - procname(-x))
                          else 
                            if iszero(n) then
                              return(procname(x))
                            else
                              return(procname(x, n))
                            end_if:
                          end_if))

    of DOM_SET do
    of "_union" do
      return(map(x, y -> erfc(y, n)))
  end_case;
  
  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, erfc));
      else
        return(Dom::ImageSet(erfc(#x), #x, x));
      end_if;
    end_if;

    error("the argument must be of 'Type::Arithmetical'")
  end_if;

  if x = infinity then 
     return(0) 
  end_if;
  if x = RD_INF then 
     return(float(0)) 
  end_if;

  if x = -infinity then 
     case n 
     of -1 do return(0);
     of  0 do return(2);
     otherwise
        return(infinity) 
     end_case;
  end_if;
  if x = RD_NINF then 
     case n 
     of -1 do return(float(0));
     of  0 do return(float(2));
     otherwise
        return(RD_INF) 
     end_case;
  end_if;
  if x = RD_NAN then
     return(RD_NAN):
  end_if:
  
  case type(x)
  of "_mult" do
       f:= op(x, nops(x));
       if testtype(f, Type::Real) then
          if f < 0 then 
             if iszero(n) then
                return(2-erfc(-x)) 
             else
                // Do we really want the following? This may lead
                // to very messy results if x contains symbols!
                // return(orthpoly::hermite(n, I*x)*I^n/2^(n-1)/n!- (-1)^n*erfc(-x, n))
             end_if:
          end_if;
          if domtype(f) = DOM_FLOAT then break end_if;
       end_if;
       break;
  of "inverf" do
     if n = 0 then 
       return(1-op(x, 1))
     end_if;
     break;
  of "inverfc" do
     if n = 0 then 
       return(op(x, 1))
     end_if;
     break;
  end_case;
  if iszero(n) then
     procname(x) 
  else
     procname(x, n) 
  end_if:
end_proc:

erfc := prog::remember(erfc, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("erfc")]):

erfc(0) := 1:
erfc(infinity) := 0:
erfc(-infinity) := 2:
erfc(RD_INF) := float(0):
erfc(RD_NINF) := float(2):
erfc(infinity*I) := 1 - I*infinity:
erfc(-infinity*I) := 1 + I*infinity:
erfc(RD_INF*I) := 1 + I*RD_NINF:
erfc(RD_NINF*I) := 1 + I*RD_INF:

erfc:= funcenv(erfc, op(specfunc::erfc, 2)):
erfc::print:= "erfc":
erfc::info:="erfc -- the complementary error function":
erfc::type:= "erfc":

//=======================================================
erfc::rectform:=
proc(x, n)
  local y, a;
begin
  y:= rectform::new(x);
  if args(0) = 1 then 
    n:= null():
  end_if:
  // special case:

  if iszero(extop(y, 1)) and iszero(extop(y, 3)) then 
    // the argument of erfc is purely imaginary, use
    // erfc(x) = 1 - erf(x)
    if n = null() or n = 0 then
       return(new(rectform, 1, I*hold(erf)(x, n), 0))
    end_if;
  end_if:
  if extop(y, 3) <> 0 or not iszero(extop(y, 2)) then
    new(rectform, 0, 0, hold(erfc)(x, n))
  else
    a := extop(y, 1);
    if domtype(a) = DOM_FLOAT then
      rectform::new(erfc(a, n))
    else
      new(rectform, erfc(a, n), 0, 0)
    end_if
  end_if
end_proc:

//=======================================================
erfc::Re:= proc(x, n)
begin
   if is(x in R_) = TRUE then
      return(hold(erfc)(args()))
   elif is(x/I in R_) = TRUE and
        args(0) = 1 then
      return(1)
   end_if:
   hold(Re)(erfc(args()));
end_proc:

//=======================================================
erfc::Im:= proc(x, n)
begin
   if is(x in R_) = TRUE then
      return(0)
   elif is(x/I in R_) = TRUE and
        args(0) = 1 then
      return(hold(erfc)(args())/I + I)
   end_if:
   hold(Im)(erfc(args()));
end_proc:

// ==================================================
// asympt = utility for the float attribute. Computes 
// erfc(x, n) via the asymptotic series. Returns FAIL 
// if the precision goal cannot be reached.
// ==================================================

erfc::asympt:= proc(x, n)
local tol, absx2, tmp1, tmp2, k, k1, k2,
      x2, t, s, n2k;
begin
  x:= float(x);
  if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
     return(FAIL);
  end_if;

  if (domtype(x) = DOM_FLOAT and x < 0) or
      specfunc::abs(arg(x)) > 3/4*float(PI) then
      // the asymptotics is available only for  |arg(x)|< 3*PI/4
     return(FAIL);
  end_if:

  tol:= 10.0^(-DIGITS):
  absx2:= specfunc::abs(x)^2;

  // The terms (n + 2*k)!/n!/k!/(2*x)^(2*k))
  // of the asymptotic series of erfc(x, n)
  // are increasing for k = 0 .. k1, then 
  // decreasing from k = k1 .. k2, then again
  // increasing for k = k2 .. infinity:

  tmp1:= (absx2 - 3/2 - n)/2:
  tmp2:= absx2*(absx2 - 2*n + 1) + 1/4:
  if tmp2 < 0 then // All terms in the series are increasing. 
     return(FAIL); // The asymptotics cannot be used
  end_if;

  k1:= ceil(tmp1 - sqrt(tmp2)/2):
  k2:= ceil(tmp1 + sqrt(tmp2)/2):

  if k2 <= 0 then
    return(FAIL);
  end_if;

  if float(n + 2*k2)!/(4*absx2)^k2 < tol*float(n)!*float(k2)! then
     x2:= 4*x^2:
     t:= float(1):
     s:= t:
     n2k:= n:
     for k from 1 to k2 do
         n2k:= n2k + 2: // = n + 2*k
         t:= -(n2k-1)*n2k/k/x2 * t;
         s:= s + t:
         if specfunc::abs(t) <= tol*specfunc::abs(s) then
           return(1/sqrt(float(PI))/2^n*exp(-x^2)/x^(n+1)*s);
         end_if;
     end_for:
  end_if:
  return(FAIL);
end_proc:

// ==================================================
// asympt = utility for the float attribute. Computes 
// erfc(x, n) via the Taylor series around x = 0. 
// Returns FAIL if serious cancellation occurs.
// ==================================================
erfc::taylor:= proc(x, n)
local tol, t, s, k, kk;
begin
  x:= float(x);
  if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) then
     return(FAIL);
  end_if;
  tol:= 10.0^(-DIGITS);
  t:= 1/2^n:
  s:= t/gamma(1.0 + n/2):
  for k from 1 to RD_INF do
    t:= -t*2*x/k;
    kk:= 1.0 + (n-k)/2;
    if iszero(frac(kk)) and kk <= 0 then
       next;
    else
       s:= s + t/gamma(kk);
       if specfunc::abs(t/gamma(kk)) <= tol*specfunc::abs(s) then
          return(s);
       end_if;
    end_if:
  end_for: 
  return(FAIL);
end_proc:

// ===========================================
// float attribute
// ===========================================
erfc::float:= proc(x, n)
local fx, fn, r, r1m2, i, r_0, r_1, tmp, 
      abort, r1, r2, extraDIGITS;
save DIGITS;
begin
  if args(0) = 1 or iszero(n) then
     return(specfunc::erfc(x));
  end_if:

  fx:= float(x); 
  fn:= float(n):
  if not contains({DOM_COMPLEX, DOM_FLOAT}, domtype(fx)) then
     return(hold(erfc)(fx, fn));
  end_if:

  if (domtype(n) = DOM_INT and n < -1) or
     (domtype(fn) = DOM_FLOAT and iszero(frac(n)) and fn < -1) or
     (domtype(fn) = DOM_FLOAT and not iszero(frac(n))) or
     (domtype(fn) = DOM_COMPLEX) then
     // n is a number, but not an integer
        error("2nd argument: expecting an integer >= -1. Got: ".expr2text(n)):
  elif domtype(fn) <> DOM_FLOAT then
     // n is symbolic
     return(hold(erfc)(fx, fn)):
  end_if:

  // ======================================================
  // Try the Taylor series around the origin
  // ======================================================
  if specfunc::abs(x) <= 1.1 then 
     if (r:= erfc::taylor(x, n)) <> FAIL then
       return(r);
     end_if:
  end_if:

  // ======================================================
  // Try the asymptotic formula
  // ======================================================
  if (r:= erfc::asympt(x, n)) <> FAIL then
     return(r);
  end_if:

  // ======================================================
  // For not too large n, use the recurrence
  //   erfc(x,n) = -x/n*erfc(x,n-1) + 1/2/n*erfc(x,n-2)
  // (see the expand attribute) to trace erfc(x, n) back to 
  // erfc(x, 0) and erfc(x, -1)
  // ======================================================
  if domtype(n) = DOM_INT or
     (domtype(n) = DOM_FLOAT and  iszero(frac(n))) then
     abort:= FALSE;
     if n <= 100 and specfunc::abs(x) <= specfunc::abs(n) then
        r_1:= float(2/sqrt(PI)*exp(-fx^2)): 
        r_0:= erfc(fx):
        tmp:= r_1/2 - fx*r_0;
        if not iszero(tmp) then
           extraDIGITS:= max(10, ceil(-ln( abs(tmp)/abs(fx*r_0) )/ln(10.0)));
           if extraDIGITS > 1000 then
              abort:= TRUE;
           end_if;
           DIGITS:= DIGITS + extraDIGITS;
        else
           // go ahead
        end_if:
        if not abort then
          r_1:= float(2/sqrt(PI)*exp(-fx^2)): 
          r_0:= erfc(fx):
          for i from 1 to n do
            [r_0, r_1]:= [(-fx*r_0 + r_1/2)/i, r_0];
          end_for: 
          return(r_0);
        end_if;
     end_if:
  end_if:

  // ======================================================
  // Use erfc(x,n) = 
  //   exp(-x^2)*M((n+1)/2, 1/2, x^2)/2^n/gamma(1+n/2)
  // - exp(-x^2)*M(n/2 + 1, 3/2, x^2)/2^(n-1)/gamma((1+n)/2)
  // with M(a, b, z) = hypergeom([a], [b], z).
  // Problem: There can be severe cancellation between these
  // two term requiring a massive boost of DIGITS
  // ======================================================
  extraDIGITS:= 0:
  while TRUE do
    if extraDIGITS > 1000 then
       warning("boosting DIGITS by ".expr2text(extraDIGITS));
    end_if:
    r1:=  float(1/2^n  /gamma(n/2 + 1)*hypergeom([(n+1)/2],[1/2],x^2)):
    r2:=  float(2/2^n*x/gamma((n+1)/2)*hypergeom([ n/2+1 ],[3/2],x^2)):
    if not (contains({DOM_FLOAT, DOM_COMPLEX}, domtype(r1)) and
            contains({DOM_FLOAT, DOM_COMPLEX}, domtype(r2))) then
       // Something could not be floated. Return symbolically:
       if iszero(n) then
         return(hold(erfc)(float(x)))
       else
         return(hold(erfc)(float(x), float(n)))
       end_if:
    else
       // Check whether the precision suffices. 
       r:= max(abs(r1), abs(r2)); 
       r1m2:= r1 - r2:
       if abs(r1m2)*10^extraDIGITS > 0.1*r then
          return(float(exp(-x^2))*r1m2)
       else
          // Insufficient precision. Boost DIGITS!
          if iszero(r1m2) then 
             extraDIGITS:= extraDIGITS + 100:
          else 
             extraDIGITS:= max(extraDIGITS + 10,
                               ceil(- 0.4342945*ln(abs(r1m2)/r)));
          end_if:
          DIGITS:= DIGITS + extraDIGITS;
       end_if:
    end_if:
  end_while
end_proc:

//=======================================================
// expand: use the recurrence 
//    erfc(x,n) = -x/n*erfc(x,n-1) + 1/2/n*erfc(x,n-2) 
// to express erfc(x,n) in terms of erfc(x,1) and erfc(x,-1):
//=======================================================
erfc::expand:= proc(f)
local x, n, r_0, r_1, i;
begin
  if nops(f) = 1 then 
     [x, n]:= [op(f), 0]:
  else
     [x, n]:= [op(f)]:
  end_if:
  if domtype(n) <> DOM_INT then
     return(erfc(expand(x), expand(n)));
  end_if:
  // Do a recursive expansion using the 2 step recurrence.
  // Do not use option remember to avoid MAXDEPTH problems.
  // Use a straightforward recurrence instead:
  if iszero(n+1) then 
    return(2/sqrt(PI)*exp(-x^2))
  elif iszero(n) then
    return(erfc(x))
  else
    r_1:= 2/sqrt(PI)*exp(-x^2): 
    r_0:= erfc(x):
    for i from 1 to n do
      [r_0, r_1]:= [expand(-x/i*r_0 + r_1/2/i), r_0];
    end_for: 
    return(r_0);
  end_if:
end_proc:

erfc::conjugate:= 
    loadproc(erfc::conjugate, pathname("STDLIB", "CONJ"), "erfc"):

erfc::diff:=
  proc()
    local x, n;
  begin 
    if nops(args(1)) = 1 then 
       x := op(args(1),1);
       return(-2/PI^(1/2) * exp(-x^2) * diff(x, args(2..args(0))))
    else
       x := op(args(1),1);
       n := op(args(1),2);
       return(-erfc(x, n-1)* diff(x, args(2..args(0)))):
    end_if:
  end_proc:

erfc::series:= loadproc(erfc::series, pathname("SERIES"), "erfc"):

// end of file 
