/*+++++++++++++++++++++++++++
Euler numbers and polynomials

Calls: 
     euler(n)
     euler(n, x)

Parameters:
     n - a nonnegative integer
     x - an expression
++++++++++++++++++++++++++++*/

euler :=
proc(n, x)
  local t, k, s, k0, y, x2;
begin
  case args(0)
    of 1 do
       if domtype(n) = DOM_FLOAT and 
          iszero(frac(n)) then
          n:= round(n);
       end_if:
       if domtype(n) = DOM_INT and n >= 0 then
          if n mod 2 = 1 then
             return(0)
          elif n <= Pref::autoExpansionLimit() then
             return(specfunc::eulerNumber(n));
          else
             break;
          end_if:
       end_if;
       if testtype(float(n), Type::Numeric) then
          error("the first argument must be symbolic or a nonnegative integer")
       end_if;
       if not testtype(n,Type::Arithmetical) then
          /* generic handling of sets */
          if testtype(n, Type::Set) then
            if type(n)=Dom::ImageSet then
              return(map(n, euler));
            else
              return(Dom::ImageSet(euler(#n), #n, n intersect Z_ intersect Dom::Interval([0], infinity)));
            end_if;
          end_if;

          error("the first argument must be of type 'Type::Arithmetical'")
       end_if;
       return(procname(n));
    of 2 do
        if not testtype(n,Type::Arithmetical) then
          if testtype(n, Type::Set) then
            if testtype(x,Type::Arithmetical) then
              return(Dom::ImageSet(euler(#n, x), #n, n intersect Z_ intersect Dom::Interval([0], infinity)));
            elif testtype(x,Type::Set) then
              return(Dom::ImageSet(euler(#n, #x), [#n, #x], [n intersect Z_ intersect Dom::Interval([0], infinity), x]));
            end_if;
          end_if;
          error("the first argument must be of type 'Type::Arithmetical'")
        elif not testtype(x,Type::Arithmetical) then
          if testtype(x,Type::Set) then
            return(Dom::ImageSet(euler(n, #x), #x, x));
          end_if;
          error("the second argument must be of type 'Type::Arithmetical'")
        end_if;

        // most probable case first: n is a nonnegative integer
        if domtype(n) = DOM_INT and n>=0 then 
           // special cases, no need to evaluate the polynomial:
           if n = 0 then return(1)
           elif n = 1 then return(x-1/2)
           elif n = 2 then return(x^2-x)
           elif n = 3 then return(x^3-3*x^2/2+1/4)
           elif n = 4 then return(x^4-2*x^3+x)
           elif x = 0 then return(-2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1))
           elif x = float(0) then return(float(-2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1)))
           elif x = 1 then return( 2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1))
           elif x = float(1) then return(float( 2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1)))
           elif x = 1/2 then return(1/2^n*specfunc::eulerNumber(n)) 
           elif x = float(1/2) then return(float(1/2^n*specfunc::eulerNumber(n))) 
           elif x = 1/3 and domtype((n+1)/2)=DOM_INT then
                n:= n+1;;
                return(-(1-3^(1-n))*(2^n - 1)/n*bernoulli(n))
           elif x = float(1/3) and domtype((n+1)/2)=DOM_INT then
                n:= n+1;;
                return(float(-(1-3^(1-n))*(2^n - 1)/n*bernoulli(n)))
           elif x = 2/3 and domtype((n+1)/2)=DOM_INT then
                n:= n+1;;
                return((1-3^(1-n))*(2^n - 1)/n*bernoulli(n))
           elif x = float(2/3) and domtype((n+1)/2)=DOM_INT then
                n:= n+1;;
                return(float((1-3^(1-n))*(2^n - 1)/n*bernoulli(n)))
           elif x = 3/2 then
                return(2^(1-n) - specfunc::eulerNumber(n)/2^n)
           elif x = float(3/2) then
                return(float(2^(1-n) - specfunc::eulerNumber(n)/2^n))
           end_if;

           //====================================================
           // euler::float only works for real x between 0 and 1.
           // There, however, it is much faster than the evaluation
           // of the polynomial further down below.
           //====================================================
           if domtype(x) = DOM_FLOAT and
              x > 0 and
              x < 1 then
              return(euler::float(n, x));
           end_if:

           // evaluate the polynomial, use
           // E(n,x) = 2/(n+1)*sum(binomial(n+1,k)*x^k*B(n+1-k)*(1-2^(n+1-k)), k=0..n)
           if n mod 2 = 1
           then  y:= 1:
                 t:= 2/(n+1);
                 s:= t*(1 - 2^(n+1))*bernoulli(n+1);
                k0:= 2;
           else  y:= x:
                 t:= 2;
                 s:= t*(1-2^n)*bernoulli(n)*y:
                k0:= 3;
           end_if;
           //compute  2/(n+1)*sum(binomial(n+1,k)*x^k*B(n+1-k)*(1-2^(n+1-k)), k=0..n-1)
           x2:= x^2:
           for k from k0 to n-1 step 2 do
               y:= y*x2;
               t:= t*(n-k+3)*(n-k+2)/k/(k-1);
               s:= s + t*(1 - 2^(n+1-k))*bernoulli(n+1 - k)*y;
           end_for:
           // add the last term:
           return(s+y*x);
        end_if:

        //============================================================
        // Various simplifications for symbolic values of n.
        // The call euler(n) checks, if n is valid:
        //============================================================
        if   x = 0 then return(-2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1))
        elif x = float(0) then return(float(-2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1)))
        elif x = 1 then return( 2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1))
        elif x = float(1) then return(float( 2/(n+1)*(2^(n+1) - 1)*bernoulli(n+1)))
        elif x = 1/2 then return(euler(n)/2^n) 
        elif x = float(1/2) then return(float(euler(n)/2^n)) 
        elif x = 1/3 and domtype((n+1)/2)=DOM_INT then
                  n:= n + 1;
                  return(-(1-3^(1-n))*(2^n - 1)/n*bernoulli(n))
        elif x = float(1/3) and domtype((n+1)/2)=DOM_INT then
                  n:= n + 1;
                  return(float(-(1-3^(1-n))*(2^n - 1)/n*bernoulli(n)))
        elif x = 2/3 and domtype((n+1)/2)=DOM_INT then
                  n:= n+1;;
                  return((1-3^(1-n))*(2^n - 1)/n*bernoulli(n))
        elif x = float(2/3) and domtype((n+1)/2)=DOM_INT then
                  n:= n+1;;
                  return(float((1-3^(1-n))*(2^n - 1)/n*bernoulli(n)))
        elif testtype(x, Type::Real) and x > 1/2 and x < 1 then
             return((-1)^n*euler(n,1-x))
        elif testtype(x, Type::Real) and x < 0 then
             return(2*x^n + (-1)^(n+1)*euler(n,-x))
        // errors: 
        elif testtype(float(n), Type::Numeric) then
             error("the first argument must be symbolic or a nonnegative integer")
        // symbolic return
        else return(procname(n, x))
        end_if;
        break;
    otherwise
        error("1 or 2 arguments expected.")
  end_case;
  procname(args());
end_proc:

euler := prog::remember(euler, 
                        () -> [property::depends(args()),
                               Pref::autoExpansionLimit(), DIGITS,
                               slotAssignCounter("euler")]):

euler:= funcenv(euler):
euler::type:= "euler":
euler::print:= "euler":

euler::conjugate:=
  proc(n, x)
  begin
    if args(0) = 1 then 
         euler(n)
    else euler(n, conjugate(x))
    end_if;
  end_proc:

//----------------------------------------------------------------
// euler::diff implements euler'(n, x) = n*euler(n-1, x)
//----------------------------------------------------------------
euler::diff:= proc(e, x)
local n, f;
begin
  if has(op(e, 1), x) then
     return(FAIL);
  end_if;
  if not has(e, x) then
     return(0);
  end_if:
  [n, f]:= [op(e)];
  return(n*euler(n-1, f)*diff(f, x));
end_proc:

//----------------------------------------------------------------
// euler::float
// This is just a utility called by euler(n, x) for numerical n
// and x = real floating point value between 0 and 1. 
// It uses the series expansion
//  euler(2*n-1, x) = (-1)^n*4*(2*n-1)!/PI^(2*n) *
//                  sum( cos((2*k+1)*PI*x)/(2*k+1)^(2*n), k = 0 .. infinity)
//  euler(2*n, x) = (-1)^n*4*(2*n)!/PI^(2*n+1) *
//                  sum( sin((2*k+1)*PI*x)/(2*k+1)^(2*n+1), k = 0 .. infinity)
// This computation is much faster than the direct computation
// of euler(n, x) as a finite sum.
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// !!!! Only use for 0 < x < 1.!!!!!
// !!!! It produces completely !!!!!
// !!!! wrong values for other !!!!!
// !!!! values of x!           !!!!!
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//----------------------------------------------------------------

euler::float:= proc(n, x)
local tol, fPI, fPIx, s, k, n2, a, b, ds;
begin
   if args(0) = 1 then
      return(hold(euler)(n));
   end_if:
   x:= float(x):
   if domtype(n) = DOM_INT and
      domtype(x) = DOM_FLOAT and
      x > 0 and
      x < 1 then
     tol:= 10^(-DIGITS);
     fPI:= float(PI);
     fPIx:= fPI*x;
     s:= 0:
     k:= 1:
     if n mod 2 = 0 then // n is even
        [n2, n]:= [n, n/2];
        while TRUE do
          a:= sin::float(k*fPIx):
          b:= 1/k^(n2+1);
          ds:= a*b;
          if max(b, specfunc::abs(ds)) <= tol*specfunc::abs(s) then
             return((-1)^n*4*fact(n2)/fPI^(n2+1)*(s + ds))
          end_if:
          s:= s + ds;
          k:= k + 2;
        end_while;
      else // n is odd
        [n2, n]:= [n, (n+1)/2];
        while TRUE do
          a:= cos::float(k*fPIx):
          b:= 1/k^(n2+1);
          ds:= a*b;
          if max(b, specfunc::abs(ds)) <= tol*specfunc::abs(s) then
             return((-1)^n*4*fact(n2)/fPI^(n2+1)*(s + ds))
          end_if:
          s:= s + ds;
          k:= k + 2;
        end_while;
      end_if:
   end_if:
   if domtype(n) = DOM_INT and
      (domtype(x) = DOM_FLOAT or
       domtype(x) = DOM_COMPLEX) then
      return(euler(n, x));
   end_if:
   hold(euler)(n, x);
end_proc:


//----------------------------------------------------------------
// expand implements 
// (1) euler(n, x + 1) = 2*x^n - euler(n, x)
// (2) euler(n, m*x) = m^n*sum( (-1)^k*euler(n, x + k/m), k = 0 .. m-1)
// (3) euler(n, -x) = (-1)^(n+1)*euler(n, x) + 2*(-1)^n*x^n
//----------------------------------------------------------------
euler::expand:= proc(e)
local n, x, m, k, s;
begin
  if nops(e) = 1 then
     x:= op(e, 1);
     if domtype(x) = DOM_INT and x > 0 then
        return(specfunc::eulerNumber(x))
     else
        return(e)
     end_if:
  end_if:
  [n, x]:= map([op(e)], expand);

  case type(x)
     of "_plus" do
       //----------------------------------------------------------
       // implement rule (1): euler(n, x + 1) = 2*x^n - euler(n, x)
       //----------------------------------------------------------
       m:= select(x,  testtype, Type::Numeric);
       m:= floor(Re(m));
       if m >= 1 then
          //euler(n, x) = -euler(n, x-1) + 2*(x-1)^n
          //            = +euler(n, x-2) - 2*(x-2)^n + 2*(x-1)^n
          //            = -euler(n, x-3) + 2*(x-3)^n - 2*(x-2)^n + 2*(x-1)^n
          //            = ...
          return(expand((-1)^m*euler(n, x-m) + _plus(2*(-1)^(k+1)*(x-k)^n $ k = 1..m)))
       elif m = 0 then
          // Do nothing. Fall through to the code for expand(euler(n, -x));
          break;
       elif m < 0 then
          //euler(n, x) = -euler(n, x+1) + 2*(x + 0)^n
          //            = +euler(n, x+2) - 2*(x + 1)^n + 2*(x+0)^n
          //            = -euler(n, x+3) + 2*(x + 2)^n - 2*(x+1)^n + 2*(x+0)^n
          //            = ...
          return(expand((-1)^m*euler(n, x-m) + _plus(2*(-1)^k*(x+k)^n $ k = 0..-m-1)))
       end_if:
     of "_mult" do
       //-------------------------------------------------------
       // implement rule (2):
       //   euler(n, m*x) = m^n*sum( (-1)^k*euler(n, x + k/m), k = 0 .. m-1)
       //-------------------------------------------------------
       m := op(x, nops(x));
       // Do rule (2), if m is integer and x contains at least one symbol,
       // because we do not want to expand euler(n, 5*sqrt(2)) etc.
       if indets(x) minus Type::ConstantIdents = {} then
         break;
       end_if:
       if type(m) = DOM_INT then
          if m < 0 then
             return(expand(2*x^n + (-1)^(n+1)*euler(n, -x)))
          else // m > 0
             x := x / m;
             if testtype(m, Type::Odd) then
                return(expand(m^n*_plus((-1)^k*euler(n, x + k/m) $ k = 0..m-1)));
             else
                return(expand(-2/(n+1)*m^n*_plus((-1)^k*bernoulli(n+1, x + k/m) $ k = 0..m-1)));
             end_if:
          end_if;
       end_if;
       break;
  end_case:

  //-------------------------------------------------------
  // implement rule (3): 
  //    euler(n, -x) = (-1)^(n+1)*euler(n, x) + 2*(-1)^n*x^n
  //-------------------------------------------------------
  [s, x]:= stdlib::normalizesign(x);
  if s = -1 then
     return(expand(2*(-1)^n*x^n + (-1)^(n+1)*euler(n, x)))
  else
     return(euler(n, x));
  end_if:
end_proc:

//-----------------------------------------------------------------------
// auxiliary routine for computing the Euler numbers
//-----------------------------------------------------------------------
specfunc::eulerNumber:=
proc(n: Type::PosInt)
  local prefact, bound, betaval, k;
  save DIGITS;
  option remember;
begin
  if n mod 2 <> 0 then
    return(0)
  end_if:

  // We use
  //   euler(2*n) = 2*(2*n)! / (PI/2)^(2*n + 1)  * beta(2*n + 1)
  // with
  //   beta(2*n + 1) = sum( (-1)^k/(2*k + 1)^(2*n + 1), k = 0 ..infinity).
  // We compute the first factor with low precision
  // to get get an idea about the floating point precision
  // needed for the exact calculation:
  
  DIGITS:= 8;
  prefact:= 2*gamma(float(n+1))/float(PI/2)^(n + 1);

  // We know that our result euler(2*n) must be an integer; 
  // hence it suffices to sum the beta series up to a bound 
  // such that 
  // |sum((-1)^k/k^(2*n + 1), k=bound..infinity)| * prefact  < 1.
  // Note: abs(left hand side)  < 1/bound^(2*n + 1) * prefact,
  // hence we need bound^(2*n + 1) > prefact
  bound:= ceil(prefact^(1/(n + 1)));
  userinfo(10, "Bound:".expr2text(bound));
  // Since zeta equals 1 (rather precisely), we need as many digits
  // as prefact has in front of the decimal point (and add some 
  // safety digits):
  DIGITS:= max(8, ceil(ln(prefact) / ln(10.0)) + 10);
  userinfo(10, "Setting DIGITS to ".expr2text(DIGITS));
  prefact:= 2*expand(fact(n))/float(PI/2)^(n + 1);
  betaval:= _plus((-1)^k/float((2*k+1)^(n + 1)) $ k=0..bound-1);
  // The missing tail of the series
  // _plus((-1)^k/float((2*k+1)^(n + 1)) $ k=bound .. infinity);
  // is dominated by the first term.
  // Get the exact (integer) value of prefact*betaval by rounding.
  // Beware! The sum may be too close to an integer, such that
  // roundoff makes the use of ceil or floor dangerous! 
  // Do use round! 
  (-1)^(n div 2)*round(prefact*betaval);
end_proc:

specfunc::eulerNumber(0):= 1:
specfunc::eulerNumber(2):= -1:
specfunc::eulerNumber(4):= 5:
specfunc::eulerNumber(6):=-61:
specfunc::eulerNumber(8):= 1385:
specfunc::eulerNumber(10):= -50521:

/* end of file */
