/*++
fresnelS -- the Fresnel sine function

fresnelS(x) = int( sin(PI/2*t^2), t = 0.. x)
            = (1 + I)/4* (    erf(sqrt(PI)/2*(1+I)*x)
                          - I*erf(sqrt(PI)/2*(1-I)*x));

Properties:
  fresnelS(-x) = -fresnelS(x)
  fresnelS(conjugate(x)) = conjugate(fresnelS(x))

Parameters:
  x - an arithmetical expression

TODO: there is no hull attribute
++*/

fresnelS:=
proc(x)
  option noDebug;
  local s;
begin
  if args(0) <> 1 then 
     error("expecting one argument")
  end_if;
  if x::dom::fresnelS <> FAIL then 
     return(x::dom::fresnelS(args()));
  end_if;

  case type(x)
    of DOM_FLOAT do
      return(fresnelS::float(x));
    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT then
        return(fresnelS::float(x))
      end_if;
      break;
   of DOM_INT do
   of DOM_RAT do
      s:= specfunc::sign(x);
      return(s*procname(s*x));
/*
   of DOM_INTERVAL do
      return(fresnelS::hull(x))
*/
    of DOM_SET do
    of "_union" do
      return(map(x, fresnelS))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, fresnelS));
      else
        return(Dom::ImageSet(fresnelS(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  procname(args())
end_proc:

fresnelS := prog::remember(fresnelS, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("fresnelS")]):

fresnelS(0):= 0:
fresnelS(infinity):= 1/2:
fresnelS(-infinity):= -1/2:
fresnelS(I*infinity):= -I/2:
fresnelS(-I*infinity):= I/2:

fresnelS:= funcenv(fresnelS):
fresnelS::print := "fresnelS":
fresnelS::type := "fresnelS":
fresnelS::info := "fresnelS -- the Fresnel sine integral":

fresnelS::float := proc(x)
  local y, r, fPI;
  begin
     x:= float(x):
     fPI:= float(PI):
     if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(x)) and
        specfunc::abs(x) <= 10.0^(-DIGITS/5) then
           // numerical stabilization:
           return(fPI/6*x^3);
     end_if;
     y:= fPI^(1/2)*x/2:
     if domtype(y) = DOM_FLOAT then
        r := (1 + I)/4*(erf((1 + I)*y) - I*erf((1 - I)*y));
        return(op(r, 1)); // ignore imginary trash
     end_if;
     if domtype(y/I) = DOM_FLOAT then
        r := (1 + I)/4*(erf((1 + I)*y) - I*erf((1 - I)*y));
        return(I*op(r/I, 1)); // ignore real trash
     end_if;
     if domtype(y) = DOM_COMPLEX then
        r := (1 + I)/4*(erf((1 + I)*y) - I*erf((1 - I)*y));
        return(r);
     end_if;
     hold(fresnelS)(x);
  end_proc:

//------------------------------
fresnelS::conjugate:= proc(x)
 begin
   x:= conjugate(x);
   if type(x) = "conjugate" then
      hold(fresnelS)(x)
   else
      fresnelS(x)
   end_if;
end_proc:

//-------------------------------------------
// fresnelS(z) is real for z on the real line.
// Further, fresnelS(I*z) = -I*fresnelS(z) for
// all complex z
//-------------------------------------------
fresnelS::Re := proc(z)
  begin
   if is(z in R_) = TRUE then
      return(hold(fresnelS)(z))
   end_if;
   if is(z/I in R_) = TRUE then
      return(0);
   end_if;
   return(hold(Re)(hold(fresnelS)(z)));
end_proc:

fresnelS::Im := proc(z)
  begin
   if is(z in R_) = TRUE then
      return(0)
   end_if;
   if is(z/I in R_) = TRUE then
      return(-fresnelS(z/I));
   end_if;
   return(hold(Im)(hold(fresnelS)(z)));
end_proc:

fresnelS::sign := proc(z)
  begin
   // z == fresnelS(z)
   z:= op(z, 1):
   if is(z > 0) = TRUE then
      return(1)
   end_if;
   if is(z < 0) = TRUE then
      return(-1)
   end_if;
   if is(z/I > 0) = TRUE then
      return(-I)
   end_if;
   if is(z/I < 0) = TRUE then
      return(I)
   end_if;
   return(hold(sign)(hold(fresnelS)(z)));
end_proc:

//------------------------------
fresnelS::simplify := proc(a)
local s, x;
begin
  x:= simplify(op(a)); // a = fresnelS(x)
  [s, x]:= stdlib::normalizesign(x);
  return(s*hold(fresnelS)(x));
end_proc:

//------------------------------
fresnelS::Simplify := fresnelS::simplify:

//------------------------------
fresnelS::diff := proc(f, x)
local y;
begin
  y:= op(f):
  sin(PI/2*y^2)*diff(y, x);
end_proc:

/*------------------------------
fresnelS::inverse := "???":
------------------------------*/
fresnelS::undefined := {}:
fresnelS::realDiscont := {}:
fresnelS::complexDiscont := {}:

/* ----------------------------------------------------------- 
  The following version traces series back to erf.
   Problem: erf cannot be expanded around z = (1 +-I)*infinity
            With finite expansion points x0, the zero order term
            is not frsenelS(x0), but involves(erf((1+-I)*z0)!

fresnelS::series := proc(y, x) // series(fresnelS(y), x)
local z;
begin
  z:= sqrt(PI)/2*y;
  series((1 + I)/4*(erf((1 + I)*z) - I*erf((1 - I)*z)), args(2..args(0)));
end_proc:
--------------------------------------------------------------*/
// we need to reimplement the series attribute:

fresnelS::series:= loadproc(fresnelS::series, pathname("SERIES"), "fresnelS"):

fresnelS::Content := stdlib::genOutFunc("CfresnelS",1):

// end of file 
