/*++
The harmonic function

harmonic(x) = psi(x + 1) + EULER

    x - an arithmetical expression

++*/

harmonic:=
proc(x)
  option noDebug;
  local fx, i, j;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif args(0) > 1 then
    error("expecting no more than 1 argument");
  elif x::dom::harmonic <> FAIL then
    return(x::dom::harmonic(args()))
  end_if;

  if testtype(x, Type::Numeric) then
    fx:= float(x);
    if domtype(fx) = DOM_FLOAT and iszero(frac(x)) and fx < 0 
      then error("singularity");
    end_if
  end_if;

  if args(0) = 1 then
     case type(x)
       of DOM_FLOAT do
         return(harmonic::float(x) );
       of DOM_COMPLEX do
         if type(op(x,1)) = DOM_FLOAT or 
            type(op(x,2)) = DOM_FLOAT then
            return(harmonic::float(x) );
         end_if;
         break;

       // psi returns explicit results only
       // for certain integers and rationals.
       // Make sure, harmonic does the same:
       of DOM_INT do
         if 0 <= x and x <= Pref::autoExpansionLimit() then 
           return(_plus(1/i $ i=1..x) );
         end_if;
         if x < 0 then
            error("singularity");
         end_if;
         break;
       of DOM_RAT do
          if domtype(2*x) = DOM_INT and
             specfunc::abs(x) <= Pref::autoExpansionLimit() then
            if x < 0 then 
               return(- 2*ln(2) + 2*_plus(1/(2*i-1)$i=1..(op(-x,1)) div 2))
            else
               return(- 2*ln(2) + 2*_plus(1/(2*i-1)$i=1..op(x,1)+1 div 2))
            end_if
          end_if;
          if x > 1 and x <= Pref::autoExpansionLimit() then
             fx:= floor(x);
             x:= x - fx;
             return(harmonic(x) + _plus(1/(x + j) $ j = 1 .. fx));
          end_if:
          if x < 0 and x >= -Pref::autoExpansionLimit() then
             fx:= floor(x);
             x:= x - fx;
             return(harmonic(x) - _plus(1/(x + j) $ j = fx + 1 .. 0));
          end_if;
          break
       of DOM_SET do
       of "_union" do
         return(map(x, harmonic))
       otherwise
         if not testtype(x,Type::Arithmetical) then
            /* generic handling of sets */
            if testtype(x, Type::Set) then
              if type(x)=Dom::ImageSet then
                return(map(x, harmonic));
              else
                return(Dom::ImageSet(harmonic(#x), #x, x));
              end_if;
            end_if;

           error("argument must be of 'Type::Arithmetical'")
         end_if
     end_case;
  end_if;

  procname(args())
end_proc:

harmonic := prog::remember(harmonic,
                           () -> [property::depends(args()), DIGITS,
                                  Pref::autoExpansionLimit(),
                                  slotAssignCounter("harmonic")]):

harmonic(-1/2) := - 2*ln(2):
harmonic(-2/3) := - 3/2*ln(3) - sqrt(3)/6*PI:
harmonic(-1/3) := - 3/2*ln(3) + sqrt(3)/6*PI:
harmonic(-3/4) := - 3*ln(2) - PI/2:
harmonic(-1/4) := - 3*ln(2) + PI/2:
harmonic(-5/6) := - 2*ln(2) - 3/2*ln(3) - sqrt(3)/2*PI:
harmonic(-1/6) := - 2*ln(2) - 3/2*ln(3) + sqrt(3)/2*PI:
harmonic(0) := 0:
harmonic(1/2) := 2  - 2*ln(2):
harmonic(1/3) := 3  - 3/2*ln(3) - sqrt(3)/6*PI:
harmonic(2/3) := 3/2- 3/2*ln(3) + sqrt(3)/6*PI:
harmonic(1/4) := 4  - 3*ln(2) - PI/2:
harmonic(3/4) := 4/3- 3*ln(2) + PI/2:
harmonic(1/6) := 6  - 2*ln(2) - 3/2*ln(3) - sqrt(3)/2*PI:
harmonic(5/6) := 6/5- 2*ln(2) - 3/2*ln(3) + sqrt(3)/2*PI:
harmonic(1) := 1:
harmonic(infinity) := infinity:

harmonic:= funcenv(harmonic):
harmonic::type:= "harmonic":
harmonic::print:= "harmonic":
harmonic::info:= "harmonic -- the harmonic function [try ?harmonic for details]":

harmonic::diff:=
  proc() 
    local f, x;
  begin
    f:= op(args(1),1);
    x:= args(2..args(0));
    psi(f + 1, 1)*diff(f, x)
  end_proc:


harmonic::float:=
  proc(x)
  local fx, y, absy, x0, dx, extraDIGITS;
  save DIGITS;
  begin
     fx:= float(x);
     if domtype(fx) = DOM_FLOAT or
       (domtype(fx) = DOM_COMPLEX and domtype(op(fx, 1)) = DOM_FLOAT) then
       if specfunc::abs(fx) < 10^(-DIGITS) then
          // We are near the zero at the origin. Use the Taylor expansion
          // harmonic(x) = PI^2/6*x + zeta(3)*x^2 + O(x^3)
          // to produce a numerically stable result. 
          return(float(PI)^2/6*fx); 
       end_if:
       extraDIGITS:= 0:
/*
       if domtype(x) = DOM_FLOAT or
         (domtype(x) = DOM_COMPLEX and 
          domtype(op(x, 1)) = DOM_FLOAT) then
          x:= numeric::rationalize(x);
       end_if:
*/
       // Danger: we wish to call psi(1 + x), but 1 + x 
       // may come close to a negative integer and may 
       // cause an error in psi::float when rounded to 
       // this integer:
       if domtype(fx) = DOM_FLOAT and fx < 0 then
          x0:= round(x);
          dx:= float(x - x0);
          if specfunc::abs(dx) < 10^(-0.6*DIGITS) then
             // For x0 = negative integer we have
             // series(harmonic(x), x = x0) = 
             // -1/(x-x0) + harmonic(-1-x0) + O((x - x0))
             return(-1/dx + harmonic(-1-x0));
          end_if:
       end_if:
       // evaluate harmonic(x) = psi(x + 1) + EULER.
       // Boost DIGITS, if necessary:
       repeat
          y:= psi::float(1 + x) + float(EULER);
          absy:= specfunc::abs(y);
          if absy > 10^(-1 - extraDIGITS) then
             return(y)
          else
             extraDIGITS:= extraDIGITS+ min(5, ceil(-ln(absy)/2.302585093));
             DIGITS:= DIGITS + extraDIGITS;
          end_if;
       until FALSE end_repeat;
     end_if;
     return(hold(harmonic)(fx));
  end_proc:

harmonic::expand:=
    loadproc(harmonic::expand, pathname("STDLIB", "EXPAND"), "harmonic"):

harmonic::series:=
    loadproc(harmonic::series, pathname("SERIES"), "harmonic"):

harmonic::rectform:=
    loadproc(harmonic::rectform, pathname("STDLIB", "RECTFORM"), "harmonic"):


// end of file 
