/*++
	heaviside -- the Heaviside function

	heaviside(x)

	x - expression

	heaviside(x) :=      1            if x>0
                        0            if x<0
                        1/2          if x=0
                        undefined    if x is a non-real complex number
                        heaviside(x) otherwise.
++*/

heaviside :=
proc(x)
  local X, a;
  option noDebug;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif x::dom::heaviside <> FAIL then
    return(x::dom::heaviside(args()))
  elif args(0) <> 1 then
    error("expecting one argument")
  end_if;

  case type(x)
    of DOM_SET do // fall through
    of "_union" do
      return(map(x, heaviside))
  end_case;

  if domtype(x) = DOM_FLOAT then
    if x > 0 then
      return(float(1));
    end_if;
    if x < 0 then
      return(float(0));
    end_if;
    // this only leaves x = 0.0
    // return(procname(x));
    return(0.5)
  end_if;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, heaviside));
      else
        return(Dom::ImageSet(heaviside(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  elif is(x>0)=TRUE then
    return(1);
  elif is(x<0)=TRUE then
    return(0);
  elif domtype(x)=DOM_COMPLEX then
    return(undefined);
  end_if;

  // heaviside(a*X + b, n) = heaviside(sign(a)*X + b/abs(a))
  // Identify symbolic X and numerical a, b
  // Use numeric::indets rather than indets, because
  // indets(2*x[1]+3) -> {x}, but one needs
  // numeric::indets(2*x[1]+3) -> {x[1]}
  // We can do this simplification only if there is
  // exactly one indeterminate X. Otherwise we do
  // not know, which indeterminate is to represent
  // the distribution variable:
  X:= numeric::indets(float(x)):
  if nops(X)=1 then
     X:= op(X);
     a:= lcoeff(poly(x, [X])):
     if domtype(float(a)) = DOM_FLOAT then
        return(procname(x/abs(a)));
     end_if;
  end_if;

  procname(x)
end_proc:

heaviside := prog::remember(heaviside, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("heaviside")]):

// the best choice for the sake of integration and Fourier analysis 
heaviside(0):= 1/2:

heaviside:= funcenv(heaviside):
heaviside::type:= "heaviside":
heaviside::info:= "heaviside -- the Heaviside step function":
heaviside::print:= "heaviside":
heaviside::float:= x -> heaviside(float(x)):

heaviside::series:=
	loadproc(heaviside::series, pathname("SERIES"), "heaviside"):

heaviside::diff:=
  proc(f,x)
  begin
    eval(diff(subsop(f,0=dirac)*diff(op(f),x),args(3..args(0))))
  end_proc:

heaviside::"transform::laplace":=
    loadproc(heaviside::"transform::laplace",
             pathname("TRANS","LAPLACE"), "L_heavis"):

heaviside::simplify:=
    loadproc(heaviside::simplify, pathname("STDLIB","SIMPLIFY"), "heaviside"):


// end of file 
