// code needed for hypergeometric function
alias(hypergeom=hypergeom::float):
alias(Hypergeom=specfunc::Hypergeom):

Hypergeom:=newDomain("specfunc::Hypergeom"):
Hypergeom::create_dom:=hold(specfunc::Hypergeom):

alias(slater=Hypergeom::slater):
alias(slater2=Hypergeom::slater2):
alias(ordinary=Hypergeom::ordinary):
alias(confluent=Hypergeom::confluent):
alias(flip=Hypergeom::flip):
alias(f2F1_reflect=Hypergeom::f2F1_reflect):
alias(typenegfloatint=Hypergeom::typenegfloatint):
alias(typenumcomplex=Hypergeom::typenumcomplex):
alias(sel=Hypergeom::sel):
alias(complexfilter=Hypergeom::complexfilter):
alias(asy_slater_cond=Hypergeom::asy_slater_cond):
alias(asympt=Hypergeom::asympt):
alias(CondC=Hypergeom::CondC):
alias(CondA=Hypergeom::CondA):
alias(spf1=Hypergeom::spf1):
alias(spf2=Hypergeom::spf2):
alias(Lpq=Hypergeom::Lpq):
alias(Kpq=Hypergeom::Kpq):
alias(cluke=Hypergeom::cluke):
alias(Ts=Hypergeom::Ts):
alias(Us=Hypergeom::Us):
alias(contain=Hypergeom::contain):
//
alias(sexp=specfunc::exp):
alias(sabs=specfunc::abs):
alias(sgamma=specfunc::gamma):
alias(spsi=specfunc::psi):
alias(strunc=specfunc::trunc):
alias(sround=specfunc::round):
alias(scos=specfunc::cos):
alias(ssin=specfunc::sin):
//
alias(ln10=0.4342944819):
alias(fln=specfunc::Bessel::fln):
alias(Fln=Hypergeom::Fln):


//------------------------------------------------------
// Problem: Voldemort passes parameters as float values.
// After boosting DIGITS, iszero(frac(a - b)) is a highly
// questionable test whether a and b differ by an integer.
// Make this test numerically more stable by setting the
// fractional part to 0 if it is small relative to the 
// input parameter:
//------------------------------------------------------
specfunc::Hypergeom::frac:= proc(x)
local fx, fr;
begin
   fx:= float(x);
   fr:= specfunc::frac(fx);
   if domtype(fx) <> DOM_FLOAT then
      return(fr);
   elif iszero(fx) then
      return(0);
   elif sabs(fr) <= 10^(-DIGITS)*sabs(fx) then
      return(0);
   else
      return(fr);
   end_if;
end_proc:

alias(sfrac=specfunc::Hypergeom::frac):

//----------------------------------------
Fln:=proc(t)
begin
 if not(iszero(t)) then
        fln(sabs(t)):
 else
        infinity:
 end_if:
end_proc:

//----------------------------------------
// check if the argument is a nonpositive integer:
typenegfloatint := proc(a)
local af;
begin
  if domtype(a) = DOM_INT and a <= 0 then
     return(TRUE);
  end_if;
  // double check if float(a) is a nonpositive
  // integer within machine precision
  af:=float(a):
  if type(af)=DOM_FLOAT and 
     af <= 0 and 
     sabs(af-sround(af)) < float(10^(1-DIGITS)) then
          return(TRUE):
     else return(FALSE):
  end_if:
end_proc:

//----------------------------------------
// select the largest negative integer in the list as:
sel:=proc(as)
local q;
begin
    q:=select(as,typenegfloatint):
    if q = [] then
      -infinity:
    else
       max(op(q)):
    end_if:
end_proc:

//----------------------------------------
typenumcomplex := proc(a)
local ar,aI,af;
save DIGITS;
begin
        DIGITS:=10:
        af:=float(a):
        if type(af)=DOM_FLOAT then
                return(TRUE):
        elif type(af/I)=DOM_FLOAT then
                return(TRUE):
        elif type(af)=DOM_COMPLEX then
                ar:=op(af,1):
                aI:=op(af,2):
                if type(ar)=DOM_FLOAT and type(aI)=DOM_FLOAT then
                return(TRUE):   
        else
                return(FALSE):
        end_if:
else
        return(FALSE):
end_if:
end_proc:

// Verifies condition of C of Y.L. Luke 5.7 (4)
CondC:=proc(as,p)
local astar,i,j,df,res;
begin
res:=TRUE:
for i from 1 to p do
        astar := subsop(as,i=null()):
        for j from 1 to p-1 do
                df:=as[i]-astar[j]:
                if iszero(df) or iszero(sfrac(df)) then
                        return(FALSE):
                end_if:
        end_for:
end_for:
res:
end_proc:

// Verifies condition of C of Y.L. Luke 5.7 (3) p. 178
CondA:=proc(as,bs,n,m)
local i,j,df,res;
begin
res:=TRUE:
for i from 1 to n do
        for j from 1 to m do
                df:=as[i]-bs[j]:
                if type(float(df))=DOM_FLOAT then
                        if iszero(sfrac(df)) and round(df) > 0 then
                                return(FALSE):
                        end_if:
                end_if:
        end_for:
end_for:
res:
end_proc:

// ===============================================================
// routine spf1 computes diff((x^(n-b))*exp(x)*(g(b-1,x)),x$(n-1))
// where g(b,x) = gamma(b)-igamma(b,x) for n=1..6
// ===============================================================
spf1:=proc(nm1,x,b)
local a,xb,gbx,gbx1,gbx2,oldDIGITS;
save DIGITS;
begin   
  oldDIGITS:=DIGITS:
  DIGITS:=DIGITS+30:
  // code for small igamma(b-1,x) is unstable for large b and small x:
  if sabs((x^(b-1)/(b-1))) < float(10^(1-oldDIGITS)) then       
     // if gbx registers zero against our better judgement, use
     // Taylor series expansion of A and S. 6.5.12: i.e.:
     // little igamma(a,x) = (x^a/a)*hypergeom([a],[1+a],-x)
        a:=b-1:
        gbx:=float((x^a/a)*(1+(-a/(1+a)+(1/2*a/(2+a)+(-1/6*a/(3+a)+
        (1/24*a/(4+a)+(-1/120*a/(5+a)+(1/720*a/(6+a)+(-1/5040*a/(7+a)+
        (1/40320*a/(8+a)-1/362880*a/(9+a)*x)*x)*x)*x)*x)*x)*x)*x)*x)):
  else
        gbx1:= float(sgamma(b-1)):         
        gbx2:= float(igamma(b-1,x)):         
        gbx := gbx1 - gbx2:
        if specfunc::abs(gbx) <= 10^(oldDIGITS-DIGITS)*specfunc::abs(gbx1) then
           // catastrophic cancellation that we could
           // not compensate by the DIGITS boost above
           return(FAIL);
        end_if;
  end_if: 
  xb  := float(x^(1-b)*exp(x)):
  case nm1 
  of 0 do return(xb*gbx): break;
  of 1 do return(xb*(2-b+x)*gbx+1): break;
  of 2 do return(x+4-b+xb*(6+(-5+b)*b+(6-2*b+x)*x)*gbx): break;
  of 3 do return(18+(-8+b)*b+(10-2*b+x)*x+xb*(24+(-26+(9-b)*b)*b
           +(36+(-21+3*b)*b+(12-3*b+x)*x)*x)*gbx): break;
  of 4 do return(96+(-58+(13-b)*b)*b+(86+(-31+3*b)*b+
          (-3*b+18+x)*x)*x+xb*(120+(-154+(71+(-14+b)*b)*b)*b+
          (240+(-188+(48-4*b)*b)*b+(120+(-54+6*b)*b+(20-4*b+
          x)*x)*x)*x)*gbx): break;
  of 5 do return(600+(-444+(136+(-19+b)*b)*b)*b+(756+(-374+
          (66-4*b)*b)*b+(246+(-75+6*b)*b+(-4*b+28+x)*x)*x)*x+
          xb*(720+(-1044+(580+(-155+(20-b)*b)*b)*b)*b+(1800+
          (-1710+(595+(-90+5*b)*b)*b)*b+(1200+(-740+(150-10*b)*b)*
          b+(300+(-110+10*b)*b+(30-5*b+x)*x)*x)*x)*x)*gbx): break;
  otherwise
    error("illegal call");
  end_case:
end_proc:

//===================================================================
// routine spf2 computes diff(exp(1/x)*(x^(-n+n))*g(-b+n+1,1/X),X$n): 
// where g(b,x) = igamma(b,x) for n=1..6
//===================================================================
spf2:=proc(n,x,b)
  local exb,gbx;
  save DIGITS;
begin
  DIGITS:=DIGITS+30:
  gbx := float(igamma(-b+n+1,1/x)):
  exb := float(x^(n-b)*exp(1/x)):
  case n 
  of 0 do return(exb*gbx): break:
  of 1 do return(1-exb*(1+(b-1)*x)*gbx): break:
  of 2 do return(-1-x*b+exb*(1+(-2+2*b+(2+(-3+b)*b)*x)*x)*gbx): break:
  of 3 do return(1+(2*b+(3+(b-2)*b)*x)*x-exb*(1+(3*b-3+(6+(-9+3*b)*b+(-6+(11+(-6+b)*b)*b)*x)*x)*x)*gbx): break:
  of 4 do return(-1+(-3*b+(-8+(5-3*b)*b+(-10+(-b+5)*b)*b*x)*x)*x+exb*(1+(-4+4*b+(12+(-18+6*b)*b+(-24+(44+(-24+4*b)*b)*b+(24+(-50+(35+(-10+b)*b)*b)*b)*x)*x)*x)*x)*gbx): break:
  of 5 do return(1+(4*b+(15+(-9+6*b)*b+((38+(-18+4*b)*b)*b+(40+(-39+(31+(-9+b)*b)*b)*b)*x)*x)*x)*x-exb*(1+(5*b-5+(20+(-30+10*b)*b+(-60+(110+(-60+10*b)*b)*b+(120+(-250+(175+(-50+5*b)*b)*b)*b+(-120+(274+(-225+(85+(-15+b)*b)*b)*b)*b)*x)*x)*x)*x)*x)*gbx): break:
  of 6 do return(-1+(-5*b+(-24+(14-10*b)*b+((-92+(42-10*b)*b)*b+(-180+(168+(-145+(42-5*b)*b)*b)*b+(-252+(196+(-77+(14-b)*b)*b)*b)*b*x)*x)*x)*x)*x+exb*(1+(-6+6*b+(30+(-45+15*b)*b+(-120+(220+(-120+20*b)*b)*b+(360+(-750+(525+(-150+15*b)*b)*b)*b+(-720+(1644+(-1350+(510+(-90+6*b)*b)*b)*b)*b+(720+(-1764+(1624+(-735+(175+(-21+b)*b)*b)*b)*b)*b)*x)*x)*x)*x)*x)*x)*gbx): break:
  otherwise
    error("illegal call");
  end_case:
end_proc:

//=====================================================
// Evaluates pFq(z) for p>q+1 or p=q+1 and |z|>1 via Slater's theorem 
// (for this special case of the Meijer G function). It 
// computes pFq(..,..,z) as a sum of (q+1)F(p-1)(.., .., 1/z) terms.
// We allow this routine to error-return if any of the GAMMA's 
// hits a singularity -- this represents a multiple pole problem, 
// which we deal with in the calling routine.
//=====================================================
slater := proc(as, bs, z)
    local p, q, z1, lnz, prod1, prod2, s, a, astar, i, j, flag,res;
begin
    p     := nops(as):
    q     := nops(bs):
    z1    := (-1)^(p+q+1)/z:
    lnz   := ln(-z):
    prod1 := _mult(sgamma(bs[j])$j=1..q)/_mult(sgamma(as[j])$j=1..p):
    s     := float(0):
    for i from 1 to p do
      a :=as[i]:
      astar:=subsop(as,i=null()):
      prod2:=_mult(sgamma(astar[j]-a)$j=1..p-1)/_mult(sgamma(bs[j]-a)$j=1..q):
      // Walter 2.10.09: the original code was
      //([flag,res]):=ordinary([a,(1-bs[j]+a)$j=1..q], [(1-astar[j]+a)$j=1..p-1], z1):
      // Since ordinary does a straightforward Taylor expansion around 0,
      // there is a problem when z is close to 0, i.e., when z1 is large.
      // Hence, call hypergeom to let it decide, whether to use asymptotics 
      // or whatever may be the best algorithm:
      if abs(z1) < 1000 then
        ([flag,res]):=ordinary([a,(1-bs[j]+a)$j=1..q], [(1-astar[j]+a)$j=1..p-1], z1):
      else
        res:= hypergeom([a,(1-bs[j]+a)$j=1..q],
                        [(1-astar[j]+a)$j=1..p-1], z1):
      end_if;
      if res = RD_NAN then
         return(RD_NAN);
      end_if;
      s := s + sgamma(a) *prod1*prod2*sexp(-a*lnz)*res:
    end_for:
    s:
end_proc:

//=====================================================
// Evaluates pFq(z) via the basic series definition. 
// The p=1, q=0 case is handled directly, 
// the p=2, q=1 case is handled for |z| ~ 1 
// by the linear transformation formula A&S 15.3.6, and 
// the general case is dealt with by simple summation.
//
// Note: The denominator singularity case is handled before
// this routine is called.
//=====================================================
ordinary := proc(as, bs, z)
    local a0,b0,df,dfg,s,abs_s,oldDIGITS,p,q,d,pred,
    min_k,max_k,mag,old_s,k,t,abs_t, psi, Repsi,t1,t2,t3,
    i,try,tol,exdigits,ans,flag,conv,phiq, tmp;
    save DIGITS;
begin
    tol:=float(10^(-DIGITS)): 
    ans:=array(1..6):
    oldDIGITS := DIGITS:
    p := nops(as):
    q := nops(bs):
    // double-check possibilites of poles
    a0 := sel(as):
    b0 := sel(bs):
    if b0 > a0 then
        error("singularity encountered"):
    end_if:
    if contains(float(as),float(0)) > 0 and 
                iszero(contains(float(bs),float(0))) then
        return([0,float(1)]):
    end_if:
    if p = 1 and q = 0 then
        // 1F0(a:z) = (1-z)^(-a) except at z=1
        if iszero(z-1) then
            if Re(as[1]) < 0 then
                return([0,float(0)]):
              else
                error("singularity encountered"):
            end_if:

          else
            return([0,float((1-z)^(-as[1]))]):
        end_if:
    end_if:

    if p = 0 and q = 1 then
        return([0,float(sgamma(bs[1])*(z^((1-bs[1])/2))*
                                        besselI(bs[1]-1,2*sqrt(z)))]):
    end_if:

    if p=1 and q=1  then
        if sabs(as[1]-bs[1]) < tol then
                return([0,sexp(z)]):
        end_if:
    end_if:
    d    := sround(ln10*max(DIGITS/ln10, op(map([op(as),op(bs)],Fln)))):
    pred := max(0,sround(ln10*Fln(z)))+6:
    // extra guard digits for the case when Re(z) is negative
    // (alternating series) and z gets large.
    if Re(z) < 0 and sabs(z) > float(10) then
       exdigits:= min(200, round(sabs(z/2))):
    else
       exdigits:=0:
    end_if:
    // denominator coefficient terms b = u+I*v with u < 0 and |v| < 1 can
    // cause trouble, as then the terms of the series can increase after
    // starting to decrease (the problem occurs when b+k ~ 0): we deal with
    // this by simply forcing the loop to run at least -2*floor(u) iterations:
    // this is not the best solution, but it is by far the simplest.
    min_k:= max(0,op(map(bs,x->if abs(Im(x))<1 then -2*floor(Re(x)) else null() end_if))):
    flag := 0: 
    for try from 1 to 6 do
        DIGITS := d + pred + 2 + exdigits:
/*
        if DIGITS > 10^4 then
           error("cannot compute float value");
        end_if;
*/
        old_s  := float(0):
        s      := float(1):
        t      := float(1):
        mag    := float(0):
        // the "and t <> 0" check allows this loop to handle the polynomial
        // case, even if there is a negative integer denominator term (if
        // there is such a denominator term, we know at this point that it
        // is more negative than the least negative integer numerator term)<

        // case p <= q  Taylor series converges (no matter what) 
        // |z=1| must be handled with care though
        // follow conditions PBM 7.2.3.1 
        phiq := Re(float(_plus(op(bs))-_plus(op(as)))):
        if (a0 > b0 and a0 > -infinity) then 
                conv:=TRUE:
        // Walter 7.8.07: changed the following condition 
        // sabs(z) < float(1) to sabs(z) < 0.99, otherwise
        // the convergence may be too slow
        elif p <= q or (p = q+1 and sabs(z) < 0.99) then
                conv:=TRUE:     
        elif p = q+1 and sabs(z)=float(1) and phiq > float(0) then      
                conv:=TRUE:
        elif p = q+1 and sabs(z)=float(1) and sabs(z-1) > tol and
                float(-1) < phiq and phiq <= float(0) then
                conv:=TRUE:
        else
                conv:=FALSE:    
        end_if:
        if conv then
          if p = q and Re(z) > 10^7 then
             // The asymptotic form of the hypergeom
             // involves exp(z) which will overflow.
             return([1, RD_NAN]);
          end_if:
          for k from 0 to RD_INF do 
             tmp:= _mult((as[i]+k)$i=1..p):
             if iszero(tmp) then 
                break; 
             end_if;
             t := t*z*_mult((as[i]+k)$i=1..p)/
                      _mult((bs[i]+k)$i=1..q)/(k+1):     
             abs_t := sabs(t):
             s := s + t:     
             abs_s := sabs(s):
             if (abs_t < tol*abs_s or iszero(t)) and k > min_k then 
                 break: 
             elif abs_t > 0 then
                 mag:=max(mag,sabs(fln(abs_t))): 
             end_if:
             //================================================================
             // Bug fix (Walter 22.8.08): In hypergeom([-3/2,1/2],[-1/2],x),
             // x = -1/2 is a root. This loop, however, would never stop for 
             // this value. So we need an additional stopping criterion. 
             // Since we have started with s = 1 and add s:= s + t, we can
             // stop when the increment t has a small absolute value:
             //================================================================
             if abs_s < tol and  abs_t < tol^2 then
                break;
             end_if:
          end_for: 
        elif p = q+1 and
             (psi:= _plus(op(as)) - _plus(op(bs)); 
              Repsi:= Re(psi);
              Repsi > 0  // otherwise the limit (0) below is not correct
             ) 
             and Repsi < 0.9 // otherwise the convergence of the
                             // following series is too slow
             and sabs(z) < 1
             and sabs(z-1) <= 0.01 then
          //====================================================================
          // Walter 27.4.09: implemented the following acceleration of the
          // Taylor series for values z close to the critical point z = 1.
          // Accelerate the convergence using
          //    limit((1-z)^psi*pFq(a,b,z), z=1) = B/A*gamma(psi)  (0)
          // where
          //     psi:= _plus(op(a)) - _plus(op(b));   Re(psi) > 0
          //       A:= _mult(op(map(a,gamma)))
          //       B:= _mult(op(map(b,gamma)))
          // Using
          //   gamma(psi)*(1-z)^(-psi) = sum(gamma(psi+k)*z^k/k!, k = 0..infinity)
          // and 
          //   pFq(a, b, z), z = 1) = sum(z^k/k! *            (1)
          //              _mult(map(a, gamma@_plus, k)) /     (1)
          //              _mult(map(b, gamma@_plus, k))       (1)
          //             , k = 0 .. infinity)                 (1)
          // we find
          //   pFq(a, b, z), z = 1) =
          //    B/A * ( gamma(psi)/(1-z)^psi                  (0)
          //           +sum(z^k/k! *                          (2)
          //            ( _mult(map(a, gamma@_plus, k)) /     (2)
          //              _mult(map(b, gamma@_plus, k))       (2)
          //              - gamma(psi + k)                    (2)
          //            ), k = 0 .. infinity)                 (2)
          // Note that the coefficients t(k) of the Taylor series (1) = sum(t(k)*z^k)
          // behave like t(k) = O(k^(psi-1)) as k -> infinity, whereas the coeffs
          // tt(k) of the series (2) = sum(tt(k)*z^k) behave like tt(k) = O(k^(psi-2))
          // as k -> infinity. Hence, both (1) and (2) converge for |z| < 1, but the 
          // convergence of (2) is somewhat faster than the convergence of (1).
          //====================================================================
          tmp:= _mult(op(map(as,gamma)))/_mult(op(map(bs,gamma)));
          t1:= tmp;
          t2:= gamma(psi):
          t3:= 1:
          s:= t2/(1-z)^psi + t1 - t2;
          for k from 0 to RD_INF do
            t1 := t1*_mult((as[i]+k)$i=1..p)/  // t1 = gamma(a[1]+k+1)*...*gamma(a[p]+k+1)
                     _mult((bs[i]+k)$i=1..q);  //     /gamma(b[1]+k+1)*...*gamma(b[q]+k+1)
            t2:= t2*(psi + k);                 // t2 = gamma(psi + k + 1)
            t3:= t3*z/(k+1):                   // t3 = z^(k+1)/(k+1)
            t:= (t1 - t2)*t3;
            s := s + t:
            // An estimate of the truncation error is 
            //   sum(k^(psi - 2), k = K..infinity) 
            //   = K^(psi-1)/(1-psi) + O(K^(psi - 2))
            //  ~=  t*K/(1-psi), where t = K^(psi-1) is
            // the last term that was added to the sum
            if abs(t*(k+1)) <= tol*abs(s*(1 - psi)) then
               break;
            end_if;
          end_for;
          s:= s/tmp;
        else
          assert(abs(z) <= 1):
          // conditionally convergent at z=1 or 
          // divergent but often asymptotic series
          psi:= _plus(op(as)) - _plus(op(bs)); 
          Repsi:= Re(psi);

          max_k:= 1000*DIGITS:
          if abs(1-z) < 1/1000 then
             // Close the the critical point z = 1, we
             // have to expect very poor convergence.
             // The k-th term in the sum has magnitude
             // k^(Repsi - 1)*|z|^k  in the series.
             // The maximal value is attained for
             // k ~= 1/(|z|^(1/(1-Repsi) - 1). We
             // need to sum up beyond that value:
             if iszero(Repsi - 1) then
                max_k:= -DIGITS*10*ln(10.0)/ln(abs(z))
             else
                max_k:= DIGITS/(abs(z)^(1/(1-Repsi)) - 1);
             end_if;
          end_if;
          df:=float(2):
          for k from 0 to RD_INF do
            old_s:=s:
            t := t*z*_mult((as[i]+k)$i=1..p)/
                     _mult((bs[i]+k)$i=1..q)/(k+1):
            abs_t := sabs(t):
            s := s + t:
            abs_s := sabs(s):
            df := min(df,sabs(s-old_s)):
            if abs_t < tol*abs_s or iszero(t) or 
               (sabs(s-old_s) > df and df < float(1/2)) 
               and k > min_k then
                 break:
            elif k > max_k then
                 break:
            else
                mag:=max(mag,sabs(fln(abs_t))):
            end_if:
          end_for: 
          // if number is hopeless, flag = 1
          if abs_t > 100*tol*abs_s and not(iszero(t)) and 
             sabs(df) > 100*tol then
               flag:=1:
          end_if:
        end_if:
        ans[try]:=s:
        if try > 1 then
          dfg:=sabs(ans[try]-ans[try-1]):
        else
          dfg:=infinity:
        end_if:
        if iszero(s) or 
          sround(ln10*(mag - Fln(s))) <= pred or dfg < tol*abs_s then
          break:
        else
          pred := sround(ln10*(mag - Fln(s))) + 2:
        end_if:
    end_for:
    [flag,s]:
end_proc:

//====================================================
// flip
//
//  Evaluates pFq(z) for p>q+1 or p=q+1 and |z|>1 via Slater's theorem.
//
//====================================================
flip := proc(as, bs, z)
local oldDIGITS, p, q, s, eps, as1, bs1, old_s, i, try, tol;
save DIGITS;
begin
     tol:=float(10^(-DIGITS)):
     oldDIGITS := DIGITS:
     DIGITS    := DIGITS + 3:
     p         := nops(as):
     q         := nops(bs):
     if CondC(as,p) and CondA(as,bs,p,q) 
        // Walter, 13.8.02: I added the following
        // 'and sabs(z) > 1', because of Voldemort's comment
        // above that Slater is appropriate for this case
        and sabs(z) > 1 
     then
        // add guard digits
        DIGITS := DIGITS + 10:
        s := slater(as,bs,z):
        if s = RD_NAN then
           return(s);
        end_if;
     else
        // We must use a limit process
        // eps is used to avoid multiple pole problems in Slater's theorem.
        // Basically, we are trying a numerical limit here. Numerical
        // experiments suggest we need at least p*oldDIGITS digits.
        // Walter, 2.1.2009: redefined Voldemort's
        // eps := min(op(map([op(as),op(bs)],sabs)))*tol:
        // New version:
        eps    := tol:
        DIGITS := p*oldDIGITS+20:
        s:=float(0):
        old_s:=float(1):
        for try from 1 to 6 do
           DIGITS := DIGITS + 20:
           eps := eps/float(10000):
           // Walter, 2.1.2009: I changed the shift to some
           // combined absolute/relative shift (this fixes a bug):
           as1 := [as[i] +  (i-1) *max(abs(as[i]), 10^(-10))*eps $ i=1..p]:
           bs1 := [bs[i] + (p+i-1)*max(abs(bs[i]), 10^(-10))*eps $ i=1..q]:
           old_s := s:
           s := slater(as1,bs1,z):
           if s = RD_NAN then
              return(s);
           end_if;
           if sabs(s-old_s) <= tol*max(10^(-10), sabs(s)) then 
              break: 
           end_if:
        end_for:
   end_if:
   return(s);
end_proc:

// 2F1_reflect
//
// Evaluates a 2F1 hypergeometric function for arguments z such that |1-z| < 1
// (and not particularly close to 1) via the linear transformation A&S 15.3.5
// (and its special cases A&S 15.3.10/11/12).
//
// The function being evaluated is 2F1(a,b: c; 1-z) (i.e., the parameter to
// this routine is assumed to have already been transformed by z->1-z)
//
// Note: The case c a non-positive integer is handled elsewhere, but this
// routine must handle the possibility that a or b is a non-positive integer.
//

f2F1_reflect := proc(a, b, c, zin)
    local df,oldDIGITS, m, s, c_a, c_b, abs_t, abs_s1, s1,s2,old_s2, a1, a2,
        b1, b2, z1, z2, t, t1, t2, r1, r2, k, m1, tol, z;
    save DIGITS;
begin
    tol := float(10^(-DIGITS)):
    oldDIGITS := DIGITS:
    z := 1 - zin:
    // use 15.3.4 (or 15.3.5) of A & S
    if Re(zin) < float(0) and sabs(zin) < float(2) then
        s := float((z^(-a))*hypergeom([a,c-b],[c],-zin/z)):
        return(s):
    end_if:

    // The series representations used further down below are only valid
    // for |1 - zin| = |z| < 1. (Small |z| ==> fast convergence).
    // If |z| is too close to 1, we better use linear transformation
    // formulas (A&S 15.3) to map to different arguments
    if abs(z) >= 0.99 then
       s := float((z^(-a))*hypergeom([a,c-b],[c],-zin/z)):
       return(s):
    end_if:

    m := c - a - b:
    if not(type(float(m))=DOM_FLOAT) or not(iszero(sfrac(m))) then
        DIGITS := DIGITS + 3:
        c_a := c - a:
        c_b := c - b:
        // formula 15.3.6
        if typenegfloatint(c_a) or typenegfloatint(c_b) then
            s1 := float(0):
        else
            s1 := sgamma(m)/sgamma(c_a)/sgamma(c_b)*hypergeom([a,b],[1-m], z):
        end_if:
        if typenegfloatint(a) or typenegfloatint(b) then
            s2 := float(0):
        else
            s2 := z^m*sgamma(-m)/sgamma(a)/sgamma(b)*hypergeom([c_a,c_b],[1+m],z):
        end_if:
        s := sgamma(c)*(s1 + s2):
    else
        DIGITS := DIGITS + 5:
        m := strunc(m):
        m1 := abs(m):
        // condition: abs(1-z) < 1
        if m > 0 then // A&S 15.3.11
            a1 := a:
            b1 := b:
            a2 := a + m:
            b2 := b + m:
            z1 := 1:
            z2 := z^m:
        else // A&S 15.3.12
            a1 := a + m:
            b1 := b + m:
            a2 := a:
            b2 := b:
            z1 := z^m:
            z2 := 1:
        end_if:
        if iszero(m) or typenegfloatint(a2) or typenegfloatint(b2) then
            s1 := float(0):
        else
            t      := float(1):
            abs_t  := sabs(t):
            s1     := t:
            abs_s1 := sabs(s1):
            for k from 1 to m1-1 do
                t      := t*(a1+k-1)*(b1+k-1)/k/(k-m1)*z:
                abs_t  := sabs(t):
                s1     := s1 + t:
                abs_s1 := sabs(s1): 
                if abs_t < tol*abs_s1 then break: end_if:
            end_for:
            s1 := s1*sgamma(m1)/sgamma(a2)/sgamma(b2)*z1:
        end_if:
        if typenegfloatint(a1) or typenegfloatint(b1) then
            s2 := float(0):
        else
            // Note: a2 > a1 and b2 > b1 so these calls to psi() are safe
            old_s2 := float(0):
            t1 := float(1)/sgamma(m1+1):
            t2 := ln(z) - spsi(float(1))-spsi(float(1)+m1)+spsi(a2)+spsi(b2):
            s2 := t1*t2:
            df:=float(2):
            for k from 1 to RD_INF do 
                old_s2 := s2:
                r1 := 1/k:
                r2 := 1/(k+m1):
                t1 := t1*(a2+k-1)*(b2+k-1)*r1*r2*z:
                /* Walter, 5.1.09: Bug-Fix. Voldemort implemented the following line:
                   t2 := t2 - r1 - r2 + 1/(a2+k) + 1/(b2+k):
                   I corrected it to : 
                */
                t2 := t2 - r1 - r2 + 1/(a2+k-1) + 1/(b2+k-1):
                s2 := s2 + t1*t2:
                if sabs(s2-old_s2) < tol*max(tol, abs(s2)) then 
                   break: 
                end_if:
            end_for:
            s2 := (-1)^m1/sgamma(a1)/sgamma(b1)*z2*s2:
        end_if:
        // A&S 15.3.10/11/12
        s := sgamma(c)*(s1 - s2):
    end_if:
    s:
end_proc:

// Evaluates 1F1(z) via the reflection formula 7.6.6 of Y. Luke p.292
// of Mathematical Functions and their Approximations for large argument.
// This corresponds to the asymptotic series ... of A & S eq. 13.5.1.

confluent := proc(as, bs, z)
    local s,s1,s2,sc1,sc2,tol,flag,flag1,flag2;
    save DIGITS;
begin
  if Re(z) > 10.0^7 then
    // we are close to overflow of sexp(z).
    return([1, RD_NAN]);
  end_if;
  flag:=0:
  tol:=float(10^(-DIGITS)):
  sc1:=(sgamma(bs)/sgamma(as))*sexp(z)*(z^(as-bs)):
  flag1:=0:
  flag2:=0:
  if sabs(1-as) < tol or sabs(bs-as) < tol then
      s1:=1:
  else
      // see if we can get away with an asymptotic series
      [flag1, s1]:= ordinary([bs-as,1-as],[],1/z):
      if s1 = RD_NAN then
         return([flag1, RD_NAN]);
      end_if:
    // give hypergeom a chance to use something else (such as asymptotics)?
    //[flag1,s1]:= [1,hypergeom([bs-as,1-as],[],1/z)]:
  end_if:
  if (not(iszero(sfrac(bs-as))) or not(testtype(sround(bs-as),Type::NegInt)))
     and sabs(bs-as) > tol then 
    sc2:=sexp(signIm(z)*I*PI*as)*(sgamma(bs)/sgamma(bs-as))*(z^(-as)):
    if sabs(as) < tol or sabs(1+as-bs) < tol then
      s2:=1:
    else
      // see if we can get away with an asymptotic series
      [flag2,s2]:=ordinary([as,1+as-bs],[],-1/z):
      if s2 = RD_NAN then
         return([flag2, RD_NAN]);
      end_if:
    // give hypergeom a chance to use something else (such as asymptotics)?
    //[flag2,s2]:= [1, hypergeom([as,1+as-bs],[],-1/z)]:  
    end_if:
  else
    sc2:=float(0):
    s2:=float(0):
  end_if:
  s:=float(sc1*s1+sc2*s2):
  s:=complexfilter(as,bs,z,float(s)):
  if Re(z) > 0 then
    flag:=round(abs(sign(sc1))*flag1):
  elif Re(z) < 0 then
    flag:=round(abs(sign(sc2))*flag2):
  else
    flag:=flag1+flag2:
  end_if:
  [flag,s]:
end_proc:

asy_slater_cond:=proc(as,bs,z)
local crit,g,tol,B1,C1;
begin
tol:=float(10^(-DIGITS)):
if Re(z) <= float(0) and sabs(z) > float(1) then
        if as <> [] then
                B1:=_plus(op(as)):
        else
                B1:=0:
        end_if:
        if bs <> [] then 
                C1:=_plus(op(bs)):
        else
                C1:=0:
        end_if:
        g:=B1-C1:
        crit:=sabs(float(sexp(z)*z^g)):
        if crit < tol then
                TRUE:
        else
                FALSE:
        end_if: 
else
        FALSE:
end_if:
end_proc:

// Basic definition of the Lpq function (ressembles result
// of Slater theorem) as found in Luke 5.11.1 (7):
// an inversion formula for pFq(...z)  -> q+1Fp-1 (..+-1/z)

slater2:=proc(as,bs,p,q,z)
local flag,astar,sm,p1,p2,p3,p4,term,term2,t,phase,i;
save DIGITS;
begin
flag:=0:
sm:=0:
for t from 1 to p do
        astar:=subsop(as,t=null()):
        p1 := (astar[i]-as[t])$i=1..p-1:
        p2 := (bs[i]-as[t])$i=1..q:
        p3 := (1+as[t]-bs[i])$i=1..q:
        p4 := (1+as[t]-astar[i])$i=1..p-1:
        if nops([p1]) = 1 then p1:=sgamma(p1): 
        else p1:=_mult(op(map([p1],sgamma))): end_if:
        if nops([p2]) = 1 then p2:=sgamma(p2): 
        else p2:=_mult(op(map([p2],sgamma))): end_if:
        // Include a phase factor using signIm not mentioned
        // in 5.11.7 of Luke.  Use Walter's signIm function
        phase:=sexp(signIm(z)*float(PI)*I*as[t]):
        ([flag,term2]):=ordinary([as[t],p3],[p4],((-1)^(q-p-1))/z):
        if term2 = RD_NAN then
           return([flag, term2])
        end_if;
        term:=phase*((z^(-as[t]))*sgamma(as[t])*p1/p2)*term2:
        sm:=sm+term:
end_for:
[flag,float(sm)]:
end_proc:

Lpq:=proc(aso,bs,p,q,z)
local flag,as,s,old_s,eps,tol,try,i;
save DIGITS;
begin
flag:=0:
tol:=float(10^(-DIGITS)):
// In case Condition C is not met, we have to use the
// limit process and ensure guard digits.
if CondC(aso,p) and CondA(aso,bs,p,q) then
        ([flag,s]):=slater2(aso,bs,p,q,z):
else
        s:=0:
        old_s:=1:
        DIGITS:=p*DIGITS:
        eps := min(op(map([op(aso),op(bs)],sabs)))*tol:
        for try from 1 to 5 do
                DIGITS:=DIGITS+10: 
                old_s :=s:
                eps:=float(eps/100):    
                as := float([(aso[i] - i*eps)$i=1..p]):
                [flag,s]:=slater2(as,bs,p,q,z):
                if sabs(old_s-s) < 10*tol then break: end_if:
        end_for:
end_if:
[flag,float(s)]:
end_proc:

Ts:=proc(s,k,w,q)
local i,r,sm,t;
begin
        t:=((-1)^s)/sgamma(s+1): 
        sm:=t*(_mult((k+w[i])$i=0..q)):
        for r from 1 to s do
                t:=-t*(s-r+1)/r:
                sm:=sm+t*(_mult((r+k+w[i])$i=0..q)):
        end_for:
        sm:
end_proc:

Us:=proc(s,k,lam,p)
local i,r,sm,t;
begin
        t:=((-1)^s)/sgamma(s+1):
        sm:=t*(_mult((k+lam[i])$i=1..p)):
        for r from 1 to s do
                t:=-t*(s-r+1)/r:
                sm:=sm+t*(_mult((r+k+lam[i])$i=1..p)):
        end_for:
        sm:
end_proc:

// Definition for the c coefficients for the Kpq function is given
// by Luke in 5.11.5 of the "Special Functions and their Approximations"
// Vol. 1
cluke:=proc(k,w,lam,param)
local bet,p,q,sm1,sm2,s;
option remember;
begin
p:=param[1]: q:=param[2];
//param:=[p,q,bet,g]
if k < 0 then
        0:
// 2.11.29 of Luke reference
elif k = 0 then
        1: 
elif p <= 1 then
        bet:=param[3]:
        sm1:=0:
        for s from 1 to q do
                sm1:=sm1+Ts(q-s,s-k,w,q)*cluke(k-s,w,lam,param):
        end_for:
        sm1/(bet*k):
else 
        bet:=param[3]: 
        sm1:=0:
        for s from 1 to q do
                sm1:=sm1+Ts(q-s,s-k,w,q)*cluke(k-s,w,lam,param):
        end_for:
        sm2:=0:
        for s from 1 to p-1 do
                sm2:=sm2+Us(p-s-1,s-k,lam,p)*cluke(k-s,w,lam,param):
        end_for:
        (sm1-sm2)/(bet*k):
end_if:
end_proc:

// One of the asymptotic forms of the pFq functions for large
// argument (see section 5.11.2 if Luke)

Kpq:=proc(as,bs,p,q,zin)
local a,ar,b,i,j,k,l,m,bet,B1,C1,w,g,lam,sc,df,fac,old_s,s,t,t1,
                        abs_t,abs_s,param,phase,flag,tol,z;
save DIGITS;
begin
tol:=float(10^(-DIGITS)):
bet:=q+1-p:
flag:=0:
if bet > 0 then 
        DIGITS:=DIGITS+20:
   // for the standard confluent hypergeometric function p=1, q=1
   // use 5.11.3 (5) of Y.D. Luke Vol. 1.  This formula is also in
   // A & S.  eq. 13.5.1. (dominant term as z-> +infinity)
   if p=1 and q=1 then
        a  := as[1]: 
        b  := bs[1]: 
        t  := float(1):
        s  := t:
        df := float(2):
        z  := zin:
        sc:=sexp(z)*(z^(a-b)):
        for k from 1 to RD_INF do
            old_s:=s:
            t := -t*(a-k)*(b-a+k-1)/(z*k):
            abs_t := sabs(t):
            s := s + t:
            abs_s := sabs(s):
            if (abs_t < tol*abs_s or (sabs(s-old_s)> df and 
                        df < float(1/2))) and k>2 then break: end_if:

        end_for:
   else 
        B1:=float(_plus(as[l]$l=1..p)):
        C1:=float(_plus(bs[m]$m=1..q)):
        g:=float((((bet-1)/2)+B1-C1)/bet):
        w:=array(0..q): 

        w[0]:=float(bet*g+1-1):
        for i from 1 to q do
                w[i]:=float(bet*bs[i]-bet+bet*g):
        end_for:
        if p>1 then
                lam:=array(1..p):
                for j from 1 to p do
                        lam[j]:=float(bet*as[j]+bet*g):
                end_for:
        else
                lam:=array(1..1):
                lam[1]:=bet*g:
        end_if:
        param:=[p,q,float(bet)]:
        t1:=float(1):
        df := float(2):
        if Re(zin) > 0 or bet = 1 then
                // Use Luke 5.11.1 (18) and 5.11.5. (4)
                // and definitions therein
                z:=float(zin):
                ar:=float(z^(-1/bet)/bet):
                sc:=float((((2*PI)^((1-bet)/2))*(z^g)*
                                sexp(bet*(z^(1/bet)))/sqrt(bet))):
                s:=t1:
                for k from 1 to RD_INF do
                        old_s := s:
                        t1 := t1*ar:
                        t := float(cluke(k,w,lam,param)*t1):
                        abs_t := sabs(t):
                        s := s + t:
                        abs_s := sabs(s):
                        if k>2 then df:=min(df,sabs(s-old_s)): end_if:
                        if (abs_t < tol*abs_s or 
                         (sabs(s-old_s)> df and df < float(1/2)))
                                and k>2 then break: end_if:
                end_for:
        else
                // Use 5.11.1 (21)
                z:=float(-zin):
                ar:=float(z^(-1/bet)/bet):
                sc:=2*float((((2*PI)^((1-bet)/2))*(z^g)*
                        sexp(bet*(z^(1/bet))*scos(PI/bet))/sqrt(bet))):
                phase := float(PI*g+bet*(z^(1/bet))*ssin(PI/bet)):
                s:=float(scos(-phase)):
                for k from 1 to RD_INF do
                        old_s := s:
                        t1 := t1*ar:
                        fac := float(scos((PI*k/bet)-phase)):
                        t := float(fac*cluke(k,w,lam,param)*t1):
                        abs_t := sabs(t):
                        s := s + t:
                        abs_s := sabs(s):
                        if k>2 then df:=min(df,sabs(s-old_s)): end_if:
                        if (abs_t < tol*abs_s or
                         (sabs(s-old_s)> df and df < float(1/2)))
                                and k>2 then break: end_if:
                end_for:
        end_if:
    end_if:
    if abs_t > 10*tol*abs_s and 
                sabs(df) > 10*tol and sabs(sc*df) > 10*tol then
                flag:=1:
    end_if:
        return([flag,float(sc*s)]):
else
        return([0,float(0)]):   
end_if:
end_proc:

complexfilter:=proc(as,bs,z,res)
local tol,rz,iz;
begin
tol:=float(10^(1-DIGITS)):
rz:=Re(res):
iz:=Im(res):
if iszero(rz) or iszero(iz) then 
        return(res):
elif not(has(as,I)) and not(has(bs,I)) and not(has(z,I)) then
        if sabs(float(iz/rz)) < 10*tol then
                return(rz):
        else
                return(res):
        end_if:
else
        return(res):
end_if:
end_proc:

asympt:=proc(as,bs,p,q,z)
local flag1,flag2,flag,sc1,sc2,sc,d1,d2,p1,p2,eps,tol;
begin
tol:=float(10^(-DIGITS)):
flag1:=0:
flag2:=0:
sc1:=float(_mult(op(map(as,sgamma)))):
sc2:=float(_mult(op(map(bs,sgamma)))):
sc:=float(sc2/sc1):
if type(float(z))=DOM_FLOAT and z < 0 then
        DIGITS:=DIGITS+10:
        eps:=float(10^(2-DIGITS)):
        ([flag1,d1]):=Lpq(as,bs,p,q,z+eps*I):   
        ([flag2,d2]):=Lpq(as,bs,p,q,z-eps*I):   
        p1:=float((d1+d2)/2):
        flag1:=flag1+flag2:
        DIGITS:=DIGITS-10:
else
        ([flag1,p1]):=Lpq(as,bs,p,q,z):
end_if:
([flag2,p2]):=Kpq(as,bs,p,q,z):
if iszero(p2) then
        flag:=flag1:    
elif iszero(p1) then
        flag:=flag2:
elif sabs(p2/p1) < 10*tol then
        flag:=flag1:            
elif sabs(p1/p2) < 10*tol then
        flag:=flag2:    
else
        flag:=flag1+flag2:
end_if:
if flag=0 then
        return([flag,complexfilter(as,bs,z,float(sc*(p1+p2)))]):
else
        return([flag,float(sc*(p1+p2))]):
end_if:
end_proc:

contain:=proc(a,x)
local i;
begin
if has(a,x) then
        for i from 1 to nops(a) do
                if has(op(a,i), x) then
                        return(i):
                end_if:
        end_for:
else
        0:
end_if:
end_proc:

unalias(Hypergeom):
unalias(hypergeom):
unalias(slater):
unalias(slater2):
unalias(ordinary):
unalias(confluent):
unalias(Lpq):
unalias(Kpq):
unalias(cluke):
unalias(Us):
unalias(Ts):
unalias(contain):
unalias(f2F1_reflect):
unalias(flip):
unalias(typenegfloatint):
unalias(typenumcomplex):
unalias(sel):
unalias(complexfilter):
unalias(asympt):
unalias(asy_slater_cond):
unalias(CondC):
unalias(CondA):
unalias(spf1):
unalias(spf2):
unalias(sexp):
unalias(ssin):
unalias(scos):
unalias(sabs):
unalias(sgamma):
unalias(spsi):
unalias(strunc):
unalias(sround):
unalias(sfrac):
unalias(ln10):
unalias(fln):
unalias(Fln):
