//    hypergeom(a,b,z) : general hypergeometric function
//
//       hypergeom( [a ,a ,...a ], [b ,b ,...b ], z )  ==  F(a,d,z)
//                    1  2     p     1  2     q
//
//                        product( pochhammer(a[i],k), i=1..p ) z^k
// Defn : F(n,d,z) = sum( -----------------------------------------, k >= 0 )
//                        product( pochhammer(b[i],k), i=1..q ) k!
//
// where this sum converges, and its analytical continuation in the other
// cases.  In the divergent case, the function is regarded as the solution 
// of the generalized hypergeometric differential equation, which
// has the above series as asymptotic series at 0.
//
// Some elementary simplifications are handled automatically.
//
//
alias(Hypergeom=specfunc::Hypergeom):
alias(typenumcomplex=Hypergeom::typenumcomplex):
alias(sel=Hypergeom::sel):
alias(mysqrt=specfunc::Wurzelbehandlung):

// for symbolic manipulation as in intlib,
// identifiers are accepted for all arguments.
hypergeom:=
proc(l1 : Type::Union(Type::ListOf(Type::Arithmetical), DOM_IDENT),
     l2 : Type::Union(Type::ListOf(Type::Arithmetical), DOM_IDENT),
     z  : Type::Union(Type::Arithmetical, Type::Set))
local a,a0,b0,ak,bk,i,j,k,s,t;
begin
  if args(0) <> 3 then 
     error("wrong number of arguments")
  end_if:
  
  if not (testtype(l1, DOM_LIST) and
          testtype(l2, DOM_LIST)) then
    return(procname(args()));
  end_if;
  
  // Step 1: remove common terms
  if {op(l1)} intersect {op(l2)} <> {} then
    for i from nops(l1) downto 1 do
       j:=contains(l2, l1[i]):
       if j <> 0 then
         delete l1[i]:
         delete l2[j]:
       end_if:
    end_for:
  end_if:

  if testtype(z, Type::Set) and not testtype(z, Type::Arithmetical) then
    return(Dom::ImageSet(hypergeom(l1, l2, #z), #z, z));
  end_if;

  /*--------------------------------------------------
  // Voldemort, I do not understand the following.
  // I commented this out. Please, check.

  // Check that the function is defined.

  i:= select(l2, x -> _lazy_and(type(float(x))=DOM_FLOAT,
                                frac(x)=0,x<=0)):
  if nops(i) <> 0 then 
     i:= min(op(i)):
     j:= select(l1, x -> _lazy_and(type(float(x))=DOM_FLOAT,
                                   frac(x)=0, x<=0)):
     if nops(j) <> 0 and min(op(j)) < i then
        error("illegal parameters"):
     end_if:
  end_if:
  --------------------------------------------------*/
  
  // special cases, numbers refer to Abramowitz/Stegun
  if l1 = [] and l2 = [] then
    return(exp(z));
  end_if;
  if nops(l1)=2 and nops(l2)=1 then
    if z = 1 then // functions.wolfram.com
      for a in [l1, revert(l1)] do
        if is(a[1], Type::NegInt)=TRUE then
          return(pochhammer(l2[1]-a[2], -a[1])/
                 pochhammer(l2[1], -a[1]));
        end;
        if is(Re(l2[1]-a[1]-a[2])>0)=TRUE then
          return(gamma(l2[1])*gamma(l2[1]-a[1]-a[2])/
                (gamma(l2[1]-a[1])*gamma(l2[1]-a[2])));
        end;
      end_for;
    end;
    
    case [l1, l2]
      of [[1,1],[2]] do // 15.1.3
        return(-ln(1-z)/z);
      of [[1,1/2],[3/2]] do // 15.1.5
      of [[1/2,1],[3/2]] do // 15.1.5
        return(arctan(mysqrt(-z))/mysqrt(-z));
      of [[1/2, 1/2],[3/2]] do // 15.1.6
        return(arcsin(mysqrt(z))/mysqrt(z));
      of [[1,1],[3/2]] do // 15.1.6
        return(arcsin(mysqrt(z))/mysqrt(z)/sqrt(1-z));
    end;
    
    if abs(l1[1]-l1[2])=1/2 then
      if l2[1] = 1/2 then // 15.1.9
        // F(a,a+1/2;1/2;z^2) = ((1+z)^(-2*a)+(1-z)^(-2*a))/2
        if l1[1]-l1[2] = 1/2 then
          l1 := l1[2]
        else
          l1 := l1[1]
        end_if;
        if domtype(l1) = DOM_INT then
          return(normal(((1+mysqrt(z))^(-2*l1)+(1-mysqrt(z))^(-2*l1))/2));
        else
          return(((1+mysqrt(z))^(-2*l1)+(1-mysqrt(z))^(-2*l1))/2);
        end_if: 
      end_if;
      if l2[1] = 3/2 then // 15.1.10
        if l1[1]-l1[2] = 1/2 then
          l1 := l1[2]
        else
          l1 := l1[1]
        end_if;
        if domtype(l1) = DOM_INT then
           return(normal(
                  ((1+mysqrt(z))^(1-2*l1)
                  -(1-mysqrt(z))^(1-2*l1))
                  /(2*mysqrt(z)*(1-2*l1))));
        else
           return(((1+mysqrt(z))^(1-2*l1)
                  -(1-mysqrt(z))^(1-2*l1))
                  /(2*mysqrt(z)*(1-2*l1)));
        end_if:
      end_if;
    end_if;
    
    if l2[1] = 1/2 then
      if l1[1]=-l1[2] then // 15.1.11
        if domtype(l1[1]) = DOM_INT then
           return(normal(((sqrt(1-z)+mysqrt(-z))^(2*l1[1]) +
                          (sqrt(1-z)-mysqrt(-z))^(2*l1[1]))/2));
        else
           return(((sqrt(1-z)+mysqrt(-z))^(2*l1[1]) +
                   (sqrt(1-z)-mysqrt(-z))^(2*l1[1]))/2);
        end_if:
      end_if;
      if l1[2] = 1-l1[1] then // 15.1.12
        if domtype(l1[1]) = DOM_INT then
          return(normal(
                        sqrt(1-z)/2*(
                        (sqrt(1-z)+mysqrt(-z))^(2*l1[1]-1) +
                        (sqrt(1-z)-mysqrt(-z))^(2*l1[1]-1))));
        else
          return(sqrt(1-z)/2*(
                        (sqrt(1-z)+mysqrt(-z))^(2*l1[1]-1) +
                        (sqrt(1-z)-mysqrt(-z))^(2*l1[1]-1)
                 ));
        end_if:
      end_if;
    end_if;
    
    if abs(l1[1]-l1[2]) = 1/2 then
      if l1[1]-l1[2] = 1/2 then
        if l2[1]=1+2*l1[2] then // 15.1.13
          if domtype(l1[2]) = DOM_INT then
             return(normal(2^(2*l1[2])/(1+sqrt(1-z))^(2*l1[2])));
          else
             return(2^(2*l1[2])/(1+sqrt(1-z))^(2*l1[2]));
          end_if:
        elif l2[1]=2*l1[1]-1 then // 15.1.13 (2)
          if domtype(l1[1]) = DOM_INT then
            return(normal(2^(2*l1[1]-2)/(1+sqrt(1-z))^(2*l1[1]-2)/sqrt(1-z)));
          else
            return(2^(2*l1[1]-2)/(1+sqrt(1-z))^(2*l1[1]-2)/sqrt(1-z));
          end_if:
        elif l2[1]=2*l1[2] then //15.1.14
          if domtype(l1[2]) = DOM_INT then
             return(normal(2^(2*l1[2]-1)/sqrt(1-z)*(1+sqrt(1-z))^(1-2*l1[2])));
          else
             return(2^(2*l1[2]-1)/sqrt(1-z)*(1+sqrt(1-z))^(1-2*l1[2]));
          end_if:
        end_if;
      elif l1[2]-l1[1] = 1/2 then
        if l2[1]=1+2*l1[1] then // 15.1.13
          if domtype(l1[1]) = DOM_INT then
            return(normal(2^(2*l1[1])/(1+sqrt(1-z))^(2*l1[1])));
          else
            return(2^(2*l1[1])/(1+sqrt(1-z))^(2*l1[1]));
          end_if:
        elif l2[1]=2*l1[2]-1 then // 15.1.13 (2)
          if domtype(l1[2]) = DOM_INT then
            return(normal(2^(2*l1[2]-2)/(1+sqrt(1-z))^(2*l1[2]-2)/sqrt(1-z)));
          else
            return(2^(2*l1[2]-2)/(1+sqrt(1-z))^(2*l1[2]-2)/sqrt(1-z));
          end_if:
        elif l2[1]=2*l1[1] then //15.1.14
          if domtype(l1[1]) = DOM_INT then
            return(normal(2^(2*l1[1]-1)/sqrt(1-z)*(1+sqrt(1-z))^(1-2*l1[1])));
          else
            return(2^(2*l1[1]-1)/sqrt(1-z)*(1+sqrt(1-z))^(1-2*l1[1]));
          end_if;
        end_if;
      end_if;
    end_if;
  end_if;
  
  // ====================== 0F1 ============================
  // All 0F1 hypergeoms can be expressed in terms of
  // besselJ, besselI. The following special cases 
  // are the bessels with half integer indices. The
  // implemented set of case can be increased. However,
  // these are the simple case. Beyond this, the user
  // should call simplify(0F1([],[v], z)) to obtain
  // the bessels and, in case of halfinteger indices,
  // the trig(h) expressions:
  if nops(l1)=0 and nops(l2)=1 then // functions.wolfram.com
    case l2[1]
      of -5/2 do
        return((8/5*z+1)*cosh(2*mysqrt(z))
               -2/15*mysqrt(z)*(4*z+15)*sinh(2*mysqrt(z)));
      of -3/2 do
        return((4/3*z+1)*cosh(2*mysqrt(z))
               -2*mysqrt(z)*sinh(2*mysqrt(z)));
      of -1/2 do
        return(cosh(2*mysqrt(z))-2*mysqrt(z)*sinh(2*mysqrt(z)));
      of 1/2 do
        return(cosh(2*mysqrt(z)));
      of 3/2 do
        return(sinh(2*mysqrt(z))/(2*mysqrt(z)));
      of 5/2 do
        return(3/8/z*(2*cosh(2*mysqrt(z))-sinh(2*mysqrt(z))/mysqrt(z)));
      of 7/2 do
        return(15/32*z^(-2)*(
                  (4*z+3)*sinh(2*mysqrt(z))/mysqrt(z)
                    -6*   cosh(2*mysqrt(z))));
      of 9/2 do
        return(105/128*z^(-3)*(2*(4*z+15)*cosh(2*mysqrt(z))
                              -3*(8*z+ 5)*sinh(2*mysqrt(z))/mysqrt(z)));
    end_case;
  end_if;
  
  if nops(l1)=1 and nops(l2)=0 then // 15.1.8
    if iszero(1-z) and is(l1 < 0) = TRUE then
        return(0);
    end_if;
    return((1-z)^(-l1[1]));
  end_if;

  // Step 2: Sort l1 and l2 over i since F is symmetric in l1[i] and l2[i]
  if l1 <> [] then l1:= sort(l1): end_if:
  if l2 <> [] then l2:= sort(l2): end_if:

  // check possibilities of poles

  // sel(l) returns the largest nonpositive integer in
  // the list l. If there is none, sel(l) returns -infinity.
  a0:=sel(l1): 
  b0:=sel(l2):
  if b0 > a0 then
     error("illegal parameters"):
  end_if:

  if _and(op(map(l1,typenumcomplex)),
          op(map(l2,typenumcomplex)),
          typenumcomplex(z))
  and has([map((op(z),op(map(l1,op)),op(map(l2,op))),type)],DOM_FLOAT) then
      return(hypergeom::float(l1,l2,z)):
  end_if:

  // Simplification for special case
  if contains(l1, 0) <> 0 then return( 1 ) end_if:
  if iszero(z) then return(1) end_if:

  //-----------------------------------
  // Reduction to the polynomial case
  // Walter 15.2.02: changed the functionality.
  // Only reduce to the polynomial case, if a number
  // is produced. Otherwise, the polynomial may
  // be numerically unstable. The user should use
  // simplify to get the polynomial reduction.
  /* -----------------------------------------
  // Voldemort's original code:
  if a0 <> -infinity and a0 > b0 
     and 
    (testtype(z,Type::Rational) or 
     testtype(z,Type::Complex) or 
     testtype(z, Type::Indeterminate)) and
    _and(op(map(l1,testtype,Type::Rational)),
         op(map(l2,testtype,Type::Rational))) 
    then
    t:=1: s:=t:
    for k from 1 to RD_INF do
        ...
  ----------------------------------------- */
  // Walter's new code:
  if a0 <> -infinity and 
     a0 > b0 and
     (indets(l1) union
      indets(l2) union
      indets(z)
     ) minus Type::ConstantIdents = {} then
     // all indices and the argument z are numerical
     t:=1: s:=t:
     for k from 1 to round(-a0) do
       ak:=_mult(op(map(l1,((x,k)->x+k-1),k))):
       bk:=_mult(op(map(l2,((x,k)->x+k-1),k))):
       t:=t*ak*z/bk/k:
       s:=s+t:
       if iszero(t) then break: end_if:
     end_for:
     return(s): 
  end_if:

  //--------------------------
  // generic result
  return(procname(l1, l2, z)):

end_proc:

hypergeom := prog::remember(hypergeom, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("hypergeom")]):

hypergeom := funcenv(hypergeom):
hypergeom::type:= "hypergeom":
hypergeom::print:= "hypergeom":
hypergeom::info:=
"hypergeom(as,bs,z) -- Hypergeometric function with lists of parameters as,bs and argument z":

hypergeom::Content:=
proc(Out, data)
  local l1, l2;
begin
  if nops(data) <> 3 then
    return(Out::stdFunc(data));
  end_if;
  l1 := op(data, 1);
  l2 := op(data, 2);
  Out::Capply(Out::Chypergeom,
              Out(if nops(l1) = 0 then hold(``) else op(l1) end),
              Out(if nops(l2) = 0 then hold(``) else op(l2) end),
              Out(op(data, 3)),
              Out(nops(l1)),
              Out(nops(l2))):
end_proc:

hypergeom::float := loadproc(hypergeom::float, 
                             pathname("STDLIB","FLOAT"),"hypergeom"):

hypergeom::diff :=  loadproc(hypergeom::diff,
                             pathname("STDLIB","DIFF"),"hypergeom"):

hypergeom::series := loadproc(hypergeom::series,
                              pathname("SERIES"),"hypergeom"):

hypergeom::TeX :=
(h, ex, prio) ->
"F\\!\\left(\\left.\\begin{array}{c}".
generate::TeXoperator(", ", 0, output::Priority::INFINITE, op(op(ex,1))).
"\\\\\n".
generate::TeXoperator(", ", 0, output::Priority::INFINITE, op(op(ex,2))).
"\n\\end{array}\\right| ".
generate::tex(op(ex, 3), 0)."\\right)":

// according to functions.wolfram.com, the following describes the
// discontinuities of 2F1:
hypergeom::discont :=
proc(fn : "hypergeom", x : DOM_IDENT, F = null())
begin
  if nops(op(fn, 1)) = 2 and nops(op(fn, 2)) = 1 and
    not has([op(fn, 1), op(fn, 2)], x) and
    not testtype(op(fn, [1, 1]), Type::NegInt) and
    not testtype(op(fn, [1, 2]), Type::NegInt) and
    not testtype(op(fn, [2, 1]), Type::NegInt) then
    if args(0) > 2 and F = Real then
      return(solvelib::preImage(op(fn, 3), x, {1}) union
             discont(op(fn, 3), x, F));
    else
      return(solvelib::preImage(op(fn, 3), x, Dom::Interval([1], infinity)) union
             discont(op(fn, 3), x, F));
    end_if;
  end_if;
  FAIL;
end_proc:

autoload( specfunc::Hypergeom::simplify ):
stdlib::deferredAlias( hypergeom::simplify, specfunc::Hypergeom::simplify ):

unalias(typenumcomplex):
unalias(Hypergeom):
