/* ------------------------------------------
The incomplete beta function:

specfunc::ibeta(a, b, x) 
                = int(t^(a-1)*(1-t)^(b-1), t=0..x)/ beta(a, b)

Call(s):    specfunc::ibeta(a, b, x)

Parameters: x   - numerical real value from the interval [0, 1]
            a,b - numerical real values > 1

Return Value: a real floating point value from the interval [0, 1]

Details: ibeta is used to compute statistical distributions
         (beta distribution, F distribution, T distribution)
         In fact, ibeta(a, b, x) is a beta distribution,
         increasing monotonically from 0 to 1 as x increases
         from 0 to 1.

Warning: this is a utility function without interface and
         help page! It does not check its argument! So
         make sure that the calling function provides
         *numerical* real arguments a, b > 1 and *numerical*
         real x with 0 <= x <= 1.

Background: This algorithm is adapted from
            W.H. Press, S.A. Teukolski, 
            W.T. Vetterling, B.P. Flannery,
            "Numerical Recipes in C", 2nd Edition,
            Cambridge University Press, 1999
            Chapter 6.4, page 226-228

Examples:
>>  specfunc::ibeta(PI, 23/2, 0)

                              0.0

>>  specfunc::ibeta(PI, 23/2, exp(-2))

                         0.2454183345

>>  specfunc::ibeta(PI, 23/2, 1)

                              1.0

------------------------------------------ */

/* ------------------------------------------
This is the original library routine to
compute ibeta(a, b, x).

Background: This algorithm is adapted from
            W.H. Press, S.A. Teukolski, 
            W.T. Vetterling, B.P. Flannery,
            "Numerical Recipes in C", 2nd Edition,
            Cambridge University Press, 1999
            Chapter 6.4, page 226-228
------------------------------------------- */

specfunc::ibeta:=proc(a, b, x)
local fx, fa, fb, r, fw, boost, aa, e, f,
      del,h,qab,qam,qap,macheps,FPMIN,
      bt, s, s1, m2, m;
save DIGITS;
begin
     DIGITS:=DIGITS + 10:
     fx:=float(x):
     if iszero(fx) then return(float(0)) end_if;
     if iszero(x-1) then return(float(1)) end_if:
     fa:=float(a):
     fb:=float(b):
     r:=(fa+1)/(fa+fb+2):
     if fx>=float(r) then
        fw:= fx;
        if fx > 0.99 then
          // need to boost DIGITS to make sure that 1 - x is not
          // subject to cancellation:
          // determine boost such that 1 - x > 10^(-boost)
          boost:= ceil(-ln::float(1 - x)/2.302585); // ln(10) = 2.302585
          DIGITS:= DIGITS + boost;
          x:= 1-x:
          fx:= float(x);;
          DIGITS:= DIGITS - boost;
        else
          x:= 1 - x:
          fx:= float(x);;
        end_if;
        [a, b, fa, fb]:= [b, a, fb, fa]:
        s:=1:
        s1:=1:
     else
        if fx > 0.99 then
           boost:= ceil(-ln::float(1 - x)/2.302585); // ln(10) = 2.302585
           DIGITS:= DIGITS + boost;
           fw:= float(1 - x):
           DIGITS:= DIGITS - boost;
        else
           fw:= float(1 - x);
        end_if;
        s:=-1: 
        s1:=0:
     end_if:
     bt:=fx^fa*fw^fb / beta(fa,fb) / fa:
     
     macheps:=10.0^(2-DIGITS):
     FPMIN:=min(macheps^4,10.0^(-30));
     qab:=a+b:
     qap:=a+1:
     qam:=a-1:
     e:=1:
     f:=float(1 - qab*x/qap):
     if specfunc::abs(f)<FPMIN then f:=FPMIN end_if:
     f:=1/f:
     h:=f:
     m:=float(0);
     del:= float(0):

     b:= float(b):
     a:= float(a):
     qam:= float(qam):
     qab:= float(qab):
     qap:= float(qap):

     while specfunc::abs(del-1)>=macheps do
       m:=m+1:
       m2:=2*m:
       aa:=m*(b-m)*fx/((qam+m2)*(a+m2)):
       f:=float(1 + aa*f):
       if specfunc::abs(f)<FPMIN then f:=FPMIN end_if:
       e:=float(1 + aa/e);
       if specfunc::abs(e)<FPMIN then e:=FPMIN end_if:
       f:=1/f:
       h:=h*f*e:
       aa:=-(a+m)*(qab+m)*fx/((a+m2)*(qap+m2)):
       f:=float(1 + aa*f):
       if specfunc::abs(f)<FPMIN then f:=FPMIN end_if:
       e:=float(1 + aa/e):
       if specfunc::abs(e)<FPMIN then e:=FPMIN end_if:
       f:=1/f:
       del:=f*e:
       h:=h*del;
    end_while;
    return(s1 - s*bt*h);
end_proc:
