/* ------------------------------------------
inverf: The inverse of the erf function on the real line:
  y:= specfunc::inverf(x) is the solution of erf(y) = x.

Call(s):    inverf(x)

Parameters: x - an arithmetical expression

Return Value: for real floating point values x with
              -1 < x < 1 : a real floating point value
              for x =-1  :   -infinity
              for x =-1.0:    RD_NINF
              for x = 0  :    0
              for x = 0.0:    0.0
              for x = 1  :    infinity
              for x = 1.0:    RD_INF
              for x = RD_NAN: RD_NAN

Details: inverf is useful to compute alpha-quantiles of the
         statistical normal distribution

Remark: the float attribute calls the internal utitlity
        specfunc::inverf

*/

inverf:= proc(x)
begin
  if args(0) = 0 then error("expecting an argument")
  elif x::dom::inverf <> FAIL then return(x::dom::inverf(args()))
  elif args(0) <> 1 then error("1 argument expected");
  end_if;

  case type(x)
    of DOM_FLOAT do
      if -1 <= x and x <= 1 then
         return(specfunc::inverf(x));
      end_if;
      break;
    of DOM_SET do
    of "_union" do
      return(map(x, inverf))
    of "erf" do
      if is(op(x, 1) in R_) = TRUE then 
          return(op(x, 1));
      end_if;
      break;
    of "_mult" do
      if type(-x) = "erf" and // inverf(-erf(x)) = inverf(erf(-x))
         is(op(-x, 1) in R_) = TRUE then 
          return(-op(-x, 1));
      end_if;
      break;
    of "_plus" do 
      if type(1-x) = "erfc" and //inverf(1-erfc(x)) = inverf(erf(x))
         nops(1-x) = 1 and 
         is(op(1-x, 1) in R_) = TRUE then
          return(op(1-x, 1));
      end_if;
      if type(x+1) = "erfc" and //inverf(erfc(x)-1)=-inverf(1-erfc(x))
         nops(x+1) = 1 and      //=-inverf(erf(x))
         is(op(x+1, 1) in R_) = TRUE then
          return(-op(x+1, 1));
      end_if;
      break;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, inverf));
      else
        return(Dom::ImageSet(inverf(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  if stdlib::hasmsign(x) then
    return(-procname(-x));
  else
    return(procname(x));
  end_if;
end_proc:

inverf := prog::remember(inverf,
  () -> [property::depends(args()), DIGITS, slotAssignCounter("inverf")]):

inverf(0):= 0:
inverf(0.0):= float(0):
inverf(1):= infinity:
inverf(1.0):= RD_INF:
inverf(-1):=-infinity:
inverf(-1.0):= RD_NINF:
inverf(infinity*I):=infinity*I:
inverf(RD_INF*I):=RD_INF*I:
inverf(-infinity*I):=-infinity*I:
inverf(RD_NINF*I):=RD_NINF*I:
inverf(RD_NAN):= RD_NAN:

inverf:= funcenv(inverf):
inverf::print:= "inverf":
inverf::type:= "inverf":
inverf::info:= "inverf -- the inverse of the error function [try ?inverf for details]":

inverf::conjugate:= proc(x)
                    begin
                      x := conjugate(x);
                      if type(x) = "conjugate" then
                        hold(inverf)(x)
                      else
                        inverf(x)
                      end_if
                    end_proc:

inverf::diff:= proc(x)
               begin
                 sqrt(PI)/2*exp(inverf(op(x,1))^2)*diff(op(x,1),args(2..args(0)))
               end_proc:

inverf::float:= proc(x)
local fx;
begin
   fx:= float(x):
   if fx = RD_NAN then
      return(RD_NAN);
   end_if;
   if domtype(fx) = DOM_FLOAT and
      abs(fx) <= 1 then
         // pass the exact x to specfunc::inverf and
         // let it float x again (it may compute 1-x
         // excactly before floating this value):
         return(specfunc::inverf(x));
   end_if:
   return(hold(inverf)(fx)):
end_proc:

inverf::sign:=
  proc(x)
    local op1;
  begin
    op1 := sign(op(x, 1));
    if contains({-1, 0, 1, I, -I}, op1) then
      op1
    else
      hold(sign)(x)
    end_if
  end_proc:

inverf::series:= loadproc(inverf::series, pathname("SERIES"), "inverf"):

inverf::rectform:= loadproc(inverf::rectform, pathname("STDLIB", "RECTFORM"), "inverf"):

inverf::Re:= proc(x)
begin
  if is(x >= -1) = TRUE and is(x <= 1) = TRUE then
     return(hold(inverf)(x))
  elif is(x/I in R_) = TRUE then
     return(0)
  end_if;
  hold(Re)(inverf(x));
end_proc:

inverf::Im:= proc(x)
begin
  if is(x/I in R_) = TRUE then
     return(hold(inverf)(x)/I)
  elif is(-1 <= x) = TRUE and is(x <= 1) = TRUE then
     return(0)
  end_if;
  hold(Im)(inverf(x));
end_proc:

inverf::getprop:=proc(xpr)
  local prop;
begin
  prop := getprop(op(xpr));
  if _subset(prop, Dom::Interval(-1,1)) = TRUE then
      return(R_);
  end_if;
  C_;
end_proc:


/* ------------------------------------------
utility specfunc::inverf: the numerical inverse
   of erf on the real line:
   y:= specfunc::inverf(x) satisfies erf(y) = x.

Call(s):    specfunc::inverf(x)

Parameters: x - a numerical real value from the interval [-1, 1]

Return Value: for -1 < x < 1: a real floating point value
              for x =-1 or -1.0:  RD_NINF
              for x = 1 or  1.0:  RD_INF

Details: inverf is used to compute alpha-quantiles of the 
         statistical normal distribution

Warning: this is a utility function without interface and
         help page! It does not check its argument! So
         make sure that the calling function provides a
         *numerical* real x with 0 <= x <= 1.

Remark: works also for x > 1 and x < -1, if x is
        not too far away from 1 and -1, respectively.
        But there is no guarantee that the result makes sense!

Examples:
>>  specfunc::inverf(0)
                              0.0

>>  specfunc::inverf(PI/30 + sqrt(2)/3)

                         0.5654841632

>>  specfunc::inverf(1)
                            RD_INF
------------------------------------------ */

//--------------------------------------------------------------------
// Solve erf(y) = x.
// Some estimates for the solution close to x = 1:
//  For x = 1 - eps,  eps small:
//  x = erf(y) = 1 - 1/sqrt(PI) *exp(-y^2) * ( 1/y -1/2/y^3+3/4/y^5 + .. )
//  --> exp(y^2) = 1/sqrt(PI)/(1-x)* 1/y* (1 -1/2/y^2 + 3/4/y^4)
//  First approximations are given by:
//  y^2 = -ln(sqrt(PI)) -ln(1-x) - ln(y) + ln(1-1/2/y^2 +3/4/y^4 + .. )  
//--------------------------------------------------------------------

specfunc::inverf:= proc(x)
local fx, macheps, absx, sqrtPI, i, maxit, y, y0, dy;
save DIGITS;
begin
   if iszero(x-1) then 
      return(RD_INF);
   end_if;
   if iszero(x) then 
      return(float(0)) 
   end_if;
   if float(x)<0 then 
      return(-specfunc::inverf(-x)) 
   end_if;
   macheps:= float(10^(-DIGITS));
   DIGITS:= DIGITS+2;
   sqrtPI:= float(PI)^(1/2):
   fx:= float(x):
   absx:= specfunc::abs(fx):

   //---------------------------------------------------------------------
   if absx<0.7
   then // Find good starting point for Newton-Iteration, when |x| is small: 
        sqrtPI:= sqrtPI/2:
        x:= fx:
        y:=x*sqrtPI;
        y:=y*(1+y^2/3);
        if absx < 10*macheps then return(y) end_if;
        dy:= y;
        // do Newton search: 
        while specfunc::abs(dy) > macheps*specfunc::abs(y) do
          dy:= sqrtPI*(erf(y)-x)*exp(y^2);
          y:= y-dy;
        end_while;
        return(y);
   end_if;

   //---------------------------------------------------------------------
   // Now x >= 0.7.
   //---------------------------------------------------------------------

   //---------------------------------------------------------------------
   // Find good starting point for Newton-Iteration, when |x-1| is small: 
   // Solve erfc(y) = 1 - x rather than erf(y) = x, because erfc(y)
   // is better conditioned than erf(y) for large y:
   // 
   // By an asymptotic expansion of erf/erfc, we need to solve
   //   exp(-y^2)/y*(1 - 1/y^2 + ..)  = sqrt(PI)*(1 - x)
   // The solution of
   //   exp(-y^2)/y = sqrt(PI)*(1 - x)
   // is y = sqrt(lambertW(k, 2/PI/(1-x)^2))/sqrt(2)
   // However, lambertW produces Overflows! Do not use it!
   //---------------------------------------------------------------------
   x:= float(1-x):  // i.e., 0 <= x <= 0.3
   if iszero(x) then 
      return(RD_INF); 
   end_if;
   absx:= specfunc::abs(x):
   if absx>0.2 then // for 0.2 < x <= 0.3
        y0:= (-ln::float(sqrtPI) - ln::float(x))^(1/2); 
   elif absx>0.01 then // for 0.01 < x <= 0.2
        y0:= -ln::float(sqrtPI) - ln::float(x); 
        y0:= (y0 - ln::float(y0)/2)^(1/2);
   else // for 0.0 < x < 0.01
        y0:= -ln::float(sqrtPI) - ln::float(x); 
        y0:= (y0 - ln::float(y0)/2 +ln::float(1-1/2/y0))^(1/2);
   end_if;
   
   //---------------------------------------------------------------------
   // try direct Newton search first to avoid the overhead
   // of calling numeric::realroot:
   y:= y0:
   dy:= y;
   sqrtPI:= sqrtPI/2:
   maxit:= 10;
   for i from 1 to maxit do
       dy:= sqrtPI*(x-erfc::float(y))*exp::float(y^2);
        y:= y-dy;
       if specfunc::abs(dy) <= macheps*specfunc::abs(y) then
          return(y);
       end_if;
   end_for:
   //---------------------------------------------------------------------
   // The Newton iteration did not converge after maxit steps.
   // Finally, try sure-fire bisectioning/quadratic interpolation
   // implemented by numeric::realroot.
   // For 0.7 <= x < 1 the solution y of erf(y) = x satisfies
   //                   0.85*y0 < y < 1.1*y0,
   // where y0 is the previous starting point for the Newton iteration.
   // With this bounding interval, numeric::realroot *must* succeed:
   y:= genident():
   numeric::realroot(erfc(y) - x, y = 0.85*y0 .. 1.1*y0); 
end_proc:
