/* --------------------------------------------------------
inverfc: The inverse of the erfc function on the real line:
y:= specfunc::inverfc(x) is the solution of erfc(y, 0) = x.

Call(s):    inverfc(x)

Parameters: x - an arithmetical expression

Return Value: for real floating point values x with
              0 < x < 2 : a real floating point value
              for x = 2  :   -infinity
              for x = 2.0:    RD_NINF
              for x = 1  :    0
              for x = 1.0:    0.0
              for x = 0  :    infinity
              for x = 0.0:    RD_INF
              for x = RD_NAN: RD_NAN

Details: inverfc is useful to compute alpha-quantiles of the
         statistical normal distribution

         inverfc(x) = inverf(1 - x)
         inverfc(x) = -inverfc(2 - x)

Remark: the float attribute calls the internal utitlity
        specfunc::inverfc

*/

inverfc:= proc(x)
begin
  if args(0) = 0 then error("expecting an argument")
  elif x::dom::inverfc <> FAIL then return(x::dom::inverfc(args()))
  elif args(0) <> 1 then error("1 argument expected");
  end_if;

  case type(x)
    of DOM_FLOAT do
      if 0 <= x and x <= 2 then
         return(specfunc::inverfc(x));
      end_if;
      break;
    of DOM_SET do
    of "_union" do
      return(map(x, inverfc))
    of "erfc" do
      if nops(x) = 1 and // inverfc(erfc(x))
         is(op(x, 1) in R_) = TRUE then
          return(op(x, 1));
      end_if;
      break;
    of "_plus" do  // inverfc(2 - erfc(x)) = inverfc(erfc(-x)) 
      if type(2-x) = "erfc" and 
        nops(2-x) = 1 then
        if is(op(2-x, 1) in R_) = TRUE then
           return(-op(2-x, 1));
        end_if;
      end_if;
      if type(1-x) = "erf" then // inverfc(1-erf(x)) = inverfc(erfc(x))
        if is(op(1-x, 1) in R_) = TRUE then
           return(op(1-x, 1));
        end_if;
      end_if;
      if type(x-1) = "erf" then // inverfc(1+erf(y)) = inverfc(1-erf(-y))
                                // = inverfc(erfc(-y))
        if is(op(x-1, 1) in R_) = TRUE then
           return(-op(x-1, 1));
        end_if;
      end_if;
      break;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, inverfc));
      else
        return(Dom::ImageSet(inverfc(#x), #x, x));
      end_if;
    end_if;
    error("argument must be of 'Type::Arithmetical'")
  end_if;

  // Decide, which result we prefer: inverfc(x) or -inverfc(2-x).
  // Note that we have hasmsign(1-x) = not hasmsign(x - 1), so
  // the criterion hasmsign(1-x) is decisive:
  if stdlib::hasmsign(1-x) then
    return(procname(x));
  else
    return(-procname(2-x));
  end_if;
end_proc:

inverfc := prog::remember(inverfc,
  () -> [property::depends(args()), DIGITS, slotAssignCounter("inverfc")]):

inverfc(0):= infinity:
inverfc(0.0):= RD_INF:
inverfc(2):= -infinity:
inverfc(2.0):= RD_NINF:
inverfc(1):=0:
inverfc(1.0):= float(0):
inverfc(1 - infinity*I):= infinity*I:
inverfc(1 + RD_NINF*I):= RD_INF*I:
inverfc(1 + infinity*I):= -infinity*I:
inverfc(1 + RD_INF*I):= RD_NINF*I:
inverfc(RD_NAN):= RD_NAN:

inverfc:= funcenv(inverfc):
inverfc::print:= "inverfc":
inverfc::type:= "inverfc":
inverfc::info:= "inverfc -- the inverse of the complementary error function [try ?inverfc for details]":

inverfc::conjugate:= proc(x)
                     begin
                      x := conjugate(x);
                      if type(x) = "conjugate" then
                        hold(inverfc)(x)
                      else
                        inverfc(x)
                      end_if
                     end_proc:

inverfc::diff:= proc(x)
               begin
                 -sqrt(PI)/2*exp(inverfc(op(x,1))^2)*diff(op(x,1),args(2..args(0)))
               end_proc:

inverfc::float:= proc(x)
local fx;
begin
   fx:= float(x):
   if fx = RD_NAN then
      return(RD_NAN);
   end_if;
   if domtype(fx) = DOM_FLOAT and
      0 <= fx and fx <= 2 then
         // pass the exact x to specfunc::inverfc and
         // let it float x again (it may compute 2-x
         // excactly before floating this value):
         return(specfunc::inverfc(x));
   end_if:
   return(hold(inverfc)(fx)):
end_proc:

inverfc::sign:=
  proc(x)
    local op1;
  begin
    op1 := sign(1 - op(x, 1));
    if contains({-1, 0, 1, I, -I}, op1) then
      op1
    else
      hold(sign)(x)
    end_if
  end_proc:

inverfc::series:= loadproc(inverfc::series, pathname("SERIES"), "inverfc"):

inverfc::rectform:= loadproc(inverfc::rectform, pathname("STDLIB", "RECTFORM"), "inverfc"):

inverfc::Re:= proc(x)
begin
  if is(x >= 0) = TRUE and is(x <= 2) = TRUE then
     return(hold(inverfc)(x))
  elif is(x/I in R_) = TRUE then
     return(0)
  end_if;
  hold(Re)(inverfc(x));
end_proc:

inverfc::Im:= proc(x)
begin
  if is(x/I in R_) = TRUE then
     return(hold(inverfc)(x)/I)
  elif is(0 <= x) = TRUE and is(x <= 2) = TRUE then
     return(0)
  end_if;
  hold(Im)(inverfc(x));
end_proc:


inverfc::getprop:=proc(xpr)
  local prop;
begin
  prop := getprop(op(xpr));
  if _subset(prop, Dom::Interval(0,2)) = TRUE then
      return(R_);
  end_if;
  C_;
end_proc:


/* ------------------------------------------
utility specfunc::inverfc: the numerical inverse
   of erfc on the real line:
   y:= specfunc::inverfc(x) satisfies erfc(y) = x.

Call(s):    specfunc::inverfc(x)

Parameters: x - a numerical real value from the interval [0, 2]

Return Value: for 0 < x < 2: a real floating point value
              for x = 0 or 0.0:  RD_INF
              for x = 2 or 2.0:  RD_NINF

Warning: this is a utility function without interface and
         help page! It does not check its argument! So
         make sure that the calling function provides a
         *numerical* real x with 0 <= x <= 2.

Remark: works also for x < 0 and x > 2, if x is
        not too far away from 0 and 2, respectively.
        But there is no guarantee that the result makes sense!

Examples:
>>  specfunc::inverfc(1)
                              0.0

>>  specfunc::inverfc(PI/30 + sqrt(2)/3)

                         ?.??????????

>>  specfunc::inverfc(0)
                            RD_INF
------------------------------------------ */

//--------------------------------------------------------------------
// Solve erf(y) = x.
// Some estimates for the solution close to x = 1:
//  For x = 1 - eps,  eps small:
//  x = erf(y) = 1 - 1/sqrt(PI) *exp(-y^2) * ( 1/y -1/2/y^3+3/4/y^5 + .. )
//  --> exp(y^2) = 1/sqrt(PI)/(1-x)* 1/y* (1 -1/2/y^2 + 3/4/y^4)
//  First approximations are given by:
//  y^2 = -ln(sqrt(PI)) -ln(1-x) - ln(y) + ln(1-1/2/y^2 +3/4/y^4 + .. )  
//--------------------------------------------------------------------

specfunc::inverfc:= proc(x)
local macheps, abs1mx, absx, sqrtPI, i, maxit, y, y0, dy;
save DIGITS;
begin
   if iszero(x) then 
      return(RD_INF);
   end_if;
   if iszero(1-x) then 
      return(float(0)) 
   end_if;
   if iszero(2-x) then 
      return(RD_NINF) 
   end_if;
   if float(x) > 1 then 
      return(-specfunc::inverfc(2-x)) 
   end_if;
   macheps:= float(10^(-DIGITS));
   DIGITS:= DIGITS+2;
   sqrtPI:= float(PI)^(1/2):
   abs1mx:= specfunc::abs(float(1-x)):

   //-------------------------------------------------------------
   // Using inverfc(x) = inverf(1-x), the following implementation 
   // coincides with specfunc::inverf (with x replaced by 1-x).
   //-------------------------------------------------------------
   if abs1mx<0.7
   then // Find good starting point for Newton-Iteration, when |1-x| is small: 
        sqrtPI:= sqrtPI/2:
        x:= float(1 - x):
        y:=x*sqrtPI;
        y:=y*(1+y^2/3);
        if abs1mx < 10*macheps then return(y) end_if;
        dy:= y;
        // do Newton search: 
        while specfunc::abs(dy) > macheps*specfunc::abs(y) do
          dy:= sqrtPI*(erf(y)-x)*exp(y^2);
          y:= y-dy;
        end_while;
        return(y);
   end_if;

   //---------------------------------------------------------------------
   // Now 1-x >= 0.7, i.e., x <= 0.3
   //---------------------------------------------------------------------

   //---------------------------------------------------------------------
   // Solve erfc(y) = x. Find a good starting point for the Newton-Iteration, 
   // when x is small: 
   // By an asymptotic expansion of erf/erfc, we need to solve
   //   exp(-y^2)/y*(1 - 1/y^2 + ..)  = sqrt(PI)*x
   // The solution of
   //   exp(-y^2)/y = sqrt(PI)*x
   // is y = sqrt(lambertW(k, 2/PI/x^2))/sqrt(2)
   // However, lambertW produces Overflows! Do not use it!
   //---------------------------------------------------------------------
   x:= float(x): 
   if iszero(x) then 
      return(RD_INF); 
   end_if;
   absx:= specfunc::abs(x):
   if absx>0.2 then // for 0.2 < x <= 0.3
        y0:= (-ln::float(sqrtPI) - ln::float(x))^(1/2); 
   elif absx>0.01 then // for 0.01 < x <= 0.2
        y0:= -ln::float(sqrtPI) - ln::float(x); 
        y0:= (y0 - ln::float(y0)/2)^(1/2);
   else // for 0.0 < x < 0.01
        y0:= -ln::float(sqrtPI) - ln::float(x); 
        y0:= (y0 - ln::float(y0)/2 +ln::float(1-1/2/y0))^(1/2);
   end_if;
   
   //---------------------------------------------------------------------
   // try direct Newton search first to avoid the overhead
   // of calling numeric::realroot:
   y:= y0:
   dy:= y;
   sqrtPI:= sqrtPI/2:
   maxit:= 10;
   for i from 1 to maxit do
       dy:= sqrtPI*(x-erfc::float(y))*exp::float(y^2);
       y:= y-dy;
       if specfunc::abs(dy) <= macheps*specfunc::abs(y) then
          return(y);
       end_if;
   end_for:
   //---------------------------------------------------------------------
   // The Newton iteration did not converge after maxit steps.
   // Finally, try sure-fire bisectioning/quadratic interpolation
   // implemented by numeric::realroot.
   // For 0.7 <= x < 1 the solution y of erf(y) = x satisfies
   //                   0.85*y0 < y < 1.1*y0,
   // where y0 is the previous starting point for the Newton iteration.
   // With this bounding interval, numeric::realroot *must* succeed:
   y:= genident():
   numeric::realroot(erfc(y) - x, y = 0.85*y0 .. 1.1*y0); 
end_proc:
