/*
kummerU(a,b,z) is, with kummerM(a,b,z)=1F1(a,b,z), a solution of the
Kummer ODE : z*y''(z)+(b-z)*y'(z)-ay(z)=0

Some particular cases are implemented for particular values of a and b.

Reference : Abramowitz and Stegun (A&S), Chap. 13
*/


kummerU:=
proc(a, b, z)
  local fa, fb, fz, ra;
  //option remember;
begin
  if args(0) <> 3 then
    error("wrong number of arguments")
  elif z::dom::kummerU <> FAIL then
    return( z::dom::kummerU(args()) )
  end_if:

  if testargs() then
    if not (testtype(a, Type::Arithmetical) and
            testtype(b, Type::Arithmetical) and
            testtype(z, Type::Arithmetical)) then
      error("arguments must be of 'Type::Arithmetical'")
    end_if;
  end_if:

  fa:= domtype(float(a));
  fb:= domtype(float(b));
  fz:= domtype(float(z));
  if (fa=DOM_FLOAT or fa=DOM_COMPLEX) and 
     (fb=DOM_FLOAT or fb=DOM_COMPLEX) and 
     (fz=DOM_FLOAT or fz=DOM_COMPLEX) and 
     has([map((op(a),op(b),op(z)),type)],DOM_FLOAT) then
      return(kummerU::float(a,b,z))
  end_if:

  // if a=1 then kummerU(1,b,z)=z^(1-b)*kummerU(2-b,2-b,z) by A&S 13.1.29,
  // then use A&S 13.6.28 : kummerU(1-a,1-a,z)=exp(z)*igamma(a,z) :
  if iszero(a-1) then
    if iszero(z) then
      // The series expansion around z = 0 is 
      // kummerU(1, b, z) = z^(1-b)*gamma(b-1)
      if fb=DOM_COMPLEX or fb=DOM_FLOAT then
        fb:= float(1 - b);
        if Re(fb) > 0 and not iszero(fb) then 
           return(1/(1-b))
        else
           error("singularity");
        end_if;
      end_if:
    else
      return(z^(1-b)*exp(z)*igamma(b-1,z))
    end_if:
  end_if:

  // if a is a positive integer greater than 1 then we use recurrence
  // relation A&S 13.4.17 :
  ra:=round(a);
  if domtype(ra)=DOM_INT and ra > 1 and iszero(frac(a)) then
    return(1/(a-1)*expand(kummerU(a-1,b,z)-kummerU(a-1,b-1,z), ArithmeticOnly))
  end_if:

  // if a is zero or a negative integer, then kummerU can be expressed
  // in terms of the Laguerre polynomial (c.f. A&S 13.6.27) :
  ra:=round(-a);
  if domtype(ra)=DOM_INT and ra >= 0 and iszero(frac(a)) then
    return( (-1)^ra * ra! * laguerreL(ra, b-1, z)); 
  end_if:

  //same thing if a=b-1 using A&S 13.1.29 :
  ra:=round(-a+b-1);
  if domtype(ra)=DOM_INT and ra>=0 and iszero(frac(-a+b-1)) then
      if iszero(z) then
        if fb=DOM_COMPLEX or fb=DOM_FLOAT then
          fb:= float(1 - b);
          if Re(fb) > 0 then
             return(0)
          elif iszero(fb) then
             return(kummerU(-ra, 2-b, 0));
          else
             error("singularity");
          end_if;
        end_if:
      else
        return(z^(1-b)*kummerU(-ra,2-b,z))
      end_if:
  end_if:

  // if b=2a then kummerU can be expressed in terms of modified besselK function
  // using A&S 13.6.21 :
  if iszero(b-2*a) then
    if iszero(z) then
      if contains({DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX}, fa) then
        if Re(a) >= 1/2 then
          error("singularity")
        else // we have Re(a) < 1/2
          return(gamma(1/2 - a)/4^a/sqrt(PI));
        end_if;
      end_if;
    end_if:
    return(1/sqrt(PI)*exp(z/2)*z^(-a+1/2)*besselK(a-1/2,z/2));
  end_if:

  // if a=b then one uses A&S 13.6.28 :
  if iszero(a-b) then
    if iszero(z) then
      if fa=DOM_COMPLEX or fa=DOM_FLOAT then
        if Re(1-a) > 0 then
           return(gamma(1-a))
        else
           error("singularity");
        end_if;
      end_if:
    else
      return(exp(z)*igamma(1-a,z))
    end_if:
  end_if:

  procname(a, b, z)        
end_proc:

kummerU := prog::remember(kummerU, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("kummerU")]):


kummerU := funcenv(kummerU):
kummerU::type := "kummerU":
kummerU::print := "kummerU":
kummerU::info := "kummerU(a,b,z) -- confluent hypergeometric function,
solution of the Kummer differential equation.":


// ========================================================
// float attribute
// ========================================================
kummerU::float :=
proc()
local a, b, z, fa, fb, fz, ra, rb, s1, i, n, k, s2, C, S, U,
  expz, s2k, extra;
  save DIGITS;
begin
  // U(a,b,z) 
  //   = gamma(b-1)/gamma(a)*z^(1-b)*hypergeom([a-b+1],[2-b],z)
  //    +gamma(1-b)/gamma(a-b+1)*hypergeom([a],[b],z)          
  //   = pochhammer(a,-a+b-1)*z^(1-b)*hypergeom([a-b+1],[2-b],z)
  //    +pochhammer(a-b+1,-a)*hypergeom([a],[b],z)          

  a:=args(1); b:=args(2); z:=args(3);
  fa:=float(a); fb:=float(b); fz:=float(z);
  if not((domtype(fa)=DOM_FLOAT or domtype(fa)=DOM_COMPLEX) and
         (domtype(fb)=DOM_FLOAT or domtype(fb)=DOM_COMPLEX) and
         (domtype(fz)=DOM_FLOAT or domtype(fz)=DOM_COMPLEX)) then
    return(kummerU(fa,fb,fz))
  end_if:

  // ========================================================
  // Asymptotics: |z| large 
  // ========================================================
  //  if z is large (|z|>10, say) then one uses A&S 13.5.2.
  //  However, this formula seems to give wrong results in
  //  MuPAD when z is negative, so I (Olivier) added the 
  //  following testtype:
  if  abs(fz) >= 50 or
     (abs(fz) >= 10 and not(testtype(fz, Type::Negative)))
     then
      return(fz^(-fa)*hypergeom::float([a,1+a-b],[],-1/z));
  end_if:

  // ========================================================
  // Special case: a = 1 
  // ========================================================
  // if a=1 then kummerU(1,b,z)=z^(1-b)*kummerU(2-b,2-b,z) by A&S 13.1.29,
  // then use A&S 13.6.28 : kummerU(1-a,1-a,z)=exp(z)*igamma(a,z) :
  if iszero(a-1) then
    if iszero(fz) then
      fb:= float(1 - b);
      if Re(fb) > 0 and not iszero(fb) then 
         return(1/fb)
      else
         error("singularity");
      end_if;
    else
       return(fz^(float(1-b))*exp(fz)*igamma::float(b-1,z));
    end_if:
  end_if:

  // ========================================================
  // The 1st parameter 'a' is a positive integer
  // ========================================================
  ra:=round(a);
  if domtype(ra)=DOM_INT and ra >= 1 and iszero(frac(a)) then
    // Check if the exponential factor suffers from 
    // underflow or overflow:
    expz:= exp(fz);  
    if iszero(expz) or 
       has(expz, RD_INF) or
       has(expz, RD_NINF) or
       has(expz, RD_NAN) then
        return(expz)
    end_if;
    // ========================================================
    // The following recursion/iteration has complexity O(a^2).
    // Restrict it to 1 < a <= 10 to make sure that it returns
    // in acceptable time:
    // ========================================================
    if ra <= 10 and not iszero(fz) then
      DIGITS:= DIGITS + 10 + round(ra);
      // There are several recursions relating
      // U(a-1,b), U(a,b), U(a+1,b) or
      // U(a,b-1), U(a,b), U(a,b+1) or
      // U(a,b-1), U(a,b), U(a+1,b) or
      // U(a,b), U(a+1,b), U(a+1,b+1) etc.
      // E.g.: U(a, b, z) = (a+1-b)*U(a+1,b,z) + z*U(a+1,b+1,z);
      // Note that we know how to compute:
      // U(0,b,z) = 1, 
      // U(1,b,z) = z^(1-b)*exp(z)*igamma(b-1,z)
      // U(a,a,z) = exp(z)*igamma(1-a,z)
      // U(a,a+n,z) = Poly(n,a,z)*z^(1-a-n)
      // Here, we use:
      //   U(a, b, z) = 1/(a-1)*(U(a-1,b,z) - U(a-1,b-1,z));
      // Thus, we could recursively use:
      // return(float(1/(a-1)*(kummerU(a-1,b,z)-kummerU(a-1,b-1,z))))
      // However: Avoid the recursion, use an iteration to trace
      // U(a,b,z) to U(1,b),U(1,b-1),..,U(b-a+1)!
      U:= table():
      for i from 0 to ra-1 do
        // initialize: U[i] = U(1,b-i,z)*z^(b)*exp(-z)
        U[i]:= fz^(i+1)* igamma(b-i-1, fz); 
      end_for:
      for a from 2 to ra do
        for i from 0 to ra-a do
          U[i]:= U[i] - U[i+1]; 
        end_for;
      end_for;
      return(fz^(-fb)*exp(fz)*U[0]/float((ra-1)!));
    end_if; // of if ra < 10^4
  end_if:

  // ========================================================
  // The 1st parameter 'a' is a negative integer
  // ========================================================
  ra:=round(-a);
  if domtype(ra)=DOM_INT and ra >= 0 and iszero(frac(a)) then
    return((-1)^ra * gamma(float(1-a)) * laguerreL(ra, b-1, fz)); 
  end_if:

  // ========================================================
  // The value '1-b+a' is a negative integer
  // ========================================================
  ra:=round(-a+b-1);
  if domtype(ra)=DOM_INT and ra >= 0 and iszero(frac(-a+b-1)) then
    if iszero(z) then
      fb:= float(1 - b);
      if Re(fb) > 0 then
         return(float(0))
      elif iszero(fb) then
         return(kummerU::float(-ra, 2-b, float(0)));
      else
         error("singularity");
      end_if;
    else
      return(fz^(float(1-b))*kummerU(-ra, 2-b, fz));
    end_if:
  end_if:
    
  // ========================================================
  // The special case 'b = 2*a'  (--> besselK)
  // ========================================================
  if iszero(b-2*a) then
    if iszero(fz) then
      if Re(fa) >= 1/2 then
        error("singularity")
      else // we have Re(float(a)) < 1/2
        return(float(gamma(1/2 - a)/4^a/sqrt(PI)));
      end_if;
    else
      return(float(1/sqrt(PI)*exp(fz/2)*fz^(-a+1/2)*besselK(a-1/2,fz/2)));
    end_if:
  end_if:
  
  // ========================================================
  // The special case 'b = a'  (--> igamma)
  // ========================================================
  if iszero(a-b) then
    if iszero(fz) then
      fa:= float(1 - a);
      if Re(fa) > 0 then
         return(gamma(fa))
      else
         error("singularity");
      end_if;
    else
      return(exp(fz)*igamma(1-a,fz))
    end_if:
  end_if:

  // ===========================================================
  // if b is not an integer and 1+a-b is not a negative integer, 
  // use formula A&S 13.1.3 :
  // ===========================================================
  rb:=round(b); 
  ra:=round(1+a-b);
  if not(domtype(rb)=DOM_INT and iszero(frac(b))) and
     not(domtype(ra)=DOM_INT and ra < 0 and iszero(frac(1+a-b))) then
      if iszero(z) and float(Re(b)) > float(1) then
        error("singularity");
      else
        // We may have to add some extra digits since otherwise we may loose some
        // precision of the result. I (Olivier) hope 'extra' is sufficient .... 
        // This comes from the fact that, when z>0 and z->+infinity, then 1F1(a,b,z) 
        // has exp(z)z^(a-b) has a factor : c.f. A&S 13.5.1
        if not iszero(z) then // expression below is not always defined for z=0
           s1:= hypergeom::float([a],[b],z)/gamma::float(1+a-b)/gamma::float(b):
           s2:= fz^(1-b)*hypergeom::float([1+a-b],[2-b],z)/gamma::float(a)/gamma::float(2-b);
           if abs(s1 - s2) > abs(s1)*10^(-2) then
              // no cancellation, we loose at most 2 DIGITS!
              return(float(PI/sin(PI*b))*(s1 - s2));
           end_if;
           // Cancellation: boost DIGITS and compute again:
           if iszero(abs(s1 - s2)) = 0 then
              extra:= 1 + max(0, abs(Re(round(log(10, abs(exp(fz)*fz^(a-b)))))));
           else
              extra:= 1 + max(0, abs(round(log(10, abs(s1)/abs(s1- s2)))));
           end_if;
           DIGITS:=DIGITS+extra;
        end_if:
        return(float(PI/sin(PI*b)*(
                              hypergeom([a],[b],z)/gamma(1+a-b)/gamma(b)
                     -z^(1-b)*hypergeom([1+a-b],[2-b],z)/gamma(a)/gamma(2-b)
              )))
      end_if:
  end_if:

  // if b is zero or a negative integer then 2-b is a positive integer,
  // use A&S 13.1.29 :
  if iszero(b) or (domtype(rb)=DOM_INT and rb < 0) then
    if iszero(z) then
      if Re(1 - b) > 0 then
         if iszero(frac(a-b)) and b >= 1+a then
            return(float(0));
         else
            return(float(gamma(1 - b)/gamma(1 - b + a)));
         end_if;
      else
         error("singularity");
      end_if;
    else
      return(float(z^(1-b)*kummerU(a+1-b,2-b,z)))
    end_if:
  else //b is a positive integer, use A&S 13.1.6 :
    if iszero(z) then
      if iszero(frac(a)) and a < 0 then
         return(float( (-1)^a*pochhammer(b, round(-a)) ));
      else
         error("singularity");
      end_if;
    end_if:
    
    s1:=0; n:=rb-1;

    //  We have to add extra digits not to loose precision, and for that, we
    //  have that, when z>0 and z->infinity, the hypergeometric term in S
    //  beyond is a factor of exp(z)*z^(a-n+1) :
    if Re(z) > 0 then
      extra:=max(0, abs(Re(round(log(10, exp(fz)*fz^(float(a-n+1)))))))+1
    else
      extra:=max(0, abs(Re(round(log(10, exp(fz)*(-fz)^(-fa))))))+1
    end_if:
    DIGITS:=DIGITS+extra;
    
    if not(iszero(n)) then
      s1:=_plus(pochhammer(a-n,k)/pochhammer(1-n,k)/fact(k)*z^k $ k=0..n-1);
      s1:=float(s1*z^(-n)*fact(n-1)/gamma(a));
    end_if;

    C:=float((-1)^(n+1)/fact(n)/gamma(a-n));
    S:=float(C*(hypergeom([a],[n+1],z)*ln(z))+s1);
    s2:=0; k:=0;
    repeat
      s2k:=float(gamma(a+k)/gamma(a)*gamma(n+1)/gamma(n+k+1)/fact(k)*z^k*(
                 psi(a+k)-psi(1+k)-psi(1+n+k)));
      s2:=float(s2+C*s2k);
      S:=float(S+C*s2k);
      k:=k+1;
    until abs(float(s2k)) < 10^(-DIGITS) end_repeat:
    return(S)
    
  end_if:

end_proc:


// derivation of the kummerU function is given by several formulas
// A&S 13.4.21 ... 13.4.28. We use all these relations to simplify
// as much as possible the derivative of kummerU, in such a way that
// it is a solution of the differential equation.
kummerU::diff :=
proc(e, x)
local a, b, f, Select, Select2;
begin
  if has({op(e,1),op(e,2)}, [args(2..args(0))]) then hold(diff)(args())
  elif args(0)>2 then diff(diff(e,x),args(3..args(0)))
  else
    a:=op(e,1); b:=op(e,2); f:=op(e,3);

    Select:= X -> _lazy_or(_lazy_and(type(X) = "_plus",
           _lazy_and(select([op(X)], testtype, Type::NegRat) = [],
                   not(Select(select(_plus(op(X)), testtype, DOM_COMPLEX))))),
         testtype(Re(X), Type::PosRat) ):

    Select2:= X -> _lazy_or(_lazy_and(type(X) = "_plus",
           _lazy_and(select([op(X)], testtype, Type::PosRat) = [],
                   not(Select(select(_plus(op(X)), testtype, DOM_COMPLEX))))),
         testtype(Re(X), Type::NegRat) ):
    
    if Select(a) then 
      if Select(b) then // 13.4.27
        (1/f*(1-b+f)*kummerU(a,b,f)-1/f*kummerU(a-1,b-1,f))*diff(f,x)
      elif Select2(b) and not(iszero(a-b)) then
        (f*(a-b+f)/(b-a)*kummerU(a,b+1,f)+f/(a-b)*kummerU(a-1,b,f))*diff(f,x)
      else // 13.4.26
        (1/f*(a-b+f)*kummerU(a,b,f)-1/f*kummerU(a-1,b,f))*diff(f,x)
      end_if:
    elif Select2(a) then
      if Select(b) then
        (a*(1-b)/f*kummerU(a+1,b,f)-a/f*kummerU(a,b-1,f))*diff(f,x)
      elif Select2(b) then
        (-a*kummerU(a+1,b+1,f))*diff(f,x)
      else
        (-a/f*kummerU(a,b,f)-a/f*(b-a-1)*kummerU(a+1,b,f))*diff(f,x)
      end_if:
    elif Select(b) then // 13.4.24
        (-1/f*(1+a-b)*kummerU(a,b-1,f)+1/f*(1-b)*kummerU(a,b,f))*diff(f,x)
    elif Select2(b) then // 13.4.25
        (kummerU(a,b,f)-kummerU(a,b+1,f))*diff(f,x)
    else // 13.4.23
      (-a/f*kummerU(a,b,f)-a/f*(b-a-1)*kummerU(a+1,b,f))*diff(f,x)
    end_if:
  end_if:
end_proc:


kummerU::series :=
proc(a, b, f, z, o, dir, opt)
local fb, t, l, s, P, rb, ra, n, s1, s2, k;
begin
  fb:=float(b);

  t:=Series::series(f, z, o, dir, opt);
  if domtype(t) = Series::Puiseux then
    k:=Series::Puiseux::ldegree(t);
    if k = FAIL then
      k:=Series::Puiseux::order(t);
      if k > 0 then
        return(t)
      end_if;
      Series::error("order too small")
    end_if;

    if k < 0 then // expansion at infinity : A&S 13.5.2
      l:=limit(f,z,dir);
      s:=Series::series(-1/f, z, o, dir, opt);
      P:=series(hypergeom([a,1+a-b],[],z), z, o, dir, opt) @ s;
      return(P * Series::series(f^(-a), z, o, dir, opt));

    elif iszero(coeff(t,z,0)) and (type(fb)=DOM_FLOAT or type(fb)=DOM_COMPLEX)
    then  // expansion at 0 when b is not symbolic
      rb:=round(b); 
      ra:=round(1+a-b);
      if not(domtype(rb)=DOM_INT and iszero(frac(b))) and
         not(domtype(ra)=DOM_INT and ra < 0 and iszero(frac(1+a-b))) then
// A&S 13.1.3
           return(series(PI/sin(PI*b)*(hypergeom([a],[b],f)/gamma(1+a-b)/gamma(b)
                              -f^(1-b)*hypergeom([1+a-b],[2-b],f)/gamma(a)/gamma(2-b)),z,o))
      end_if:
    
      if iszero(b) or (domtype(rb)=DOM_INT and rb < 0) then //apply A&S 13.1.29
        return(series(f^(1-b)*kummerU(a+1-b,2-b,f),z,o))
      else // formula A&S 13.1.6
        b:=rb; s1:=0; n:=b-1;
        if not(iszero(n)) then
          s1:=_plus(pochhammer(a-n,k)/pochhammer(1-n,k)/fact(k)*f^k $k=0..n-1);
          s1:=s1*f^(-n)*fact(n-1)/gamma(a);
        end_if;
        s2:=_plus(pochhammer(a,k)/pochhammer(n+1,k)/fact(k)*f^k*(psi(a+k)
                  -psi(1+k)-psi(1+n+k)) $ k=0..o);
        return(series((-1)^(n+1)/fact(n)/gamma(a-n)*
                      (hypergeom([a],[n+1],f)*ln(f)+s2)+s1,z,o))
      end_if
    end_if:
  end_if:

  Series::unknown(kummerU(a,b,f), z, o, dir)
end_proc:
