/* ---------------------------------------------
laguerreL(n, a, x): the generalized Laguerre function

Calls:   laguerreL(n, x)
         laguerreL(n, a, x)

Parameters:  n, a, x - arithmetical expressions

Details: 
    *) The call laguerreL(n, x) is equivalent to
       laguerreL(n, 0, x).

    *) For non-negative integers n, this function is just 
       an interface to the (generalized) Laguerre polynomials 
       implemented by orthpoly::laguerre. 

       These polynomials are orthogonal with respect to
       the scalar product

       <f1, f2> = int(exp(-x)*x^a*f1(x)*f2(x), x = 0..infinity).

       In particular,

          <laguerreL(n,a,x), laguerreL(m,a,x)> =
                piecewise([n <> m,  0],
                          [n = m, gamma(a+n+1)/n!])

    *) The identity

        laguerreL(n,a,x) = 
             binomial(n+a,a)*hypergeom([-n], [a+1], x)
    
       is used to extend the definition to arbitrary 
       values of n and a. (This particular hypergeometric
       function is also known as Kummer's M function).

       Note, that the laguerreL function is not well defined 
       for all values of the parameters n and a, because
       certain restrictions on the parameters exist in the 
       definition of the hypergeometric functions. In most 
       of these cases, laguerreL(n, a, x) returns 0.

    *) The case 

           n = some negative integer, a = an integer

       with a < n is special. The reflection formula

          laguerreL(n, a, x) = 
              (-1)^a*exp(x)*laguerreL(-n-a-1, a, -x)

       is used to define these functions: they are
       of the form exp(x)*polynomial(x).

    *) If n is a non-negative integer then laguerreL(n, a, x)
       returns the n-th Laguerre polynomial explicitly for
       any value of the parameter a.

       If n is a negative integer, an explicit expression
       exp(x)*polynomial(x) is returned if a is an integer
       satisfying a < n. For all other numerical values of
       a, the function returns 0.

       If at least one of the arguments is a floating point
       number and all arguments can be converted to floating
       point numbers via <cmd>float</cmd>, then a floating
       point number is returned.

       In all other cases, a symbolc call of laguerreL is
       returned.

Author: W. Oevel
First implementation: September 2008
--------------------------------------------- */

laguerreL:= proc(n, a, x)
  local tn, ta, tx, y;
begin
  if args(0) < 2 then
     error("expecting at least 2 arguments");
  elif args(0) < 3 then
     // 2 argument version laguerreL(n, x) == laguerreL(n, 0, x)
     x:= a:
     a:= 0:
  elif args(0) > 3 then
     error("expecting no more than 3 arguments");
  end_if:
  if x::dom::laguerreL <> FAIL then
    return(x::dom::laguerreL(args()))
  end_if:

  if testargs() then
    if not testtype(n, Type::Arithmetical) then
      error("the first argument must be of 'Type::Arithmetical'")
    end_if:
    if not testtype(a, Type::Arithmetical) then
      error("the second argument must be of 'Type::Arithmetical'")
    end_if:
    if not testtype(x, Type::Arithmetical) then
      error("the last argument must be of 'Type::Arithmetical'")
    end_if:
  end_if:

  // Call orthpoly::laguerre if n is a non-negative integer
  if domtype(n) = DOM_INT and n >= 0 then
     x:= orthpoly::laguerre(n, a, x);
     if domtype(x) = DOM_POLY then
        return(expr(x)) 
     else
        return(x);
     end_if:
  end_if:
  if domtype(n) = DOM_FLOAT and iszero(frac(n)) and n >= 0 then
     x:= orthpoly::laguerre(round(n), a, x);
     if domtype(x) = DOM_POLY then
        return(float(expr(x))) 
     else
        return(float(x));
     end_if:
  end_if:

  // If there is a float in the arguments, try a float evaluation
  tn:= domtype(n):
  ta:= domtype(a):
  tx:= domtype(x):
  if (tn = DOM_FLOAT or (tn=DOM_COMPLEX and domtype(op(n, 1)) = DOM_FLOAT) ) or
     (ta = DOM_FLOAT or (ta=DOM_COMPLEX and domtype(op(a, 1)) = DOM_FLOAT) ) or
     (tx = DOM_FLOAT or (tx=DOM_COMPLEX and domtype(op(x, 1)) = DOM_FLOAT) ) then
     y:= laguerreL::float(n, a, x);
     if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(y)) then
        return(y);
     else
        // expect a symbolic return further down below.
        // Use floated arguments
        [n, a, x]:= float([n, a, x]);
     end_if:
  end_if:

  // If n is a negative integer, then laguerreL(n, a, x) = 0 for
  // all a and x with the exceptions a = ..., -1, 0, 1, 2, .., -n - 1:
  if (domtype(n) = DOM_INT or
      domtype(n) = DOM_FLOAT and iszero(frac(n))) and 
     n < 0 then
     if (domtype(a) = DOM_INT or
           (domtype(a) = DOM_FLOAT and iszero(frac(a)))) and 
        a < -n then
          return((-1)^a*exp(x)*laguerreL(-n-a-1, a, -x));
     elif testtype(float(a), Type::Numeric) then
        // and not iszero(frac(a))) or
        // testtype(float(a), DOM_COMPLEX) then 
        for y in [n, a, x] do
          if domtype(y) = DOM_FLOAT or 
            (domtype(y) = DOM_COMPLEX and domtype(op(y, 1)) = DOM_FLOAT)
            then
               return(float(0));
         end_if;
        end_for:
        return(0);
     end_if:
  end_if:

  if domtype(n) <> DOM_INT and
    (domtype(n) <> DOM_FLOAT or not iszero(frac(n))) and
    (domtype(n + a) = DOM_INT or
    (domtype(n + a) = DOM_FLOAT and iszero(frac(n+a)))) and
    n + a < 0 then
    // use the reflection formula
    // return((-1)^a*exp(x)*laguerreL(-n-a-1, a, -x));
    // or shall we return an error ??
    error("laguerreL is not defined for the parameters ".
          expr2text(n)." and ".expr2text(a));
  end_if:

  if iszero(x) then
     return(binomial(n + a, a)):
  end_if:

  if iszero(a) then
     return(procname(n, x));
  else
     return(procname(n, a, x));
  end_if:
end_proc:

laguerreL:= prog::remember(laguerreL,  
  () -> [property::depends(args()), DIGITS, slotAssignCounter("laguerreL")]):

laguerreL:= funcenv(laguerreL):
laguerreL::type:= "laguerreL":
laguerreL::print:= "laguerreL":
laguerreL::info:= "laguerreL(n, a, x) -- the generalized Laguerre function":

laguerreL::float:= proc(n, a, x)
  local an, a1, b, h;
begin
   if args(0) = 2 then
      x:= args(2); a := 0;
   end_if:
   an:= float(a + n):
   n:= float(n):
   a1:= float(a + 1);
   a:= float(a);
   if domtype(n) = DOM_FLOAT and 
      iszero(frac(n)) and
      n < 0 then
       if (domtype(a) = DOM_INT or
          (domtype(a) = DOM_FLOAT and iszero(frac(a)))) and 
          a < -n then
            return((-1)^a*exp(x)*laguerreL(-n-a-1, a, -x));
       elif testtype(a, Type::Numeric) then
         return(float(0))
       end_if:
   end_if:
   x:= float(x):
   if {domtype(n), domtype(a), domtype(x)} minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
      return(hold(laguerreL)(n, a, x));
   else
      b:= binomial(an, n):
      if iszero(b) or traperror((h:= hypergeom([-n], [a1], x))) <> 0 then
         return(float(0))
      else
         return(b*h);
      end_if:
   end_if:
end_proc:


laguerreL::diff:= proc(f, y)
  local n, a, x;
begin
  if nops(f) = 3 then 
     n:=op(f,1); a:=op(f,2); x:= op(f, 3);
  elif nops(f) = 2 then
     n:=op(f,1); a:= 0; x:=op(f,2); 
  else
     error("expecting 2 or 3 arguments in laguerreL"):
  end_if:
  
  if has(n, y) or has(a, y) then
     return(hold(diff)(args()))
  else 
    // The following formula does not hold for n = 0:
    // laguerreL(0, a, x) = 1, but laguerre(-1, a+1, x) <> 0
    // if a is a non-positive integer 
    return(-laguerreL(n-1, a+1, x)*diff(x,y)):
  end_if:
end_proc:


laguerreL::conjugate:= () ->  laguerreL(op(map([args()], conjugate))):


laguerreL::series:=
proc(n, a, f, x, o, dir, opt)
  local s;
begin
  if args(0) = 6 then
    [a, f, x, o, dir, opt]:= [0, a, f, x, o, dir]:
  end_if:
  if iszero(binomial(n + a, n)) then
     return(Series::series(0, x, o, dir));
  end_if:
  case limit(f, x = 0, if dir <> Undirected then dir else null() end_if) 
  of 0 do
  of infinity do
  of -infinity do
     s:= Series::series(binomial(n+a, n)*hypergeom([-n], [a+1], f), x, o, dir, opt);
     break;
  otherwise
     s:= Series::unknown(laguerreL(n, a, f), x, o, dir);
  end_case:
  return(s);
end_proc:

