/*++
ln -- the natural logarithm

ln(x)

x - expression
++*/

ln:=
proc(x)
  option noDebug;
  local a, k, z;
begin
  if args(0) = 0 then error("no arguments given")
  elif x::dom::ln <> FAIL then return(x::dom::ln(args()))
  elif args(0) <> 1 then error("wrong no of args") 
  end_if;

  if iszero(x) then error("singularity") end_if;

  case type(x)

    of DOM_FLOAT do
      return(ln::float(x))

    of DOM_COMPLEX do
      if domtype(op(x,1)) = DOM_FLOAT or
        domtype(op(x,2)) = DOM_FLOAT
        then
        return(ln::float(x))
      end_if;
      break
      
    of DOM_INTERVAL do
      return(ln::hull(x))

    of DOM_INT do
      if x < 0 then
        return(ln(-x) + I*PI)
      end_if;
      break

    of DOM_RAT do
      if x < 0 then return(ln(-x) + I*PI) end_if;
      if op(x,1)=1 then return(-ln(op(x,2))) end_if;
      break

    of DOM_SET do
    of "_union" do
      // ln is injective!
    of "_intersect" do
    of "_minus" do
      return(map(x, ln))
  end_case;

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(ln(#x), #x, x));
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  case type(x)
    of "exp" do
      if testtype(op(x), Type::Numeric) then
        k:=round(-Im(op(x))/2/PI);
        return(op(x)+2*k*PI*I)
      end_if;
      break

    of "_power" do
      if type(op(x, 1)) = "exp" then
        // ln(exp(a)^b) = ln(exp(b*ln(exp(a))))
        //              = ln(exp(b*(a + 2*k*PI*I)))
        // with k = round(-Im(a)/2/PI);
        // If a is real then
        //    ln(exp(a)^b) = ln(exp(a*b));
        // a = op(op(x, 1)), b = op(x, 2)
        a:= op(op(x, 1));
        if testtype(a, Type::Numeric) then
          // recursive call to ln that will
          // end up in the branch above
          return(ln(exp(a * op(x,2))));
        end_if;
      end_if;
      break

    of "lambertW" do
      // ln(lambertW(k, z)) = ln(z) - lambertW(k, z)
      //     ... if k = 1 and z in [-1/e, 0)
      // ln(lambertW(k, z)) = ln(z) + 2*PI*I*k - lambertW(k, z)
      //     ... otherwise
      if nops(x) = 2 then
        k := op(x, 1);
        z := op(x, 2);
      else
        k := 0;
        z := op(x, 1);
      end_if;
    
      return(piecewise
             (
              [k = -1 and z >= -exp(-1) and z < 0, ln(z)-x],
              [k <> -1 or not z >= -exp(-1) or not z < 0, ln(z)+2*PI*I*k-x])
             );
  end_case;

  procname(x)
end_proc:

ln := prog::remember(ln, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("ln")]):

ln(1):= 0:
ln(-1):= I*PI:
ln(I):= I*PI/2:
ln(-I):=-I*PI/2:

ln:= funcenv(ln, op(specfunc::ln, 2)):
ln::print := "ln":
ln::type := "ln":
ln::info := "ln -- the natural logarithm":
ln::float := specfunc::ln:
ln::hull := DOM_INTERVAL::ln@hull:

ln::inverse := "exp":

ln::undefined := {0}:
ln::realDiscont := {0}:
ln::complexDiscont :=
    loadproc(ln::complexDiscont, pathname("STDLIB","DISCONT"), "ln"):

ln::expand :=
    loadproc(ln::expand, pathname("STDLIB","EXPAND"), "ln"):

ln::simplify :=
    loadproc(ln::simplify, pathname("STDLIB","SIMPLIFY"), "ln"):

ln::sign :=
    loadproc(ln::sign, pathname("SPECFUNC","SIGN"), "ln"):

ln::rectform :=
    loadproc(ln::rectform, pathname("STDLIB","RECTFORM"), "ln"):

ln::series := loadproc(ln::series, pathname("SERIES"), "ln"):

ln::Re := loadproc(ln::Re, pathname("STDLIB","RE"), "ln"):
ln::Im := loadproc(ln::Im, pathname("STDLIB","IM"), "ln"):

ln::conjugate:= loadproc(ln::conjugate, pathname("STDLIB","CONJ"), "ln"):

ln::diff := loadproc(ln::diff, pathname("STDLIB","DIFF"), "ln"):

ln::"transform::laplace":=
    loadproc( ln::"transform::laplace",
        pathname("TRANS","LAPLACE"), "L_ln"):

ln::"transform::invlaplace":=
    loadproc( ln::"transform::invlaplace",
        pathname("TRANS","LAPLACE"), "IL_ln"):

ln::Content := stdlib::genOutFunc("Cln", 1):

// end of file 
