/*++
The logarithm of the gamma function

lngamma(x)

x - an expression

Details: 
  Along the positive real half-line one has
    lngamma(x) = ln(gamma(x)).
  Throughout the complex plain, however, one has:
    lngamma(z) = ln(gamma(z)) + f(z)*2*PI*I
  with some integer valued function f(z).

  We do have: exp(lngamma(z)) = gamma(z).
++*/

lngamma :=
proc(x)
  local gx, fx;
  option noDebug;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif x::dom::lngamma <> FAIL then
    return(x::dom::lngamma(args()))
  elif args(0) <> 1 then
    error("1 argument expected");
  end_if;

  case type(x)
    of DOM_SET do
    of "_union" do
      return(map(x, lngamma))
  end_case;

  if domtype(x) = DOM_FLOAT or
    (domtype(x) = DOM_COMPLEX and 
     domtype(op(x, 1)) = DOM_FLOAT) then
     return(lngamma::float(x));
  end_if:

  if not testtype(x,Type::Arithmetical) then
    /* generic handling of sets */
    if testtype(x, Type::Set) then
      if type(x)=Dom::ImageSet then
        return(map(x, lngamma));
      else
        return(Dom::ImageSet(lngamma(#x), #x, x));
      end_if;
    end_if;

    error("argument must be of 'Type::Arithmetical'")
  end_if;

  if domtype(x) = DOM_INT and
     x <= 0 then
     error("singularity");
  end_if:

  if type((gx:= gamma(x))) <> "gamma" then
     fx:= float(x);
     if type(fx) = DOM_FLOAT and fx >= 0 then
           return(ln(gx));
     end_if:
  end_if:

  procname(x)
end_proc:

lngamma := prog::remember(lngamma, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("lngamma")]):

/*
lngamma(-5/2) := ln(8*PI^(1/2)/15) - 3*PI*I:
lngamma(-3/2) := ln(4*PI^(1/2)/3) - 2*PI*I:
lngamma(-1/2) := ln(2*PI^(1/2)) - PI*I:
*/

lngamma(1/2) := ln(PI^(1/2)):
lngamma(1) := 0:
lngamma(3/2) := ln(PI^(1/2)/2):
lngamma(2) := 0:
lngamma(5/2) := ln(3*PI^(1/2)/2):
lngamma(3) := 2:
lngamma(infinity) := infinity:

lngamma:= funcenv(lngamma):
lngamma::print := "lngamma":
lngamma::info  := "lngamma -- the logarithm of the gamma function":
lngamma::type  := "lngamma":

lngamma::float := proc(x)
local fx, Refx, fPI, tmp, r, y, absy, k,
      originalDIGITS, extraDIGITS;
save DIGITS;
begin
 fx:= float(x);
 if domtype(fx) = DOM_FLOAT or
   (domtype(fx) = DOM_COMPLEX and
    domtype(op(fx, 1)) = DOM_FLOAT) then
   fx:= numeric::complexRound(fx);
   //=================================================
   // Special case: take care of the logarithmic
   // singularity at the origin via the Taylor series
   //=================================================
   if specfunc::abs(fx) <= 10.0^(-DIGITS/2) then
       fPI:= float(PI);
       return(-ln::float(fx) 
              -float(EULER)*fx
              +fPI^2/12*fx^2);
   end_if:
   //=================================================
   // Special case: use the asymptotic expansion
   // lngamma(x) = (x-1/2)*ln(x) - x + ln(2*PI)/2 
   //    + sum(bernoulli(2*k)/(2*k)/(2*k -1)/x^(2*k-1), k = 1 .. infinity);
   // for very large x
    // It's asymptotic, it does not converge for fixed x!
   //=================================================
   if specfunc::abs(fx) > 4*10^7 and
      not (domtype(fx) = DOM_FLOAT and fx < 0)
       // the asymptotic expansion is not valid along the
       // negative real semix axis. Rely on the reflection
       // rule way down below for this case. 
      then
      r:= ln::float(2*PI)/2 +  float(x-1/2)*ln::float(x) - fx;
      k:= 0;
      repeat
        k:= k+2;
        tmp:= bernoulli(k)/(k*(k-1)*x^(k-1));
        r := r + tmp;
        if specfunc::abs(tmp) <= 10^(-DIGITS)*specfunc::abs(r) then
           return(r);
        end_if:
      until k = 2000 end_repeat;
      // With k = 1950 we can compute lngamma(5*10^7) up to 
      // DIGITS = 11000. We assume that this the limit of
      // the precision a user might be interested in.
      if k >= 2000 then
         warning("loss of precision");
      end_if:
      return(r);
   end_if:
   //=================================================
   // Special case: along the positive real semi axis,
   // we have lngamma(x) = ln(gamma(x)). Use the float
   // attributes of gamma and ln implemented in the
   // MuPAD kernel (this is faster than any library
   // code implemented here)
   //=================================================
   if domtype(fx) = DOM_FLOAT and 
       fx > 0 and
       fx < 4*10^7 then
       return(ln::float(gamma::float(x)));
   end_if;
   //=================================================
   // The code below works well only for rather small
   // precision goals (such as DIGITS = 10). For high
   // DIGITS, compute the integer k such that
   // lngamma(x) = ln(gamma(x)) + k * 2*PI*I
   // with small DIGITS and then use it in conjunction
   // with a precise ln(gamma(x)):
   //=================================================
   if DIGITS > 10 then
      originalDIGITS:= DIGITS;
      DIGITS:= 10;
      k:= (lngamma::float(x) - ln::float(gamma::float(x)))/2/float(PI)/I;
      k:= round(k):
      DIGITS:= originalDIGITS + 7;
      return(ln::float(gamma::float(x)) + 2*k*float(PI)*I);
   end_if:
   //=================================================
   // This is the part where the work is done: It works
   // well for small DIGITS (10, say):
   //=================================================
   Refx:= Re(fx);
   if Refx >= 1/2 then
       //--------------------------------------------------
       // Evaluate lngamma(x) in the right half plane via
       // the _float method:
       //--------------------------------------------------
       r:= lngamma::_float(x);
       return(r);
   elif Re(fx) > -1/2 then
       //-----------------------------------------------------------
       // In the vicinity of the imaginary axis, use the functional
       // equation 
       //      lngamma(x) = lngamma(x + 1) - ln(x)
       //  to move the argument of lngamma into the right half plane
       //-----------------------------------------------------------
       extraDIGITS:= 0: // boost DIGITS by extraDIGITS until we
       while TRUE do    // do not have to fear cancellation effects
          tmp:= lngamma::float(x + 1);
          r:= ln::float(x);
          y:= tmp - r;
          absy:= specfunc::abs(y);
          if absy > 10^(-1 - extraDIGITS)*specfunc::abs(r) then
            return(y);
          else
            extraDIGITS:= extraDIGITS + min(5, ceil(-ln(absy)/2.302585093))
          end_if:
       end_while;
   else
       if domtype(fx) = DOM_FLOAT and
          iszero(frac(fx)) and 
          fx <=0 then
          error("singularity");
       end_if:
       fPI:= float(PI);
       tmp:= float((2*Re(x) + 1)/4):
       // the following expression r is
       // the generic reflection rule 
       // lngamma(x) = -lngamma(-x) + PI - ln(-x)
       //              -ln(sin(PI*x)) + ...
       r:= - lngamma::float(-fx) 
           + ln::float(fPI) 
           - ln::float(-fx) 
           - ln::float(numeric::complexRound(sin(fPI*fx))) 
           + 2*fPI*I*floor(tmp)*signIm(fx):
       end_if:
       // For some special cases r is not correct and
       // a term has to be added:
       if domtype(tmp) = DOM_FLOAT and
          iszero(frac(tmp)) then
            // tmp is an integer
            r:= r - fPI*I*(signIm(fx) - 1):
       elif iszero(Im(fx)) and trunc(2*fx) mod 4 = 3 then
            r:= r + 2*fPI*I:
       end_if:
       return(r);
   end_if:
   return(hold(lngamma)(x));
end_proc:


// _float is a utility called by float:
lngamma::_float :=
  proc(x)
    local s, b, ss, k, term;
  begin
    //=======================================================
    // _float implements the asymptotic expansion 
    // lngamma(x) = (x-1/2)*ln(x) - x + ln(2*PI)/2 
    //    + sum(bernoulli(2*k)/(2*k)/(2*k -1)/x^(2*k-1), k = 1 .. infinity);
    // It's asymptotic, it does not converge for fixed x!
    //=======================================================
    if Re(x) < 5 then
      // use the functional equation to shift x 
      // further to the right in the complex plane.
      s:= -ln::float(x  )
          -ln::float(x+1)
          -ln::float(x+2)
          -ln::float(x+3)
          -ln::float(x+4)
          -ln::float(x+5);
      x:= x + 6;
    else
      s:= 0:
    end_if;

    // Here, the desired result is lngamma(original_x) = s + lngamma(x).
    // The present x satisfies Re(x) >= 5.

    s:= s + ln::float(2*PI)/2;
    b:= 1/156; // = bernoulli(k)/(k*(k-1)) with k = 14 
    repeat
      ss := s +  (x-1/2)*ln::float(x) - x;
      if specfunc::abs(b/x^13) < 10^(-DIGITS)*specfunc::abs(ss) then
         s:= ss;
         break;
      end_if;
      s:= s - ln::float(x);
      x:= x + 1
    until FALSE end_repeat;

    // The desired result is lngamma(original_x) = s + lngamma(x).
    // We have shifted x such that 8 terms of the asymptotic
    // series should suffice to reach the desired precision
   
    k:= 0;
    repeat
      k:= k+2;
      term := bernoulli(k)/(k*(k-1)*x^(k-1));
      s := s + term;
      if specfunc::abs(term) <= 10^(-DIGITS)*specfunc::abs(s) then
         return(s);
      end_if:
    until k = 16 end_repeat;
    if k >= 16 then
      // Walter, Febr. 08: Do not issue this warning:
      // warning("loss of precision");
    end_if:
    return(s);
  end_proc:



lngamma::hull  := DOM_INTERVAL::ln_gamma@hull:

lngamma::complexDiscont :=
   loadproc(lngamma::complexDiscont, pathname("STDLIB", "DISCONT"), "lngamma"):
lngamma::realDiscont:= 
   loadproc(lngamma::realDiscont,    pathname("STDLIB", "DISCONT"), "lngamma"):
lngamma::undefined := 
   loadproc(lngamma::undefined,      pathname("STDLIB", "DISCONT"), "lngamma"):

lngamma::diff := proc()
                 local x;
               begin 
                 x := op(args(1),1);
                 diff(x, args(2..args(0))) * psi(x)
               end_proc:

lngamma::conjugate :=
    loadproc(lngamma::conjugate, pathname("STDLIB","CONJ"),     "lngamma"):

lngamma::expand :=
    loadproc(lngamma::expand,    pathname("STDLIB","EXPAND"),   "lngamma"):

lngamma::rectform :=
    loadproc(lngamma::rectform,  pathname("STDLIB","RECTFORM"), "lngamma"):

lngamma::Re:= (x) -> if sign(x) = 1 then
                        return(hold(lngamma)(x))
                     else
                        return(hold(Re)(hold(lngamma)(x)));
                     end_if:
lngamma::Im:= (x) -> if sign(x) = 1 then
                        return(0)
                     else
                        return(hold(Im)(hold(lngamma)(x)));
                     end_if:

lngamma::series :=
    loadproc(lngamma::series,    pathname("SERIES"),            "lngamma"):

lngamma::Content := stdlib::genOutFunc("Clngamma", 1):

lngamma::TeX := (g, data, prio) -> "\\ln\\!\\Gamma\\!\\left(".
				generate::tex(op(data), output::Priority::Noop).
				"\\right)":
