/*++
log -- the general logarithm

log(b, x)

b - a real number (b > 0, b <> 1) or an identifier
x - an expression
++*/

log:=
proc(b, x)
  option noDebug;
  local k, dk, bk, mmin, mmax;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif args(0) <> 2 then
    error("wrong no of args") 
  elif x::dom::log <> FAIL then
    return(x::dom::log(args()))
  end_if;

// Computations with symbols should be allowed!

  if b <> exp(1) and
     testtype(b, Type::Constant) and numeric::isless(0, b) = TRUE
  then
     if not (b > 0 and b <> 1) then
        error("the base must be greater than zero and not equal one")
     end_if;
     if domtype(b) = DOM_FLOAT then 
        x:= float(x):
     end_if:
  elif b = exp(1) then
     return(ln(x))
  elif domtype(b) <> DOM_IDENT and type(b) <> "_index" then
    if testtype(b, Type::Set) then
      if testtype(x, Type::Set) and not testtype(x, Type::Arithmetical) then
        return(Dom::ImageSet(log(#b, #x), [#b, #x], [b, x]));
      else
        return(Dom::ImageSet(log(#b, x), [#b], [b]));
      end_if;
    end_if;

     error("the base must be an identifier, ".
           "an indexed identifier, ".
           "or a number of type Type::Positive")
  end_if;

  if iszero(x) then
    error("singularity")
  end_if;

  case type(x)
    of DOM_SET do
      return({log(b, k) $ k in x})
    of "_union" do
      return(_union(log(b, k) $ k in x))

    of DOM_FLOAT do
      if domtype(b) <> DOM_IDENT and type(b) <> "_index" then
        return(log::float(b, x))
      end_if;
      break;

    of DOM_COMPLEX do
      if domtype(b) <> DOM_IDENT and type(b) <> "_index" and
        (domtype(op(x,1)) = DOM_FLOAT or
         domtype(op(x,2)) = DOM_FLOAT)
        then
        return(log::float(b, x))
      elif x = I or x = -I then
        // log(b, I)  =  I*PI/2/ln(b)
        // log(b, -I) = -I*PI/2/ln(b)
        return(x*PI/2/ln(b))
      end_if;
      break

    of DOM_INT do
    of DOM_RAT do
      if x < 0 then return(log(b, -x) + I*PI/ln(b)) end_if;
      if x = 1 then return(0) end_if; // log(b, 1) = 0
      // log(b, 1/n) = -log(b, n)
      if op(x,1)=1 then return(-log(b, op(x,2))) end_if;
      // identify integer values k = log(b, b^k):
      if domtype(b) <> DOM_IDENT and type(b) <> "_index" then

           // The following k is the candidate for x=b^k
        k := round(log::float(b, x));
        bk:= b^k:
        if bk = x then return(k) end_if;

           // The last 3 lines of code should have identified
           // all integers x of the form b^k, k integer.
           // Unfortunately, k may be affected by round-off.
           // Without roundoff we must have (assuming b>1):
           //   b^(k-1) < x < b^(k+1)
           // If x is in this range and b^k <> x, then x is
           // definitely not of the form x = b^integer:

        mmin:= min(b, 1/b):
        mmax:= max(b, 1/b):
        while not (mmin < x/bk and x/bk < mmax)  do
          // k is marred by round-off.
          // The correction dk should be ok:
          dk:= round(log::float(b, x/bk));
          if dk=0 then break end_if; // give up, cannot recover
                                         // from round-off problems
          k:= k + dk;
          bk:= bk*b^dk:
          if bk = x then return(k) end_if;
        end_while;

           // If we arrive here, then there is no integer k such that x = b^k.

      end_if;
      break


    of "_power" do
      // log(b, b^y) = y + k*2*PI*I/ln(b) 
      if op(x, 1) = b then
      // The following lines are correct and operational code
      // for complex exponents in x = b^exponent. However,
      // if the exponent is not real, then a symbolic
      // round(number*ln(b)/2/PI) is returned, which looks
      // odd. So do not simplify for non-real exponents:
      //if testtype(op(x,2), Type::Numeric) then
            // make sure that Im(log(b,x)) is in the range
            //     (-PI/ln(b), PI/ln(b)]   for b>1
            //     [PI/ln(b), -PI/ln(b))   for b<1
          //k:=  round(-Im(op(x,2))*ln(b)/2/PI );
          //return(op(x,2)+2*k*PI*I/ln(b))
      //end_if;
      // Do simplify log(b, b^real_exponent):
        if testtype(op(x,2), Type::Real) then
          return(op(x,2))
        end_if;
      end_if;
      break;
  end_case;

  if not testtype(x,Type::Arithmetical) then
    if testtype(x, Type::Set) then
      return(Dom::ImageSet(eval(procname)(b, #x), [#x], [x]));
    end_if;

    error("second argument must be of 'Type::Arithmetical'")
  end_if;

  if b = x then
    return(1)  // log(b, b)   = 1
  elif b = 1/x then
    return(-1) // log(b, 1/b) = -1
  end_if;
  procname(b, x)
end_proc:

log := prog::remember(log, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("log")]):

log:= funcenv(log):
log::print := "log":
log::type := "log":
log::info := "log -- the general logarithm":

log::float := proc(b, x)
local fb, fx;
begin
    fb:= float(b):
    if args(0) = 1 then
        // workaround for the output with alias(log=ln)
        return(hold(log)(fb))
    end_if;
    fx:= float(x):
    // If at least one of the arguments is numeric, then
    //  log(b,x) -> ln(x)/ln(b) = float or ln(x)/float or float/ln(b)
    // If both b and x and symbolic, then return log(b,x)
    case domtype(fb) 
    of DOM_FLOAT do
    of DOM_COMPLEX do
       return(ln::float(fx)/ln::float(fb));
    otherwise
       case domtype(fx) 
       of DOM_FLOAT do
       of DOM_COMPLEX do
          return(ln::float(fx)/ln::float(fb));
       otherwise // both b and x are symbolic
          return(log(fb, fx))
       end_case;
    end_case
end_proc:

log::series := (b,x,t,n,dir,opt) -> series(ln(x)/ln(b),t,n,dir,opt):

// The following discont slots of ln do not work: functions
// with 2 arguments have to be handled directly in discont
//
// log::complexDiscont := Dom::Interval(-infinity, [0]):
// log::realDiscont := {0}:

log::expand :=
    loadproc(log::expand, pathname("STDLIB","EXPAND"), "log"):

log::diff :=
    loadproc(log::diff, pathname("STDLIB","DIFF"), "log"):

log::simplify :=
    loadproc(log::simplify, pathname("STDLIB","SIMPLIFY"), "log"):

log::sign := loadproc(log::sign, pathname("SPECFUNC","SIGN"), "log"):

log::rectform :=
    loadproc(log::rectform, pathname("STDLIB","RECTFORM"), "log"):

log::Re := loadproc(log::Re, pathname("STDLIB","RE"), "log"):
log::Im := loadproc(log::Im, pathname("STDLIB","IM"), "log"):

log::hull := (a,b) -> ln::hull(b)/ln::hull(a):

log::Content := stdlib::genOutFunc("Clog", 2):

log::TeX := (l, data, prio) -> "\\log_{".
			      generate::tex(op(data,1), output::Priority::Noop).
			      "}\\!\\left(".
			      generate::tex(op(data,2), output::Priority::Noop).
			      "\\right)":

// end of file 
