/*++
The Digamma and Polygamma functions

psi(x)
psi(x, n)

x - an expression
n - an non negative integer

 //--------------------------------------------------------------------------
 // some general identities:
 //    2* psi(2*x) = psi(x) + psi(x + 1/2) + 2*ln(2)
 //    3* psi(3*x) = psi(x) + psi(x + 1/3) + psi(x + 2/3) + 3*ln(3)
 // general identities for n <> 0:
 //   2^(n+1)* psi(2*x, n) = psi(x,n) + psi(x+1/2,n) 
 //   3^(n+1)* psi(3*x, n) = psi(x,n) + psi(x+1/3,n) + psi(n, x+2/3)
 // general identities for all n:
 //    psi(1 - x, n) = (-1)^n*psi(x, n) + (-1)^n*PI^(n+1)*D([1 $ n], cot)(PI*x));
 //    psi(x, n) = (-1)^n*psi(1 - x, n) - PI^(n+1)*D([1 $ n], cot)(PI*x));
 //--------------------------------------------------------------------------
++*/

psi:=
proc(x, n)
  option noDebug;
  local fx, i, j;
begin
  if args(0) = 0 then
    error("no arguments given")
  elif args(0) > 2 then
    error("expecting no more than 2 arguments");
  elif x::dom::psi <> FAIL then
    return(x::dom::psi(args()))
  end_if;

  if testtype(x, Type::Numeric) then
    fx:= float(x);
    if domtype(fx) = DOM_FLOAT and iszero(frac(fx)) and fx <= 0 
      then error("singularity");
    end_if
  end_if;

  if args(0) = 1 then
     case type(x)
       of DOM_FLOAT do
         return( specfunc::psi(x) );
       of DOM_COMPLEX do
         if type(op(x,1)) = DOM_FLOAT or type(op(x,2)) = DOM_FLOAT then
           return( specfunc::psi(x) );
         end_if;
         break;
       of DOM_INT do
         if 1 <= x and x <= Pref::autoExpansionLimit() then // See appendix 
           return(psi(1) + _plus(1/i $ i=1..x-1) );
         end_if;
         if x <= 0 then
            error("singularity");
         end_if;
         break;
       of DOM_RAT do
         if specfunc::abs(x) > Pref::autoExpansionLimit() then // See appendix
            break;
         end_if;
         if domtype(2*x) = DOM_INT then
           if x < 0 
           then return(psi(1/2) + 2*_plus(1/(2*i-1)$i=1..(op(-x,1)+1) div 2))
           else return(psi(1/2) + 2*_plus(1/(2*i-1)$i=1..op(x,1) div 2))
           end_if
         end_if;

         // shift the first argument to the interval [0, 1].
         // The values x = 1/2, 1/3, 2/3, 1/4, 3/4, 1/6, 5/6 are remembered:
         if x > 1 then 
            fx:= floor(x);
            return(psi(x-fx) + _plus(1/(x-j) $ j=1..fx))
         end_if;
         if x < 0 then
            fx:= floor(x);
            return(psi(x-fx) - _plus(1/(x+j) $ j=0..(-fx-1)))
         end_if;
         break
       of DOM_SET do
       of "_union" do
         return(map(x, psi))
       otherwise
         if not testtype(x,Type::Arithmetical) then
           /* generic handling of sets */
           if testtype(x, Type::Set) then
             if type(x)=Dom::ImageSet then
               return(map(x, psi));
             else
               return(Dom::ImageSet(psi(#x), #x, x));
             end_if;
           end_if;

           error("argument must be of 'Type::Arithmetical'")
         end_if
     end_case;
  end_if;
  if args(0) = 2 then
     if testtype(n, Type::Numeric) then
       if type(n) <> DOM_INT or n < 0 then
         error("second argument must be a nonnegative integer");
       elif n = 0 then return( psi(x) );
       end_if;
       case type(x)
         of DOM_FLOAT do
           return( specfunc::floatpsi(x, n) );
         of DOM_COMPLEX do
           if type(op(x,1)) = DOM_FLOAT or type(op(x,2)) = DOM_FLOAT then
             return( specfunc::floatpsi(x, n) );
           end_if;
           break;
         of DOM_INT do
           if 1 <= x and x <= Pref::autoExpansionLimit() then
             return((-1)^n*fact(n)*(-zeta(n+1) + _plus(1/i^(n+1) $ i=1..x-1)));
           end_if;
           break;
         of DOM_RAT do
           if x = 1/2 then
             return( (-1)^(n+1)*fact(n)*(2^(n+1)-1)*zeta(n+1) );
           end_if;
           //----------------------------------------------------
           // no attempt to simplify for large x
           //----------------------------------------------------
           if specfunc::abs(x) > Pref::autoExpansionLimit() then
              break; // See appendix
           end_if;
           //--------------------------------------------------------
           // simplify x = integer/2
           //--------------------------------------------------------
           if domtype(2*x) = DOM_INT then
             if x < 0 
             then return(psi(1/2,n) + fact(n)*2^(n+1)*
                                 _plus(1/(2*i-1)^(n+1)$i=1..(op(-x,1)+1) div 2))
             else return(psi(1/2,n) + (-1)^n*fact(n)*2^(n+1)*
                                 _plus(1/(2*i-1)^(n+1)$i=1..op(x,1) div 2))
             end_if;
           end_if;
           //--------------------------------------------------------
           // simplify x = integer/3
           //--------------------------------------------------------
           if domtype(3*x) = DOM_INT then
              // We know psi(integer/3, even n).
              // However, there is no explicit psi(integer/3, odd n):
              if x = 1/3 and n mod 2 = 0 then
                 return((3^(n+1) -1)/2*psi(1, n) - (-1)^n/2*PI^(n+1)*D([1 $ n], cot)(PI/3));
              end_if;
              if x = 2/3 and n mod 2 = 0 then
                 return((3^(n+1)-1)*psi(1, n) - psi(1/3, n));
                 // Alternatively:
                 // return((-1)^n*psi(1/3, n) + (-1)^n*PI^(n+1)*D([1 $ n], cot)(PI/3));
              end_if;
           end_if;
           //--------------------------------------------------------
           // simplify x = integer/4
           //--------------------------------------------------------
           if domtype(4*x) = DOM_INT then
              // We know psi(integer/4, even n).
              // However, there is no explicit psi(integer/4, odd n):
              if x = 1/4 and n mod 2 = 0 then
                 return(2^n*psi(1/2, n) - (-1)^n/2*PI^(n+1)*D([1 $ n], cot)(PI/4));
              end_if;
              if x = 3/4 and n mod 2 = 0 then
                 return((-1)^n*(psi(1/4, n) + PI^(n+1)*D([1 $ n], cot)(PI/4)));
              end_if;
           end_if;
           //--------------------------------------------------------
           // simplify x = integer/6
           //--------------------------------------------------------
           if domtype(6*x) = DOM_INT then
              // We know psi(integer/6, even n).
              // However, there is no explicit psi(integer/6, odd n):
              if x = 1/6 and n mod 2 = 0 then
                 return((3^(n+1)-1)/2*psi(1/2, n) - (-1)^n/2*PI^(n+1)*D([1 $ n], cot)(PI/6));
              end_if;
              if x = 5/6 and n mod 2 = 0 then
                 return((-1)^n*psi(1/6, n) + (-1)^n*PI^(n+1)*D([1 $ n], cot)(PI/6));
              end_if;
           end_if;
           //--------------------------------------------------------
           // Shift the first argument to lie in the interval (0, 1).
           // In conjunction with the explicit results for 
           // x = integer/2, integer/3, integer/4, integer/6,
           // implemented above, we get explicit results for
           // psi(x = integer/k, even n), k = 2, 3, 4, 6.
           //--------------------------------------------------------
           // No! This is much too expensive. Only do it for
           // x = integer/k with small k = 2, 3, 4, 6, since in 
           // these cases explicit results are returned for x from
           // the interval (, 1)
           //--------------------------------------------------------
           if domtype(6*x) = DOM_INT or
              domtype(4*x) = DOM_INT or
              domtype(3*x) = DOM_INT or   // redundant, just for readability
              domtype(2*x) = DOM_INT then // redundant, just for readability
              if x > 1 then 
                  fx:= floor(x);
                  return(psi(x-fx,n)+(-1)^n*fact(n)*_plus(1/(x-j)^(n+1) $ j=1..fx))
              end_if;
              if x < 0 then
                 fx:= floor(x);
                 return(psi(x-fx,n)-(-1)^n*fact(n)*_plus(1/(x+j)^(n+1) $ j=0..(-fx-1)))
              end_if;
           end_if;
           //--------------------------------------------------------
           // For any rational x with |x| < 500, we can normalize the
           // result to psi(x, n) with 1/2 < x < 1:
           //--------------------------------------------------------
           // No! This is much too expensive and leaves cot(PI*x) terms,
           // unless x = integer/k with small k = 2, 3, 4, 6.
           //--------------------------------------------------------

           /*
           if 0 < x and x < 1/2 then
              return((-1)^n*psi(1-x, n) - PI^(n+1)*D([1 $ n], cot)(PI*x));
           end_if;
           */

           break;
       of DOM_SET do
       of "_union" do
           return(map(x, psi, n))
       otherwise
          if testtype(x, Type::Set) and not testtype(x, Type::Arithmetical) then
            return(Dom::ImageSet(eval(procname)(#x, n), [#x], [x]));
          end_if;
           if x = infinity then return(0) end_if;
       end_case;
     else // n is not Type::Numeric
          case type(x)
          of DOM_SET do
          of "_union" do
            return(map(x, psi, n))
          end_case;
          if not testtype(x, Type::Arithmetical) then
            if testtype(x, Type::Set) then
              if testtype(n, Type::Set) and not testtype(n, Type::Arithmetical) then
                return(Dom::ImageSet(eval(procname)(#x, #n), [#x, #n], [x, n intersect Z_ intersect Dom::Interval([0], infinity)]));
              else
                return(Dom::ImageSet(eval(procname)(#x, n), [#x], [x]));
              end_if;
            end_if;

            error("first argument must be of 'Type::Arithmetical'");
          end_if;

          if not testtype(n, Type::Arithmetical) then
            if testtype(n, Type::Set) then
              return(Dom::ImageSet(eval(procname)(x, #n), [#n], [n intersect Z_ intersect Dom::Interval([0], infinity)]));
            end_if;

            error("second argument must be of 'Type::Arithmetical'")
          end_if;
     end_if;
  end_if;
  procname(args())
end_proc:

psi := prog::remember(psi, 
  () -> [property::depends(args()), DIGITS, slotAssignCounter("psi")]):

psi(1)   := -EULER:
psi(1/2) := -EULER - 2*ln(2):
psi(1/3) := -EULER - 3/2*ln(3) - sqrt(3)/6*PI:
psi(2/3) := -EULER - 3/2*ln(3) + sqrt(3)/6*PI:
psi(1/4) := -EULER - 3*ln(2) - PI/2:
psi(3/4) := -EULER - 3*ln(2) + PI/2:
psi(1/6) := -EULER - 2*ln(2) - 3/2*ln(3) - sqrt(3)/2*PI:
psi(5/6) := -EULER - 2*ln(2) - 3/2*ln(3) + sqrt(3)/2*PI:
psi(infinity) := infinity:


psi:= funcenv(psi):
psi:= subsop(psi, 2=op(specfunc::psi, 2)):
psi::type:= "psi":
psi::print:= "psi":
psi::Content := stdlib::genOutFunc("Cpsi", 1, 2):
psi::info:=
  "psi -- the Digamma and Polygamma functions [try ?psi for options]":

psi::TeX := 
proc(t, ex, prio)
  local s1, s2;
begin
  s1 := generate::tex(op(ex, 1), output::Priority::Fconcat);
  if length(s1) < 6 or s1[1..6] <> "\\left(" then
    s1 := "\\left(".s1."\\right)";
  end_if;
  if nops(ex) = 1 then
    "\\psi\\!".s1;
  else
    s2 := generate::tex(op(ex, 2), output::Priority::Fconcat);
    if length(s2) < 6 or s2[1..6] <> "\\left(" then
      s2 := "\\left(".s2."\\right)";
    end_if;
    "\\psi^{".s2."}\\!".s1;
  end_if;
end_proc:


/* ------------------------------------------------------------
// the following utility computes a[n]:= psi::cot_rec(n), where
// a[n]/n! = d^n/dz^n cot(z) at the point z = PI/4, i.e.,
// D([1 $ n], cot)(PI/4) = psi::cot_rec(n!)*n!
// However, D([1 $ n], cot)(PI/4) seems to be faster!
psi::cot_rec:= proc(n)
local j;
option remember;
begin 
    if n = 0 then return(1); end_if;
    if n = 1 then return(-2); end_if;
    return(-1/n*_plus( psi::cot_rec(j)*psi::cot_rec(n-1-j) $ j = 0..n-1));
end_proc:
------------------------------------------------------------ */

psi::diff:=
  proc() 
    local f, n, x;
  begin
    f:= op(args(1),1);
    x:= args(2..args(0));
    if nops(args(1)) > 1 then
       n:= op(args(1),2):
       // do not try to diff w.r.t. the 2nd argument n of psi(f,n)
       if has(n, args(2)) 
        then hold(diff)(psi(f,n),x)
        else psi(f, n+1)*diff(f,x)
       end_if;
    else psi(f, 1)*diff(f, x)
    end_if
  end_proc:

psi::float:=
  proc(x, n)
  begin
    case args(0) 
      of 1 do return(specfunc::psi(x));
      of 2 do return(specfunc::floatpsi(x,n));
    end_case;
    error("wrong no of arguments");
  end_proc:

specfunc::floatpsi:=
    loadproc(specfunc::floatpsi, pathname("STDLIB", "FLOAT"), "psi"):

psi::expand:=
    loadproc(psi::expand, pathname("STDLIB", "EXPAND"), "psi"):

psi::series:=
    loadproc(psi::series, pathname("SERIES"), "psi"):

psi::rectform:=
    loadproc(psi::rectform, pathname("STDLIB", "RECTFORM"), "psi"):


// end of file 
